#include <stdio.h>
#include <vector>

#include <ConfigSource/Module.h>
#include <ConfigSource/ConfigSource.h>
#include <StatusSource/StatusSource.h>
#include <TimeSource/TimeSource.h>

class StatusMonitorModule : public Module
{
public:
  StatusMonitorModule(const char* spec);
  virtual ~StatusMonitorModule();

  virtual bool initialize(ConfigSource* config, utils::SymbolTable* table);
  virtual bool run();

private:
  StatusSource* _status_source;
  std::vector<StatusSource::ModuleStatus*> _modules;
  double _interval;
};

StatusMonitorModule::StatusMonitorModule(const char* spec)
  : Module("StatusMonitor", spec)
{
  _status_source = NULL;
}

StatusMonitorModule::~StatusMonitorModule()
{
  if (_status_source) {
    for (unsigned i=0;i<_modules.size();i++) {
      if (_modules[i])
        _status_source->closeModuleStatus(_modules[i]);
    }
  }
}

bool StatusMonitorModule::initialize(ConfigSource* config,
                                     utils::SymbolTable* table)
{
  int num_modules = config->numValues("modules");
  if (num_modules == 0) {
    fprintf(stderr, "StatusMonitorModule: No modules to monitor\n");
    return false;
  }

  const char* spec;
  spec = config->getString("status_source_spec");
  _status_source = create<StatusSource>(spec);
  if (!_status_source) {
    fprintf(stderr,
            "StatusMonitorModule: Spec '%s', creates no valid status_source\n",
            spec);
    return false;
  }

  const char** modules = new const char*[num_modules];
  if (config->getStrings("modules", modules, num_modules) != num_modules) {
    fprintf(stderr,
            "StatusMonitorModule: Problem reading in all %d modules\n", 
            num_modules);
    return false;
  }

  _modules.reserve(num_modules);
  for (int i=0;i<num_modules;i++) {
    printf("Monitoring %s\n", modules[i]);
    _modules.push_back(_status_source->moduleStatus(modules[i]));
    if (!_modules[i]) {
      printf("\tError in setting up monitor!\n");
    }
  }
  delete [] modules;

  _interval = config->getDouble("interval", 5.0);

  return true;
}

static const char* get_state_string(ConfigSource::ModuleState state)
{
  switch (state) {
  case ConfigSource::NOT_RUNNING:
    return "NotRunning";
  case ConfigSource::INITIALIZING:
    return "Initializing";
  case ConfigSource::INITIALIZING_DISPLAY:
    return "InitializingDisplay";
  case ConfigSource::INITIALIZED:
    return "Initialized";
  case ConfigSource::RUNNING:
    return "Running";
  case ConfigSource::PAUSED:
    return "Paused";
  case ConfigSource::BLOCKED:
    return "Blocked";
  case ConfigSource::ERROR:
    return "Error";
  }

  return "";
}

bool StatusMonitorModule::run()
{
  utils::Time now = TimeSource::now();
  long secs, usecs;
  now.getValue(secs, usecs);
  printf("\nNow %f, %s", now.getValue(), ctime(&secs));

  StatusSource::Status status;
  for (unsigned i=0;i<_modules.size();i++) {
    if (_modules[i]) {
      printf("%s: \n", _modules[i]->name());
      if (!_modules[i]->getStatus(status, STATUS_NONBLOCKING)) {
        printf("No Status\n");
      } else {
        if (status.state != ConfigSource::NOT_RUNNING) {
          printf("\t");
          if (!status.last_run.isZero())
            printf("Last run %5.4f, ", (status.last_run-now).getValue());
          if (!status.last_update.isZero())
            printf("Last update %5.4f, ", (status.last_update-now).getValue());
          printf("Average cycle time %5.4f", status.avg_cycle_time);
          printf("\n");
        }
        printf("\tState %s, Status %5.2f, Conf %5.2f\n",
               get_state_string(status.state),
               status.status, status.confidence);
        printf("\tMessage '%s'\n", status.msg);
      }
    }
  }

  getConfigSource()->sleep(_interval);

  return true;
}

MODULE_MAIN(status_monitor, StatusMonitorModule);
