#include <stdio.h>
#include <string.h>

#include <utils/String.h>
#include <utils/StringDict.h>
#include <utils/ConfigFile.h>

#include <ipt/ipt.h>
#include <ipt/sharedmem.h>

#include <StatusSource/StatusSource.h>
#include <ConfigSource/ModuleStatus.h>
#include <ConfigSource/Repository.h>

class ShmemModuleStatus;
class ShmemStatusSource : public StatusSource {
public:
  bool init(utils::ConfigFile* params, utils::SymbolTable* table);

  virtual void closeModuleStatus(ModuleStatus*);

protected:
  ShmemModuleStatus* shmem_module_status(const char* name,
                                         const char* mem_spec);

  utils::ManagedStringDict<ShmemModuleStatus> _module_stati;
  IPCommunicator* _com;
};

class MappedShmemStatusSource : public ShmemStatusSource {
public:
  bool init(utils::ConfigFile* params, utils::SymbolTable* table);

  virtual ModuleStatus* moduleStatus(const char* module_name);

private:
  utils::ConfigFile _mappings;
};

class RepositoryShmemStatusSource : public ShmemStatusSource {
public:
  bool init(utils::ConfigFile* params, utils::SymbolTable* table);

  virtual ModuleStatus* moduleStatus(const char* module_name);

private:
  ConfigSource* _repository;
};


class ShmemModuleStatus : public StatusSource::ModuleStatus {
public:
  ShmemModuleStatus(const char* name, IPCommunicator* com,
                    const char* mem_spec);

  virtual ~ShmemModuleStatus();

  virtual bool getStatus(StatusSource::Status&,
                         bool blocking = STATUS_BLOCKING);

private:
  IPCommunicator* _com;
  IPSharedMemory* _shm;
  utils::String _mem_spec;
  utils::String _msg;
};

ShmemModuleStatus::ShmemModuleStatus(const char* name,
                                     IPCommunicator* com, const char* mem_spec)
  : ModuleStatus(name)
{
  _mem_spec = mem_spec;
  _com = com;
  _shm = NULL;
}

ShmemModuleStatus::~ShmemModuleStatus()
{
  if (_shm)
    _com->CloseSharedMemory(_shm);
}

bool ShmemModuleStatus::getStatus(StatusSource::Status& status, bool blocking)
{
  // printf("Get status %s %p\n", name(), _shm);
  if (!_shm) {
    // printf("ShmemModuleStatus:: Opening client status '%s'\n",
    //       _mem_spec.getString());
    _shm = 
    _com->OpenSharedMemory(_mem_spec.getString(), MODULE_STATUS_FORMAT,
                           sizeof(ModuleStatusStruct) +
                           MAX_STATUS_MESSAGE_SIZE + sizeof(int));
    if (!_shm) {
      /*
      fprintf(stderr, "ShmemModuleStatus::getStatus: "
              "Could not open module shared memory for '%s' with '%s'\n",
              name(), _mem_spec.getString());
      */
      memset(&status, 0, sizeof(status));
      return false;
    }
  }

  if (blocking) {
    if (!_shm->Wait())
      return false;
  }
  
  ModuleStatusStruct ms;
  memset(&status, 0, sizeof(status));
  if (!_shm->FormattedData(&ms)) {
    _com->CloseSharedMemory(_shm);
    _shm = NULL;
    _com->Idle(0.0);
    return getStatus(status, blocking);
  }

  if (ms.msg == 0)
    return true;

  status.last_update.setValue(ms.update_secs, ms.update_usecs);
  status.last_run.setValue(ms.last_run_secs, ms.last_run_usecs);
  status.state = (StatusSource::ModuleState) ms.state;
  status.status = ms.status;
  status.confidence = ms.confidence;
  status.avg_cycle_time = ms.avg_cycle_time;
  _msg = ms.msg;
  status.msg = _msg.getString();

  _shm->DeleteContents(&ms);

  return true;
}


void ShmemStatusSource::closeModuleStatus(ModuleStatus* module_status)
{
  if (_module_stati.remove(module_status->name())) {
    delete module_status;
  } else {
    fprintf(stderr,
            "ShmemStatusSource::closeModuleStatus: "
            "Warning: Cannot close unknown module status\n");
  }
}

bool ShmemStatusSource::init(utils::ConfigFile* params,
                             utils::SymbolTable* globals)
{
  _com = 
    IPCommunicator::Communicator(globals, 
                                 params->getString("ipt_spec",
                                                   "unix: int port=0;"));
  if (!_com) {
    fprintf(stderr, "ShmemStatusSource: Could not create communicator\n");
    return false;
  }

  return true;
}

ShmemModuleStatus* ShmemStatusSource::shmem_module_status(const char* name,
                                                          const char *mem_spec)
{
  ShmemModuleStatus* mod_status = new ShmemModuleStatus(name, _com, mem_spec);
  ShmemModuleStatus* old_status;
  if (_module_stati.find(name, old_status)) {
    _module_stati.remove(name);
    delete old_status;
  }
  
  _module_stati.enter(name, mod_status);
  return mod_status;
}

bool MappedShmemStatusSource::init(utils::ConfigFile* params,
                                   utils::SymbolTable* globals)
{
  if (!ShmemStatusSource::init(params, globals))
    return false;

  utils::ConfigFile mappings;
  if (!params->getStruct("mappings")) {
    fprintf(stderr, "MappedShmemStatusSource::init: Error: No mappings\n");
    return false;
  }

  utils::ConfigFile::copy(mappings, _mappings);
  return true;
}

StatusSource::ModuleStatus*
MappedShmemStatusSource::moduleStatus(const char* module_name)
{
  const char* mem_spec = _mappings.getString(module_name, "");
  if (!mem_spec || !*mem_spec) {
    fprintf(stderr, "MappedShmemStatusSource::moduleStatus: "
            "No mapping for module '%s'\n", module_name);
    return NULL;
  }

  return shmem_module_status(module_name, mem_spec);
}

StatusSource* create_StatusSource_mapped(StatusSourceGenerator* gen,
                                         utils::ConfigFile* params,
                                         utils::SymbolTable* globals)
{
  MappedShmemStatusSource* source = new MappedShmemStatusSource();
  if (!source->init(params, globals)) {
    delete source;
    return NULL;
  }
  return source;
}

bool RepositoryShmemStatusSource::init(utils::ConfigFile* params,
                                       utils::SymbolTable* globals)
{
  if (!ShmemStatusSource::init(params, globals))
    return false;

  _repository = Repository::instance(globals);
  if (!_repository) {
    fprintf(stderr, "RepositoryShmemStatusSource::init: "
            "Cannot contact repository\n");
    return false;
  }

  return true;
}

StatusSource::ModuleStatus*
RepositoryShmemStatusSource::moduleStatus(const char* module_name)
{
  char buffer[300];
  snprintf(buffer, 300, "Modules.%s.host", module_name);
  const char* host = _repository->getString(buffer, "");
  if (!host || !*host) {
    fprintf(stderr, "RepositoryShmemStatusSource::moduleStatus: "
            "No host for module '%s'\n", module_name);
    host = _com->ThisHost();
    fprintf(stderr, "Assuming host %s\n", host);
  }
  
  snprintf(buffer, 300, "managed: string name='%s_status@%s|%d'; mgr_tag=TCP;",
           module_name, host, 1389);
  return shmem_module_status(module_name, buffer);
}

StatusSource* create_StatusSource_repository(StatusSourceGenerator* gen,
                                             utils::ConfigFile* params,
                                             utils::SymbolTable* globals)
{
  RepositoryShmemStatusSource* source = new RepositoryShmemStatusSource();
  if (!source->init(params, globals)) {
    delete source;
    return NULL;
  }
  return source;
}
