#include <time.h>
#include <stdio.h>

#include <ConfigSource/Module.h>
#include <ConfigSource/ConfigSource.h>
#include <FrameGrabber/FrameGrabber.h>
#include <ImgDisplay/ImgDisplay.h>

class ImgDisplayRelayModule : public Module
{
public:
  ImgDisplayRelayModule(const char* spec) : Module("ImgDisplayRelay", spec) {}

  virtual bool initialize(ConfigSource* config, utils::SymbolTable* table) {
    const char* spec;
    spec = config->getString("grabber_spec");
    _grabber = create<FrameGrabber>(spec);
    if (!_grabber) {
      fprintf(stderr,
              "ImgDisplayRelayModule: Spec '%s', creates no valid grabber\n",
              spec);
      return false;
    }

    spec = config->getString("disp_spec");
    _disp = create<ImgDisplay>(spec);
    if (!_disp) {
      fprintf(stderr,
              "ImgDisplayRelayModule: Spec '%s', creates no valid ImgDisplay\n",
              spec);
      return false;
    }
    _win = _disp->createWindow("Digitizer",
                               _grabber->numCols(), _grabber->numRows(),
                               _grabber->colorFormat());

    _exit_on_no_data = config->getBool("exit_on_no_data", true);
    _interval = config->getFloat("interval", 0.0);
    _last = utils::Time::getTimeOfDay();

    return true;
  }

  virtual bool run() {
    if (!_grabber->grab()) {
      if (_exit_on_no_data)
        return false;
      fprintf(stderr, "Error getting image\n");
      return true;
    }

    _win->begin();
    utils::Time t;
    _grabber->imageTag(t);
    _win->setTimeTag(t);
    _win->fill(_grabber->getData());

    long secs, usecs;
    t.getValue(secs, usecs);
    char* time_str = ctime(&secs);
    time_str[strlen(time_str)-1] = '\0';
    _win->text(0, 0, time_str, ImgDisplay::Blue);
    
    _win->end();

    utils::Time now;
    if (_interval > 0) {
      ConfigSource* config_src = getConfigSource();
      double time_left = _interval;
      while (1) {
        config_src->processEvents(_interval);
        if (!isRunning())
          return false;
        now = utils::Time::getTimeOfDay();
        time_left -= (now-_last).getValue();
        _last = now;
        if (time_left <= 0) {
          break;
        }
      }
    }
    
    return true;
  }

private:
  FrameGrabber* _grabber;
  ImgDisplay* _disp;
  ImgDisplay::Window* _win;
  bool _exit_on_no_data;
  float _interval;
  utils::Time _last;
};

MODULE_MAIN(relay, ImgDisplayRelayModule);
