#include <stdio.h>
#include <unistd.h>

#include <utils/ConfigFile.h>
#include <utils/SymbolTable.h>

#include <ConfigSource/Module.h>
#include <ConfigSource/ConfigSource.h>
#include <FrameGrabber/FrameGrabber.h>
#include <ImgDisplay/ImgDisplay.h>

class ImgDisplayMemModule : public Module
{
public:
  ImgDisplayMemModule(const char* spec) : Module("ImgDisplayMem", spec) {}

  virtual bool initialize(ConfigSource* config, utils::SymbolTable* table) {
    utils::ConfigFile mem_params;
    config->getStruct("mem_params", mem_params);
    mem_params.set("tag", "memory");
    _mem_disp = create<ImgDisplay>(mem_params);
    if (!_mem_disp) {
      fprintf(stderr, "ImgDisplayMemModule: Cannot create memory display\n");
      return false;
    }
        
    const char* spec = config->getString("disp_spec", "XIH");
    _disp = create<ImgDisplay>(spec);
    if (!_disp) {
      fprintf(stderr,
              "ImgDisplayMemModule: Cannot create display our of '%s'\n",
              spec);
      return false;
    }

    _fmt = config->getString("format", "grey8");
    _width = config->getInt("width", 640);
    _height = config->getInt("height", 480);

    return true;
  }

  virtual bool run() {
    ImgDisplay::Window* win =
      _mem_disp->createWindow("Test", _width, _height, _fmt.getString());
    if (!win) {
      printf("Could not create window\n");
      return false;
    }
    win->begin();
    win->clear();
    win->fill(ImgDisplay::Pink);
    win->fillBox(100, 100, 200, 200, ImgDisplay::White);
    win->setLineWidth(3);
    win->box(120,120,180,180, ImgDisplay::Blue);
    win->line(300, 300, 400, 450, ImgDisplay::Red);
    win->circle(400, 50, 20, ImgDisplay::Orange);
    win->text(20, 80, "Hello world", ImgDisplay::Yellow);
    win->setLineWidth(0);
    win->text(20, 80, "Hello world", ImgDisplay::Black);
    win->text(0,0,"Corner TTTTT", ImgDisplay::White);
    win->end();

    unsigned char* mem_out = (unsigned char*)
      getSymbolTable()->get("MemoryDisplayBuffer");
    if (!mem_out) {
      printf("Memory display has no buffer named MemoryDisplayBuffer\n");
      return false;
    }

    ImgDisplay::ColorType output_type = _mem_disp->defaultWindowColorType();
    if (output_type == ImgDisplay::UnknownColorType) {
      output_type = ImgDisplay::stringToColorType(_fmt.getString());
    }
    ImgDisplay::Window* owin =
      _disp->createWindow("Test", _width, _height, output_type);
    if (!owin) {
      printf("Cannot create output window\n");
      return false;
    }
    printf("Displaying\n");
    owin->begin();
    owin->fill(mem_out);
    owin->end();

    printf("Hit return to exit: ");
    fflush(stdout);
    char buffer[20];
    fgets(buffer, 19, stdin);

    return false;
  }

private:
  ImgDisplay* _mem_disp;
  ImgDisplay* _disp;
  utils::String _fmt;
  int _width, _height;
};

MODULE_MAIN(mem_test, ImgDisplayMemModule);
