#include <stdio.h>

#include <utils/ConfigFile.h>
#include <utils/SymbolTable.h>
#include <utils/formatting/PackAction.h>

#include <ipt/ipt.h>
#include <ipt/sharedmem.h>

#include <ImgDisplay/BufferDisplay.h>
#include <ImgDisplay/ShmemFormat.h>

class ShmemWindow;
class ShmemDisplay : public BufferDisplay {
 public:
  ShmemDisplay(IPCommunicator* com, const char** mem_specs,
               const char* dest_fmt);
  virtual ~ShmemDisplay();

  virtual ImgDisplay::Window*
  createWindow(const char* title, int width, int height,
               ImgDisplay::ColorType src_type);
  virtual void destroyWindow(Window* win);
  virtual void destroyAllWindows();
  virtual ImgDisplay::ColorType defaultWindowColorType() const {
    return _dest_type;
  }
  const char* getCurrentSpec() const;
  IPCommunicator* getCommunicator() const { return _com; }

private:
  char** _mem_specs;
  int _num_mem_specs;
  int _num_wins;
  IPCommunicator* _com;
  ColorType _dest_type;
};

class ShmemWindow : public BufferWindow
{
 public:
  ShmemWindow(IPSharedMemory* shm, ImgDisplay* display, int width, int height,
              ImgDisplay::ColorType src_type, ImgDisplay::ColorType dest_type);
  virtual ~ShmemWindow();

  virtual bool begin();
  virtual void end();
  virtual void setTimeTag(utils::Time t) { _image_tag = t; }

private:
  IPSharedMemory* _shm;
  ImageOutputStruct _output_area;
  int _img_size;
  utils::Time _image_tag;
};

ShmemDisplay::ShmemDisplay(IPCommunicator* com,
                           const char** mem_specs, const char* dest_fmt)
{
  _num_wins = 0;
  _com = com;
  _dest_type = stringToColorType(dest_fmt);

  for (_num_mem_specs=0; mem_specs[_num_mem_specs]; _num_mem_specs++);

  _mem_specs = new char*[_num_mem_specs];
  for (int i=0; i<_num_mem_specs; i++) 
    _mem_specs[i] = utils::String::copy(mem_specs[i]);
}

ShmemDisplay::~ShmemDisplay()
{
  for (int i=0; i<_num_mem_specs; i++) 
    delete [] _mem_specs[i];
  delete [] _mem_specs;
}
 
const char* ShmemDisplay::getCurrentSpec() const
{
  if (_num_wins < 0 || _num_wins > _num_mem_specs) 
    return NULL;
  return _mem_specs[_num_wins];
}

ImgDisplay::Window* ShmemDisplay::createWindow(const char* title,
                                               int width, int height,
                                               ImgDisplay::ColorType src_type)
{
  const char* spec = getCurrentSpec();
  if (!spec)
    return NULL;

  ImgDisplay::Window* window;
  ImgDisplay::ColorType dest_type;
  if (_dest_type == ImgDisplay::UnknownColorType)
    dest_type = src_type;
  else
    dest_type = _dest_type;

  int max_size = height*width*BufferWindow::getBytesPerPixel(dest_type);
  // printf("Creating window %d %d %d\n", height, width, max_size);
  IPSharedMemory* shm = 
    _com->OpenSharedMemory(spec, IMAGE_SHMEM_FORMAT,
                           sizeof(ImageOutputStruct) + max_size);
  if (!shm)
    return NULL;
  window = new ShmemWindow(shm, this, width, height, src_type, dest_type);
  _num_wins++;

  return window;
}

void ShmemDisplay::destroyWindow(Window* win)
{
  BufferDisplay::destroyWindow(win);
  _num_wins--;
}

void ShmemDisplay::destroyAllWindows()
{
  BufferDisplay::destroyAllWindows();
  _num_wins = 0;
}

ShmemWindow::ShmemWindow(IPSharedMemory* shm, 
                         ImgDisplay* display, int width, int height, 
                         ImgDisplay::ColorType src_type,
                         ImgDisplay::ColorType dest_type)
  : BufferWindow(display, width, height,
                 width*getBytesPerPixel(dest_type),
                 src_type, dest_type)
{
  int bpp = getBytesPerPixel(dest_type);
  _img_size = height*width*bpp;
  _shm = shm;
  _output_area.is_interlaced = true;
  _output_area.is_row_major = true;
  _output_area.num_rows = height;
  _output_area.num_cols = width;
  _output_area.num_bands = (bpp > 1 ? 3 : 1);
  _output_area.pixel_size = bpp;
  _output_area.color_format = dest_type;
}

ShmemWindow::~ShmemWindow()
{
  ((ShmemDisplay*) display())->getCommunicator()->CloseSharedMemory(_shm);
}

bool ShmemWindow::begin()
{
  setDest(_shm->GetOutputArea() + sizeof(ImageOutputStruct));

  return true;
}

void ShmemWindow::end() 
{
  long secs, usecs;
  _image_tag.getValue(secs, usecs);
  _output_area.secs = secs;
  _output_area.usecs = usecs;
  unsigned char* output = _shm->GetOutputArea();
  utils::PackAction::pack(_shm->Formatter(), output, _shm->MaxSize(),
                          &_output_area);
  _shm->PutData(_img_size + sizeof(ImageOutputStruct));
}
    
ImgDisplay* create_ImgDisplay_shmem(utils::Generator<ImgDisplay>* gen,
                                    utils::ConfigFile* params,
                                    utils::SymbolTable* globals)
{
  IPCommunicator* com =
    IPCommunicator::Communicator(globals, 
                                 params->getString("ipt_spec",
                                                   "unix: int port=0;"));
  if (!com)
    return NULL;
  
  const char* mem_specs[11];
  memset(mem_specs, 0, 11*sizeof(const char*));
  mem_specs[0] = "managed: name=DisplayBuffer; owner=true;";
  params->getStrings("mem_specs", mem_specs, 10);
  return new ShmemDisplay(com, mem_specs, params->getString("dest_fmt",
                                                            "unknown"));
}
