// ----------------------------------------------------------------
// SPU-Toolbox
//
//        File: SPU-Image.h (include/SPU-Toolbox/SPU-Image.h)
// Description: Image class declarations.
// ----------------------------------------------------------------
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public License
// as published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS OR A PARTICULAR PURPOSE. See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston MA
// 02111-1307, USA.
//
// This file may use Doxygen style comments to facilitate automatic
// documentation generation. More information on Doxygen can be found
// at "http://www.stack.nl/~dimitri/doxygen/index.html".
//
// Author: Ross J. Micheals
//         rjm2@eecs.lehigh.edu
//
// (c) 1999-2001 Ross J. Micheals

#ifndef __SPU_Toolbox__included__Image_h__ // Include guard
#define __SPU_Toolbox__included__Image_h__

#include <SPU-Toolbox/config.h>

// System includes
extern "C" {
#include <string.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>  
#include <unistd.h>
#if SPU_HAVE_LIBIPL == 1
#include <ipl/ipl.h>  
#endif
}

// C++ includes
#include <iostream.h>

// SPU-Toolbox includes
#include <SPU-Toolbox/config.h>
#include <SPU-Toolbox/color-utils.h>
#include <SPU-Toolbox/SPU-Types.h>
#include <SPU-Toolbox/pnm_helper.h>
#include <SPU-Toolbox/clamp.h>

// Optional/support includes
#if SPU_HAVE_LIBJPEG == 1
#include <SPU-Toolbox/jpeg_helper.h>
#endif

bool SPU_Image_init_statics();
static bool SPU_Image_static_initialization_hack = SPU_Image_init_statics();

/*!

  \file  SPU-Image.h
  \brief Image class declaration

*/

/*!

  \class  SPU_Image SPU-Image.h SPU-Toolbox/SPU-Image.h
  \brief  Basic 2d scalar image class
  \author Ross J. Micheals (rjm2@cse.lehigh.edu)

  Each SPU_Image object has the responsibility of holding a single
  image, its properties (such as color format, height, and width), and
  potentially, its storage.
  
  The origin of a SPU-Image (0,0) is located in the upper-right corner
  of the image. The lower-right corner of a SPU-Image has the
  coordinates (width-1, height-1).

  Acceptable color format strings are: "grey8", "grey16", "grey32",
  "rgb1555", "rgb5155", "rgb5515", "rgb5551", "bgr1555", "bgr5155",
  "bgr5515", "bgr5551", "rgb565", "bgr565", "rgb24", "bgr24",
  "msb_rgb32", "lsb_rgb32", "msb_bgr32", "lsb_bgr32", "yuv422_planar",
  and "yuv422_packed".

  Have a section here about byte alignment and bytes-per-line.

*/

class SPU_Image {
  friend bool SPU_Image_init_statics();
public:

  //! Public-access default constructor.
  /*!  

    This constructor creates an \e empty Image instance. An instance
    which has been constructed, but not initialized (via init()), is
    considered to be in the \e empty state. All empty instances should
    be explicitly initialized (via init()) before they are
    used. Except for init(), the behavior of \e empty Image objects is
    not defined.  

  */
  inline SPU_Image(void);		

  //! Public-access copy constructor.
  /*!

    This copy constructor creates a new, \e non-empty instance which
    is a copy of \a image. 

    \param image              Image object to be copied
    \param allocate_storage   Appropriate storage should be alloated for
    the new instance 
    \param with_copy          The contents of the Image \a image should be
    copied into the new instance.

  */
  inline SPU_Image(const SPU_Image& image, bool allocate_storage = true, 
		   bool with_copy = true);

  //! Recommended public-access parameterized constructor
  /*! 

    This constructor instantiates a \e non-empty image object
    according to the input parameters.

    Invalid combinations of arguments include:

    \param color_format string describing the color format of the image
    \param width width, or number of columns, of image in pixels
    \param height height, or number of rows, of image in pixels
    \param allocate_storage the object should allocate and manage the
    memory required to store the raw image data
    \param data pointer to image data
    \param bytes_per_line the number of bytes of storage used for the
    image data pointed to by the \a data pointer. If unspecified, i.e.
    equal to 0, the constructor will infer the information from the
    image width and color format
    \param do_not_pad Indicates that the object should not attempt to
    byte align self-allocated storage.

  */
  inline SPU_Image(const char*   color_format,
		   const int     width,
		   const int     height,
		   const bool    allocate_storage = true,
		   const bool    with_copy        = false,
		   SPU_u8* const data             = 0,
		   const int     bytes_per_line   = 0,
		   const bool    do_not_pad       = false);

  //! Public-access file-based constructor
  /*!

    Instantiates an SPU_Image object from a PNM. After calling this
    constructor, calling open_status() is recommended.

    \param fname file name of PNM (pgm/ppm) to read

    \see open_status()

  */
  inline SPU_Image(const char* fname);

  //! Public-access destructor
  /*!
    
    If the object has allocated private storage for the image data,
    this data will be freed upon object destruction.

  */
  inline ~SPU_Image(void);

  //! Assignment
  /*!

    Assigns one image from another.

  */
  bool assign(const SPU_Image* image, bool allocate_storage, bool with_copy);

  //! Explicitly allocate private storage.
  /*!

    Explicitly performs private storage allocation according to the
    objects current state.

  */
  inline bool allocate(void);

  //! Get pixel pointer (fast)
  /*!
    
    Inlined method that returns pointer to specified pixel. No bounds
    checking on pixel access is performed.

    \warning Accessing an out-of-bounds pixel results in undefined behavior.

   */
  inline SPU_u8* fast_get_pixel(int x, int y);

  //! Get const pixel pointer (fast)
  /*!
    
    Inlined method that returns pointer to specified pixel (which is
    const). No bounds checking on pixel access is performed.

    \warning Accessing an out-of-bounds pixel results in undefined behavior.

   */
  inline const SPU_u8* fast_get_pixel(int x, int y) const;

  //! Fill image data
  /*!

    Fills every byte of the raw image data with the specified value.

    \param val value to fill image with

  */
  void fill(const SPU_u8 val);

  //! Get bytes per line
  /*! 

    Returns the number of bytes of storage used for each line of the
    image. Usually, this number is equal to the image width times the
    number of bytes per pixel (sometimes also considered to be the
    image depth). For effeciency purposes, however, image rows are
    often memory aligned, and therefore have some additional storage
    at the end of each row. 

    \see get_width(), get_height(), get_bytes_per_pixel()

  */
  inline int get_bytes_per_line(void) const;

  //! Get bytes per pixel
  /*!
    
    Returns the number of bytes of stoage used for each pixel. This
    value is dependent only on the color mode.
    
   */
  inline int get_bytes_per_pixel(void) const;

  //! Get image color format
  /*!

    Returns the color format of the image held by the object.

  */
  inline SPU_Color_Format get_color_format(void) const;

  //! Get image color format as string
  /*!

    Returns a string describing the color format of the image held by
    the object.

   */
  inline const char* get_color_format_as_string(void) const;

  //! Get non-const raw image data pointer.
  /*!
    
    Returns a non-const pointer to the raw image data. A null pointer
    implies that no image data has been assigned.

   */
  inline SPU_u8* get_data(void);

  //! Get const raw image data pointer
  /*!
    
    Returns a const pointer to the raw image data. A null pointer
    imples that no image data has been assigned.

    \see set_data()

   */
  inline const SPU_u8* get_data(void) const;

  //! Get image height.
  /*!

    Returns the width of the image (in pixels) associated with the
    object.

   */ 
  inline int get_height(void) const;

  //! Get image width (in pixels)
  /*!
    
    Returns the width of the image (in pixels) associated with the object.

  */
  inline int get_width(void) const;

  //! Get image memory size.
  /*!

    Calculates the number of bytes required to store a complete copy
    of the raw image data (image height multiplied by the number of
    bytes per line).

    \warning Use with caution on non-empty images. If not used
    properly this member function can disrupt the object's internal
    state  and may lead to undefined behaviour.

    \see get_height(), get_bytes_per_line()

   */
  inline int get_size(void) const;

  /*! Get image data status.

    Returns \c true if object includes a non-null pointer to raw data
    (which may been set via a deep or a shallow copy). Returns \e
    false otherwise.

    \see SPU_Image(), init

  */
  inline bool has_data(void) const;


  //! Insert a sub-image
  /*!
    \author Terry Boult (tboult@eecs.lehigh.edu)

    A rectangular subwindow, from input_image, is copies into the
    image. Boundaries are tested in the input image, and the window is
    automatically clipped if necessary. 

    Returns \c true if the copy was performed sucessfully. Returns \e
    false otherwise. Returns \c false if unsucessful normal copy if
    from input_image [(fy,fx),(fy+iheight,fx+iwidth)] to
    [(ty,tx),(fy+iheight,fx+iwidth)] in this image the actual copy may
    be smaller if any of the image boundaries would be violated
    
    \param input_image source for copy
    \param tx x coordinate of destination region
    \param ty y coordinate of destination region
    \param iwidth width of copy (in number of pixels) if <0 (default) copy is to end of input
    \param iheight height of copy (in number of pixels) if <0 (default) copy is to end of input
    \param: fx x loction from we being copy
    \param: fy y loction from which we copy    

  */
  bool insert_subimage(const SPU_Image& input_image,
		       int tx, int ty,  
		       int iwidth = -1, int iheight = -1, 
		       int fx = 0, int fy = 0);

  //! Get image empty status.
  /*

    Returns \c true if the SPU_Image is considered to be \e empty. Returns
    \c false otherwise. An image is \e empty if it has been
    instantiated, but hold not image data (e.g. an SPU_Image object
    immediately after construction, but before init() has been called.

  */

  bool extract_subimage(SPU_Image& output_image, int ox, int oy);

  inline bool is_empty(void) const;

  /*! Get image status.

    Returns \c true if image is not malformed. An image is malformed
    if it has: a negative width, a negative height, or an invlaid
    color format. (Some internal state variables are also
    checked). Returns \c false otherwise.

   */
  inline bool is_valid(void) const;

  //! Get file constructor status
  /*!
    
    Queries object state corresponding to sucessful initialization
    from a file.

    \returns Returns \c true if the image was created
    sucessfully. Returns \c false otherwise (for example, the file did
    not exist). 

  */
  inline bool open_status(void) const;


  /*! Initialize empty SPU_Image object.

    Initialize an empty SPU_Image object according to parameters.

    \param color_format string describing the color format of the image
    \param width width, or number of columns, of image in pixels
    \param height height, or number of rows, of image in pixels
    \param allocate_storate the object should allocate and manage the
    memory required to store the raw image data
    \param with_copy object should copy the raw image data as pointed
    to by \e data into its own private storage
    \param data pointer to raw image data
    \param bytes_per_line size of storage per image scan line (in
    bytes) used for the image data pointed to by the \e data pointer,
    or to be used when allocating object storage
    \param do_not_pad indicates the object should not attempt to byte
    align self-allocated storage
  
   */
  bool init(const char*   color_format,
	    const int     width, 
	    const int     height,
	    const bool    allocate_storage = true,
	    const bool    with_copy        = false, 
	    SPU_u8* const data             = 0,
	    const int     bytes_per_line   = 0,
	    const bool    do_not_pad       = false);


  //! Read image information from PNM (PGM/PPM)
  /*!
    
    Fill image information appropriate to the contents of file name \e
    fname. Supported file formats are PGM (P2 and P5) and PPM (P3 and
    P6). Returns \c true if image was loaded sucessfully. Returns \e
    false otherwise.
    
    TODO: Describe behavior if image is in non-empty state.
    
    \param fname filename of PNM input image

  */
  bool read_pnm(const char* fname);


  //! Store image as PNM (PGM/PPM) file
  /*!
   
    Saves the image information to a PNM (PGM or PPM)

    \param fname name of PNM file to write
    \param as_binary If as_binary is \c true, then the image is saved
    in binary form. The image is saved in ASCII mode otherwise.
    
   */
  bool save_pnm(const char* fname, bool as_binary = true);

#if SPU_HAVE_LIBJPEG == 1
  //! Store image as JPEG file
  /*!

    
    \param fname name of JPEG file to write \param quality JPEG
    quality parameter. A quality of 0 results in the smallest, but
    most lossy image. A quality parameter of 100 represents the least
    lossy (but not lossless), but largest miage.
    
    \note Requires \e libjpeg support.

   */
  bool save_jpeg(const char* fname, int quality = 100);
  bool read_jpeg(const char* fname);
#endif

  //! Set image width.
  /*!

    Explicitly sets width of image (in pixels). Proceed with cauthion
    on non-empty images; it is permitted, but not recommended. Other
    than the reported image size (i.e. get_size()) no other instance
    variables are updated or modified (e.g. no storage is
    reallocated).

    \param width new width of image

    \see get_height(), get_width(), set_height()
  */
  inline void set_width(int width);

  //! Set image height.
  /*!

    Explicitly sets width, in pixels, of image held by the object.
    
    \warning Proceed with caution on non-empty images. Other than the
    reported image size, i.e. get_size(), no other instance variables
    are updated or modified (e.g. no storage is
    reallocated). Therefore, the operation is permitted, but not
    recommended.
    
    \param height new height of image (in pixels).
    
    \see get_width

  */
  inline void set_height(int height);

  //! Get allocation status.
  /*!  

    Returns true if the object has allocated private storage in
    which to hold the raw image data (i.e. deep copy). Returns false
    otherwise.

  */
  inline bool has_allocated(void) const;

  enum Index_Type { 
    Index_Torroidal, /*!< out-of-bounds pixels are wrapped around.       */
    Index_Extended,  /*!< out-of-bounds pixels are clamped to boundaries */
    Index_Mirrored,  /*!< out-of-bounds pixels are mirrored              */
    Index_Constant,  /*!< out-of-bounds pixels are given constant value  */
    Index_COUNT,     /*!< number of indexing modes                       */
    Index_None       /*!< out-of-bound pixel behaviour not designated    */
  };

  //! Set indexing style
  /*!

    Sets the indexing mode which dictates the behavior of get_pixel,
    upon an attempt to access out-of-bound pixels.
    
   */
  bool set_index_sytle(Index_Type it);


  //! Get pointer to pixel value
  /*!

    Returns a pointer to the raw pixel value located at image
    coordianate (\e x ,\e y). If the pixel is out of bounds, then the
    object returns a pointer according to the indexing mode as set by
    set_index_style().

    \sa set_index_sytle(), Index_Type

  */
  SPU_u8* get_pixel(int x, int y);

  //! Get pointer to constant pixel value
  /*!

    Returns a pointer to the const raw pixel value located at image
    coordianate (\e x ,\e y). If the pixel is out of bounds, then the
    object returns a pointer according to the indexing mode as set by
    set_index_style().

    \sa set_index_sytle(), Index_Type

  */
  const SPU_u8* get_pixel(int x, int y) const;

  //! Set pointer to raw image data.
  /*!

    Explicitly set the raw image data poiner. Returns \c false if the
    object has already allocated private storage for the image
    (i.e. holds a deep copy). 

    \warning Use with caution on non-empty images. If not used
    properly this member function can disrupt the object's internal
    state which and may lead to undefined behaviour.

   */
  inline bool set_data(SPU_u8* address);
  

  //! Deallocate private storage.
  /*!
    
    Explicitly deallocates any private storage allocated by this
    object.

    \see allocate()

   */
  bool release(void);

  //! Get image timestamp.
  /*!
  
    Returns the image timestamp as set by get_timestamp(). 
    
    \see set_timestamp()

  */
  inline struct timeval get_timestamp(void) const;

  //! Explicitly set image timestamp.
  /*!

    Sets the image timestamp to the value \e tv. 

    \note All timestamps are set explicitly.

    \param tv time to use for object timestamp

   */
  inline void set_timestamp(const struct timeval& tv);

  //! Set byte alignment
  /*! 

    Sets the byte boundary to use when allocating padded, or in other
    words, byte-aligned images. Each scanline of a padded image is
    located at an address evenly divisible by the byte alignment.

    \see verify_byte_alignment(), get_byte_alignment()
  */
  static void set_byte_aligment(int bytes);

  //! Verify byte alignment
  /*!

    Checks to see if the image owned by the object is byte-aligned
    according to the member function arguement.

    \see set_byte_aligment, get_byte_alignment

   */
  bool verify_byte_alignment(int bytes);

#if SPU_HAVE_LIBIPL == 1
  //! Get IPL image pointer
  /*!

    Returns a pointer to an IPL image header appropriate to the
    object. Note the IPL image header is stored within the object,
    giving potential access to the object's internal state.

    \note Requries Intel IPL (Image Processing Library)
    (http://developer.intel.com/software/products/perflib/ipl/)
    support.

   */
  inline IplImage* ipl(void);
#endif

private:


  static int sm_byte_alignment;

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // Private helper function prototypes                         //
  //                                                            //
  ////////////////////////////////////////////////////////////////

        SPU_u8* get_Torroidal_pixel(int x, int y);
  const SPU_u8* get_Torroidal_pixel(int x, int y) const;

        SPU_u8* get_Mirrored_pixel(int x, int y);
  const SPU_u8* get_Mirrored_pixel(int x, int y) const;

        SPU_u8* get_Extended_pixel(int x, int y);
  const SPU_u8* get_Extended_pixel(int x, int y) const;

        SPU_u8* get_Constant_pixel(int x, int y);
  const SPU_u8* get_Constant_pixel(int x, int y) const;

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // General image properties                                   //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  SPU_Color_Format m_color_format;    // Image color format
  int              m_width;	      // Image width
  int              m_height;	      // Image height
  int              m_bpp;	      // Image bytes per pixel (depth, sorta)
  int              m_bytes_per_line;  // Image bytes per scanline
  char*            m_filename;        // Image filename
  SPU_u8*          m_raw_mem;	      // Pointer to raw memory
  SPU_u8*          m_aligned_mem;     // Aligned pointer into m_raw_mem
  int              m_aligned_memsize; // Size of aligned memory
  int              m_raw_memsize;     // Size of raw memory

#if SPU_HAVE_LIBIPL == 1
  ////////////////////////////////////////////////////////////////
  //                                                            //
  // IPL                                                        //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  IplImage* m_ipl_image;	   // IPL image pointer
  bool create_IPL_hook();	   // Hook in this object to IPL
#endif

  ////////////////////////////////////////////////////////////////
  //                                                            //
  // State                                                      //
  //                                                            //
  ////////////////////////////////////////////////////////////////

  bool m_empty;			   // Default constructor called
  bool m_open_status;		   // Result of attempt to load from file
  bool m_init_called;		   // Instance initialized and ready to use
  bool m_allocated_mem;		   // Instance performed allocation
  struct timeval m_timestamp;	   // Image timestamp

  // Out of bounds
  Index_Type m_index_type;	   // Indexing mode
  SPU_u8* m_constant_pixel_p;	   // Return this on out-of-bounds access

};

//! Determines if two images are dimensionally and color mode compatable
/*!

  Returns \c true if img0 and img1 are of the same dimension and color
  format. Returns \c false otherwise.

  \param img0 First image
  \param img1 Second image

*/
bool
are_compatable(const SPU_Image& img0, const SPU_Image& img1);

inline 
SPU_Image::SPU_Image(void) 
{
  
  m_color_format     = SPU_Color_Format_Unknown;
  m_width            = 0;
  m_height           = 0;
  m_bpp              = 0;
  m_filename         = 0;
  m_constant_pixel_p = 0;
  m_bytes_per_line   = 0;


  // Storage
  m_raw_mem          = 0;
  m_raw_memsize      = 0;
  m_aligned_mem      = 0;
  m_aligned_memsize  = 0;

  // State
  m_empty            = true;
  m_init_called      = false;
  m_allocated_mem    = false;
  m_open_status      = false;

#if SPU_HAVE_LIBIPL == 1
  m_ipl_image        = 0;
#endif  

}

inline 
SPU_Image::SPU_Image(const SPU_Image& image, 
		     bool allocate_storage, bool with_copy) 
{

  m_color_format     = SPU_Color_Format_Unknown;
  m_width            = 0;
  m_height           = 0;
  m_bpp              = 0;
  m_filename         = 0;
  m_constant_pixel_p = 0;
  m_bytes_per_line   = 0;

  // Storage
  m_raw_mem          = 0;
  m_raw_memsize      = 0;
  m_aligned_mem      = 0;
  m_aligned_memsize  = 0;

  // State
  m_empty            = true;
  m_init_called      = false;
  m_allocated_mem    = false;
  m_open_status      = false;

#if SPU_HAVE_LIBIPL == 1
  m_ipl_image        = 0;
#endif  

  if (image.m_empty == false)
    assign(&image, allocate_storage, with_copy);

}

inline 
SPU_Image::SPU_Image(const char* fname) {

  
  m_color_format     = SPU_Color_Format_Unknown;
  m_width            = 0;
  m_height           = 0;
  m_bpp              = 0;
  m_filename         = 0;
  m_constant_pixel_p = 0;
  m_bytes_per_line   = 0;

  // Storage
  m_raw_mem          = 0;
  m_raw_memsize      = 0;
  m_aligned_mem      = 0;
  m_aligned_memsize  = 0;

  // State
  m_empty            = true;
  m_init_called      = false;
  m_allocated_mem    = false;
  m_open_status      = false;

#if SPU_HAVE_LIBIPL == 1
  m_ipl_image        = 0;
#endif  

  m_open_status = read_pnm(fname);
}

inline 
SPU_Image::SPU_Image(const char*   color_format,
		     const int     width,
		     const int     height,
		     const bool    allocate_storage,
		     const bool    with_copy,
		     SPU_u8* const data,
		     const int     bytes_per_line,
		     const bool    do_not_pad) 
{

  m_color_format     = SPU_Color_Format_Unknown;
  m_width            = 0;
  m_height           = 0;
  m_bpp              = 0;
  m_filename         = 0;
  m_constant_pixel_p = 0;
  m_bytes_per_line   = 0;

  // Storage
  m_raw_mem          = 0;
  m_raw_memsize      = 0;
  m_aligned_mem      = 0;
  m_aligned_memsize  = 0;

  // State
  m_empty            = true;
  m_init_called      = false;
  m_allocated_mem    = false;
  m_open_status      = false;

#if SPU_HAVE_LIBIPL == 1
  m_ipl_image        = 0;
#endif  

  init(color_format, width, height, allocate_storage, with_copy,
       data, bytes_per_line, do_not_pad);
}


inline 
SPU_Image::~SPU_Image(void) {
  if (m_filename) {
    free(m_filename);
    m_filename = 0;
  }
  if (m_allocated_mem) {
    free(m_raw_mem);
    m_raw_mem         = 0;
    m_raw_memsize     = 0;
    m_aligned_mem     = 0;
    m_aligned_memsize = 0;
    m_allocated_mem   = false;
  }
#if SPU_HAVE_LIBIPL == 1
  if (m_ipl_image) {
    iplDeallocate(m_ipl_image, IPL_IMAGE_HEADER);
    m_ipl_image = 0;
  }
#endif
  return;
}

inline bool
SPU_Image::is_valid(void) const {

  if (m_color_format < SPU_Color_Format_COUNT &&
      m_width  > 0  && m_height  > 0 &&
      m_bpp    > 0  && 
      m_raw_mem     && m_raw_memsize > 0 && 
      m_aligned_mem && m_aligned_memsize > 0 &&
      m_init_called) {
    return(true);
  } else {
    return(false);
  }
}

inline bool
SPU_Image::is_empty(void) const 
{
  return(m_empty);
}

inline bool
SPU_Image::has_data(void) const 
{
  if(m_aligned_mem) 
    return(true);
  return(false);
}

inline int
SPU_Image::get_width(void) const 
{
  return(m_width);
}

inline int
SPU_Image::get_height(void) const 
{
  return(m_height);
}

inline int
SPU_Image::get_bytes_per_pixel(void) const 
{
  return(m_bpp);
}

inline bool
SPU_Image::has_allocated(void) const 
{
  return(m_allocated_mem);
}

inline SPU_u8*
SPU_Image::get_data(void) 
{
  return(m_aligned_mem);
}

inline const SPU_u8*
SPU_Image::get_data(void) const 
{
  return(m_aligned_mem);
}

inline SPU_Color_Format 
SPU_Image::get_color_format(void) const 
{
  return(m_color_format);
}

inline const char*
SPU_Image::get_color_format_as_string(void) const 
{
  return SPU_Color_Format_to_str(m_color_format);
}

inline void
SPU_Image::set_width(int width) 
{
  m_width           = width;
  int delta_memsize = m_aligned_memsize - get_size();

  m_aligned_memsize += delta_memsize;
  m_raw_memsize     += delta_memsize;
  return;
}

inline void
SPU_Image::set_height(int height) 
{
  m_height          = height;
  int delta_memsize = m_aligned_memsize - get_size();

  m_aligned_memsize += delta_memsize;
  m_raw_memsize     += delta_memsize;
  return;
}

inline bool
SPU_Image::set_data(SPU_u8* address) 
{
  if (m_allocated_mem) { 
    cerr << "SPU_Image: Failed set_data() request; mem already allocated.\n";
      return(false);
  }
  m_aligned_mem = address;
#if SPU_HAVE_LIBIPL
  // Update the IPL hook
  if (m_ipl_image) 
    m_ipl_image->imageData = (char*) address;
#endif
  return(true);
}

inline int
SPU_Image::get_size(void) const 
{
  return(m_height * m_bytes_per_line);
}

inline bool
SPU_Image::open_status(void) const 
{
  return(m_open_status);
}

inline struct timeval
SPU_Image::get_timestamp(void) const 
{
  return(m_timestamp);
}

inline void 
SPU_Image::set_timestamp(const struct timeval& tv) 
{
  m_timestamp = tv;
  return;
}

inline int
SPU_Image::get_bytes_per_line(void) const 
{
  return(m_bytes_per_line);
}

inline SPU_u8*
SPU_Image::fast_get_pixel(int x, int y)
{
  return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);
}

inline const SPU_u8*
SPU_Image::fast_get_pixel(int x, int y) const
{
  return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);
}

inline SPU_u8*
SPU_Image::get_Torroidal_pixel(int x, int y)
{

  if (x >= m_width)  x %= (m_width-1);
  if (x < 0)         x  = (m_width-1) + (x % m_width);
  
  if (y >= m_height) y %= (m_height-1);
  if (y < 0)         y  = (m_height-1) + (y % m_height);

  return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);
}

inline const SPU_u8*
SPU_Image::get_Torroidal_pixel(int x, int y) const
{
  if (x >= m_width)  x %= (m_width-1);
  if (x < 0)         x  = (m_width-1) + (x % m_width);
  
  if (y >= m_height) y %= (m_height-1);
  if (y < 0)      y   = (m_height-1) + (y % m_height);

  return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);
}

inline SPU_u8*
SPU_Image::get_Extended_pixel(int x, int y)
{
  if (x >= m_width)  x = m_width - 1;
  if (x < 0)       x = 0;

  if (y >= m_height) y = m_height - 1;
  if (y < 0)       y = 0;

  return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);
}

inline const SPU_u8*
SPU_Image::get_Extended_pixel(int x, int y) const
{
  if (x >= m_width)  x = m_width - 1;
  if (x < 0)       x = 0;

  if (y >= m_height) y = m_height - 1;
  if (y < 0)       y = 0;

  return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);
}

inline SPU_u8*
SPU_Image::get_Mirrored_pixel(int x, int y) 
{
  if (x >= m_width)  x = (m_width-2) - (x % m_width);
  if (x < 0)         x = abs(x);

  if (y >= m_height) y = (m_height-2) - (y % m_height);
  if (y < 0)         y = abs(y);

  return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);  
}

inline const SPU_u8*
SPU_Image::get_Mirrored_pixel(int x, int y) const
{
  if (x >= m_width)  x = (m_width-2) - (x % m_width);
  if (x < 0)         x = abs(x);

  if (y >= m_height) y = (m_height-2) - (y % m_height);
  if (y < 0)         y = abs(y);

  return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);  
}

inline SPU_u8*
SPU_Image::get_Constant_pixel(int x, int y)
{
  if (x >= m_width || x < 0 || y >= m_height || y < 0) 
    return(m_constant_pixel_p);
  else
    return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);  
}

inline const SPU_u8*
SPU_Image::get_Constant_pixel(int x, int y) const
{
  if (x >= m_width || x < 0 || y >= m_height || y < 0) 
    return(m_constant_pixel_p);
  else
    return(m_aligned_mem + y*m_bytes_per_line + x*m_bpp);  
}

#if SPU_HAVE_LIBIPL == 1
inline IplImage* 
SPU_Image::ipl(void) 
{
  return(m_ipl_image);
}
#endif

#endif // __SPU_Toolbox__included__Image_h__ include guard
