#include <stdio.h>

#include <utils/ConfigFile.h>
#include <utils/SymbolTable.h>

#include <ImgDisplay/BufferDisplay.h>

class MemoryWindow;
class MemoryDisplay : public BufferDisplay {
 public:
  MemoryDisplay(const char** names, utils::SymbolTable* table, 
                const char* dest_fmt);
  virtual ~MemoryDisplay();

  virtual ImgDisplay::Window*
  createWindow(const char* title, int width, int height,
               ImgDisplay::ColorType src_type);
  virtual void destroyWindow(Window* win);
  virtual void destroyAllWindows();
  virtual ImgDisplay::ColorType defaultWindowColorType() const {
    return _dest_type;
  }

  void registerRegion(unsigned char*);

private:
  char** _names;
  int _num_names;
  int _num_wins;
  utils::SymbolTable* _symbol_table;
  ColorType _dest_type;
};

class MemoryWindow : public BufferWindow
{
 public:
  MemoryWindow(ImgDisplay* display, int width, int height,
               ImgDisplay::ColorType src_type,
               ImgDisplay::ColorType dest_type);
  virtual ~MemoryWindow() {
    delete [] _data;
  }

  virtual bool begin() {
    setDest(_data);
    return true;
  }

  virtual void end() {
  }

private:
  unsigned char* _data;
};

MemoryDisplay::MemoryDisplay(const char** names, utils::SymbolTable* table,
                             const char* dest_fmt)
{
  _symbol_table = table;
  _num_wins = 0;
  _dest_type = stringToColorType(dest_fmt);

  for (_num_names=0; names[_num_names]; _num_names++);

  _names = new char*[_num_names];
  for (int i=0; i<_num_names; i++) 
    _names[i] = utils::String::copy(names[i]);
}

MemoryDisplay::~MemoryDisplay()
{
  for (int i=0; i<_num_names; i++) 
    delete [] _names[i];
  delete [] _names;
}
 
ImgDisplay::Window* MemoryDisplay::createWindow(const char* title,
                                                int width, int height,
                                                ImgDisplay::ColorType src_type)
{
  ImgDisplay::Window* window;
  if (_dest_type == ImgDisplay::UnknownColorType)
    window = new MemoryWindow(this, width, height, src_type, src_type);
  else
    window = new MemoryWindow(this, width, height, src_type, _dest_type);
  _num_wins++;

  return window;
}

void MemoryDisplay::destroyWindow(Window* win)
{
  BufferDisplay::destroyWindow(win);
  _num_wins--;
}

void MemoryDisplay::destroyAllWindows()
{
  BufferDisplay::destroyAllWindows();
  _num_wins = 0;
}

void MemoryDisplay::registerRegion(unsigned char* data)
{
  if (_num_wins < 0 || _num_wins >= _num_names || !_symbol_table)
    return;

  _symbol_table->set(_names[_num_wins], data, NULL, true);
}

MemoryWindow::MemoryWindow(ImgDisplay* display, int width, int height, 
                           ImgDisplay::ColorType src_type,
                           ImgDisplay::ColorType dest_type)
  : BufferWindow(display, width, height,
                 width*getBytesPerPixel(dest_type),
                 src_type, dest_type)
{
  int size = height*width*getBytesPerPixel(dest_type);
  _data = new unsigned char[size];
  ((MemoryDisplay*) display)->registerRegion(_data);
}
    
ImgDisplay* create_ImgDisplay_memory(utils::Generator<ImgDisplay>* gen,
                                     utils::ConfigFile* params,
                                     utils::SymbolTable* globals)
{
  const char* names[11];
  memset(names, 0, 11*sizeof(const char*));
  names[0] = "MemoryDisplayBuffer";
  params->getStrings("names", names, 10);
  return new MemoryDisplay(names, globals, params->getString("dest_fmt",
                                                             "unknown"));
}
