#include <stdio.h>
#include <ImgDisplay/ImgDisplay.h>

#define TEXT_ROWS 11
#define TEXT_COLS 7

static int EvalM(int y0, double slope, int xdelt)
{
  return (int)(y0 + slope*xdelt);
}

static int EvalRM(int x0, double slope, int ydelt)
{
  return (int)(x0 + (1/slope)*ydelt);
}
  
static bool crop_line(int& x1, int& y1, int& x2, int& y2, 
                      int min_x, int min_y, int max_x, int max_y)
{
  //maintain y2 >= y1 invariant
  if (y1>y2) {
    int tempX=x1;
    int tempY=y1;
    x1=x2; y1=y2;
    x2=tempX; y2=tempY;
  }

  //eliminate some dud lines
  if (y2<min_y || y1>max_y)
    return false;
  if (x1<min_x && x2<min_x)
    return false;
  if (x1>max_x && x2>max_x)
    return false;
    
  if (x1==x2) { //vertical line corner case
    if (y1<min_y)
      y1=min_y;
    if (y2>max_y)
      y2=max_y;
    if (x1<min_x || x1>max_x)
      return false;
    if (x2<min_x || x2>max_x)
      return false;
  }
  else {
    double slope = double(y2-y1)/double(x2-x1);
    //horizontal crop
    if (x1<min_x) {
      y1 = EvalM(y1, slope, min_x-x1);
      x1=min_x;
    }
    if (x1>max_x) {
      y1 = EvalM(y1, slope,max_x-x1);
      x1=max_x;
    }
    if (x2<min_x) {
      y2 = EvalM(y2, slope, min_x-x2);
      x2=min_x;
    }
    if (x2>max_x) {
      y2 = EvalM(y2, slope,max_x-x2);
      x2=max_x;
    }
    //vertical crop
    if (y1<min_y) {
      x1=EvalRM(x1, slope, min_y-y1);
      y1=min_y;
      if (x1<min_x || x1>max_x)
        return false;
    }
    if (y2>max_y) {
      x2=EvalRM(x2, slope, max_y-y2);
      y2=max_y;
      if (x2<min_x || x2>max_x)
        return false;
    }
  }
    
  //printf("(%d, %d)-(%d, %d)\n",x1,y1,x2,y2);

  return true;
}

ImgDisplay::Window::Window(ImgDisplay* display, int width, int height)
  : _selections(1)
{
  _point_width = 0; _default_color = White; _display = display;
  _selectable = false;
  _text_width = TEXT_COLS; _text_height=TEXT_ROWS;
  _width = width; _height = height;
}

void ImgDisplay::Window::pixel(int x, int y)
{
  pixel(x, y, _default_color);
}

void ImgDisplay::Window::point(int x, int y, Color color)
{
  if (_point_width < 2) {
    pixel(x,y,color);
    return;
  }
  
  int ps = -int(_point_width)/2;
  int pe =  -ps + (int(_point_width) % 2);
  for (int i=ps;i<pe;i++){
    for (int j=ps;j<pe;j++){
      pixel(x+i,y+j,color);
    }
  }
}

void ImgDisplay::Window::point(int x, int y)
{
  point(x, y, _default_color);
}

void ImgDisplay::Window::fill(Color color)
{
  for (int i=0;i<getHeight();i++) 
    for (int j=0;j<getWidth();j++) 
      pixel(j, i, color);
}

void ImgDisplay::Window::clear()
{
  fill(Black);
}

void ImgDisplay::Window::line(int x1, int y1, int x2, int y2, Color color)
{
  if (!crop_line(x1, y1, x2, y2, 0, 0, getWidth()-1, getHeight()-1))
    return;

  signed int dlX = x2-x1;
  signed int dlY = y2-y1;
  signed int D;

  if (y1>y2) {
    int tempY = y1;
    int tempX = x1;
    y1=y2;
    x1=x2;
    y2=tempY;
    x2=tempX;
    dlY=-dlY;
    dlX=-dlX;
  }

  if (dlX>0) {
    if (dlX>dlY) //the 0 to -45 degree case
      {
        D = -dlX;
        do {
          point(x1, y1, color);
          x1++;
          D+=dlY;
          if (D>0) {
            y1++;
            D-=dlX;
          }
        } while (x1<=x2);
      }
    else //the -45 to -90 degree case
      {
        D = -dlY;
        do {
          point(x1, y1, color);
          y1++;
          D+=dlX;
          if (D>0) {
            x1++;
            D-=dlY;
          }
        } while (y1<=y2);
      }
  }
  else {
    if ((-dlX)>dlY) //the -90 to -135 degree case
      {
        D = dlX; //dlX is now negative
        do {
          point(x1, y1, color);
          x1--;
          D+=dlY;
          if (D>0) {
            y1++;
            D+=dlX;
          }
        } while (x1>=x2);
      }
    else //the -135 to -180 degree case 
      {
        D = -dlY;
        do {
          point(x1, y1, color);
          y1++;
          D-=dlX; //dlX is now negative
          if (D>0) {
            x1--;
            D-=dlY;
          }
        } while (y1<=y2);
      }
  }
}

void ImgDisplay::Window::line(int x1, int y1, int x2, int y2)
{
  line(x1, y1, x2, y2, _default_color);
}

static bool crop_box(int& x1, int& y1, int& x2, int& y2, 
                      int min_x, int min_y, int max_x, int max_y)
{
  int tmp;
  if (x1 > x2) {
    tmp = x2;
    x2 = x1;
    x1 = tmp;
  }
  if (y1 > y2) {
    tmp = y2;
    y2 = y1;
    y1 = tmp;
  }

  if (x2 < min_x || x1 > max_x || y2 < min_y || y1 > max_y)
    return false;

  if (x1 < min_x)
    x1 = min_x;
  if (y1 < min_y)
    y1 = min_y;

  if (x2 > max_x)
    x2 = max_x;
  if (y2 > max_y)
    y2 = max_y;

  return true;
}

void ImgDisplay::Window::fillBox(int x1, int y1, int x2, int y2, Color color)
{
  if (!crop_box(x1, y1, x2, y2, 0, 0, getWidth()-1, getHeight()-1))
    return;

  for (int r=y1; r<=y2; r++)
    for (int c=x1; c<=x2; c++)
      pixel(r, c, color);
}

void ImgDisplay::Window::fillBox(int x1, int y1, int x2, int y2)
{
  fillBox(x1, y1, x2, y2, _default_color);
}

void ImgDisplay::Window::box(int x1, int y1, int x2, int y2, Color color)
{
  line(x1, y1, x1, y2, color);
  line(x1, y2, x2, y2, color);
  line(x2, y2, x2, y1, color);
  line(x2, y1, x1, y1, color);
}

void ImgDisplay::Window::box(int x1, int y1, int x2, int y2)
{
  box(x1, y1, x2, y2, _default_color);
}

/* Some tables we need for DISP_circle - adapted from the GIL via disp_lib. */
static double sin_tab[24] = {
  0, 0.0348995, 0.0697565, 0.104528, 0.139173, 0.173648, 0.207912, 0.241922,
  0.275637, 0.309017, 0.34202, 0.374607, 0.406737, 0.438371, 0.469472, 0.5,
  0.529919, 0.559193, 0.587785, 0.615661, 0.642788, 0.669131, 0.694658, 0.71934
};

static double cos_tab[24] = {
  1, 0.999391, 0.997564, 0.994522, 0.990268, 0.984808, 0.978148, 0.970296,
  0.961262, 0.951057, 0.939693, 0.927184, 0.913545, 0.898794, 0.882948,
  0.866025, 0.848048, 0.829038, 0.809017, 0.788011, 0.766044, 0.743145,
  0.71934, 0.694658
};

void ImgDisplay::Window::circle(int y, int x, double r, Color color)
{
  double sinr1, cosr1, sinr2, cosr2;
  int i;
  int rs, re, cs, ce;
  int radius = (int) r;

  /* Setup clipping region. */
  rs = (y - radius - 1);
  re = (y + radius + 1);
  cs = (x - radius - 1);
  ce = (x + radius + 1);

  /*
   * Circle is currently drawn by using a table of sin and cos values. The
   * table is in 2 degree increments so a short line segment is actually
   * drawn twice.
   */

  /* Adapted from the GIL. */
  for (i = 0; i < 23; i++) {
    sinr1 = sin_tab[i] * radius;
    cosr1 = cos_tab[i] * radius;
    sinr2 = sin_tab[i + 1] * radius;
    cosr2 = cos_tab[i + 1] * radius;
    line((int) (y + sinr1), (int) (x + cosr1),
             (int) (y + sinr2), (int) (x + cosr2),
             color);
    line((int) (y - sinr1), (int) (x - cosr1),
             (int) (y - sinr2), (int) (x - cosr2),
             color);
    line((int) (y + sinr1), (int) (x - cosr1),
             (int) (y + sinr2), (int) (x - cosr2),
             color);
    line((int) (y - sinr1), (int) (x + cosr1),
             (int) (y - sinr2), (int) (x + cosr2),
             color);
    line((int) (y + cosr1), (int) (x + sinr1),
             (int) (y + cosr2), (int) (x + sinr2),
             color);
    line((int) (y - cosr1), (int) (x - sinr1),
             (int) (y - cosr2), (int) (x - sinr2),
             color);
    line((int) (y + cosr1), (int) (x - sinr1),
             (int) (y + cosr2), (int) (x - sinr2),
             color);
    line((int) (y - cosr1), (int) (x + sinr1),
             (int) (y - cosr2), (int) (x + sinr2),
             color);
  }
}

void ImgDisplay::Window::circle(int x, int y, double radius)
{
  circle(x, y, radius, _default_color);
}

/* hpm font: developed by Hans Moravec (hpm)
 * Exception: Hans had leftarrow at 137 octal. We have changed
 * 137 octal to an underscore and put left arrow at 30 octal.
 */

/* From the GIL via disp_lib. */

static int letterdefs[340] = {
  00000000000, 00000000000,       /* null */
  00010101010, 05234100000,       /* 1 downarrow */
  00000003244, 04444320000,       /* 2 alpha */
  00000003442, 07442744040,       /* 3 beta */
  00000001024, 04200000000,       /* 4 or */
  00000000076, 00200000000,       /* 5 not */
  00000001420, 03420140000,       /* 6 epsilon */
  00000007624, 02424240000,       /* 7 pi */
  00000404020, 01024420000,       /* 8 lambda */
  07020202000, 01412141214,       /* tab */
  04040407000, 01610141000,       /* lf */
  00004121010, 01010502000,       /* vt */
  00010107610, 01076000000,       /* ff */
  03440403400, 03422342222,       /* cr */
  00000002452, 05224000000,       /* 16 infinity */
  00030040236, 04242340000,       /* 17 partial */
  00000364040, 04036000000,       /* 20 subset */
  00000740202, 00274000000,       /* 21 superset */
  00000344242, 04200000000,       /* 22 intersection */
  00000424242, 03400000000,       /* 23 union */
  00042427642, 02424100000,       /* 24 forall */
  00076020236, 00202760000,       /* 25 thereexists */
  00000346652, 06634000000,       /* 26 circlestar */
  00010047604, 01020762010,       /* 27 doublearrow  */
  00000102076, 02010000000,       /* 30 leftarrow */
  00000100476, 00410000000,       /* 31 rightarrow */
  03254000000, 00000000000,       /* 32 tilde */
  00002047610, 07620400000,       /* 33 not equal */
  00004102010, 00400340000,       /* 34 lt or eq */
  00020100410, 02000340000,       /* 35 gt or eq */
  00000760076, 00076000000,       /* 36 equivalence */
  00000004224, 01000000000,       /* 37 or */
  00000000000, 00000000000,       /* space */
  00010101010, 01000100000,       /* ! */
  02424240000, 00000000000,       /* double quote */
  00000247624, 02476240000,       /* # */
  01034525034, 01252341000,       /* $ */
  00076620410, 02046460000,       /* % */
  00020505020, 05244320000,       /* & */
  03030600000, 00000000000,       /* quote */
  00002041010, 01004020000,       /* ( */
  00040201010, 01020400000,       /* ) */
  00010523410, 03452100000,       /* * */
  00000101076, 01010000000,       /* + */
  00000000000, 00030306000,       /* , */
  00000000076, 00000000000,       /* - */
  00000000000, 00030300000,       /* . */
  00000020410, 02040000000,       /* / */
  00034424652, 06242340000,       /* 0 */
  00010301010, 01010340000,       /* 1 */
  00034420204, 01020760000,       /* 2 */
  00034420214, 00242340000,       /* 3 */
  00004142444, 07604040000,       /* 4 */
  00076407402, 00242340000,       /* 5 */
  00014204074, 04242340000,       /* 6 */
  00076020404, 01010100000,       /* 7 */
  00034424234, 04242340000,       /* 8 */
  00034424236, 00204300000,       /* 9 */
  00000003030, 00030300000,       /* : */
  00000003030, 00030306000,       /* ; */
  00000041020, 01004000000,       /* < */
  00000007600, 07600000000,       /* = */
  00000201004, 01020000000,       /* > */
  00034420410, 01000100000,       /* ? */
  00034425652, 05640340000,       /* @ */
  00034424276, 04242420000,       /* A */
  00074424274, 04242740000,       /* B */
  00034424040, 04042340000,       /* C */
  00074222222, 02222740000,       /* D */
  00076404074, 04040760000,       /* E */
  00076404074, 04040400000,       /* F */
  00034424040, 04642340000,       /* G */
  00042424276, 04242420000,       /* H */
  00034101010, 01010340000,       /* I */
  00002020202, 00242340000,       /* J */
  00042445060, 05044420000,       /* K */
  00040404040, 04040760000,       /* L */
  00042665242, 04242420000,       /* M */
  00042426252, 04642420000,       /* N */
  00034424242, 04242340000,       /* O */
  00074424274, 04040400000,       /* P */
  00034424242, 05244320000,       /* Q */
  00074424274, 05044420000,       /* R */
  00034424034, 00242340000,       /* S */
  00076101010, 01010100000,       /* T */
  00042424242, 04242340000,       /* U */
  00042424242, 02424100000,       /* V */
  00042424242, 05266420000,       /* W */
  00042422410, 02442420000,       /* X */
  00042422410, 01010100000,       /* Y */
  00076020476, 02040760000,       /* Z */
  01610101010, 01010101600,       /* [ */
  00000402010, 00402000000,       /* \ */
  07010101010, 01010107000,       /* ] */
  00010345210, 01010100000,       /* 136 uparrow */
  00000000000, 00000000076,       /* 137 underbar */
  01414060000, 00000000000,       /* ` */
  00000003402, 03642360000,       /* a */
  00040407442, 04242740000,       /* b */
  00000003442, 04040360000,       /* c */
  00002023642, 04242360000,       /* d */
  00000003442, 07440340000,       /* e */
  00014222070, 02020200000,       /* f */
  00000003442, 04242360234,       /* g */
  00040407442, 04242420000,       /* h */
  00000100010, 01010100000,       /* i */
  00000020002, 00202024234,       /* j */
  00040404244, 07044420000,       /* k */
  00010101010, 01010100000,       /* l */
  00000006452, 05252520000,       /* m */
  00000005462, 04242420000,       /* n */
  00000003442, 04242340000,       /* o */
  00000007442, 04242744040,       /* p */
  00000003442, 04242360202,       /* q */
  00000005462, 04040400000,       /* r */
  00000003640, 03402740000,       /* s */
  00010107610, 01010060000,       /* t */
  00000004242, 04242340000,       /* u */
  00000004242, 04224100000,       /* v */
  00000004242, 05252240000,       /* w */
  00000004224, 01024420000,       /* x */
  00000004242, 04224102040,       /* y */
  00000007604, 03420760000,       /* z */
  00204040410, 00404040200,       /* { */
  01010101010, 01010101010,       /* | */
  04020202010, 02020204000,       /* alt (quad) */
  00010102442, 02410100000,       /* ~ */
  07777777777, 07777777777,       /* 177 Black Box */
  01414363677, 01414141414,       /* 178 Thick Up Arrow */
  01414141414, 07736361414,       /* 179 Thick Down Arrow */
  00034420276, 00242340000,       /*  */
  00000003442, 01642160000,       /*  */
  00042424236, 00202020000,       /*  */
  00000004242, 03602020000,       /*  */
  00044525272, 05252440000,       /*  */
  00000004452, 07252440000,       /*  */
  00052525252, 05252760000,       /*  */
  00000005252, 05252760000,       /*  */
  00052525252, 05252760600,       /*  */
  00000005252, 05252760600,       /*  */
  00042424242, 04242760600,       /*  */
  00000004242, 04242760600,       /*  */
  00076222222, 04242420000,       /*  */
  00000007622, 02242420000,       /*  */
  00000006060, 03422340000,       /*  */
  00042424272, 04646720000,       /*  */
  00000004242, 07246720000,       /*  */
  00042424652, 06242420000,       /*  */
  00000004246, 05262420000,       /*  */
  03400424246, 05262420000,       /*  */
  00034004246, 05262420000,       /*  */
  00076424040, 04040400000,       /*  */
  00000007642, 04040400000,       /*  */
  00076222222, 04242764200,       /*  */
  00000007622, 02242764200,       /*  */
  00000004040, 07442740000,       /*  */
  00052525234, 05252520000,       /*  */
  00000005252, 03452520000,       /*  */
  00074404074, 04242740000,       /*  */
  00000003440, 07442740000,       /*  */
  00034103442, 03410340000,       /*  */
  00000101034, 04234101000,       /*  */
  00074020276, 00202760000,       /*  */
  00000007402, 07602740000,       /*  */
  02476404074, 04040760000,       /*  */
  00024003442, 07440340000,       /*  */
  00000007442, 07442740000,       /*  */
  00076242424, 02424240000,       /*  */
  00000004242, 07642420000,       /*  */
  00000004450, 06050440000        /*  */ 
};


void ImgDisplay::Window::text(int c, int r, const char* t, Color color,
                              bool centered)
{
  if (centered) {
    int string_width = strlen(t)*_text_width;
    text(c-string_width/2, r-_text_height/2, t, color, false);
    return;
  }
  r-=2; // otherwise 0,0 not really in the corner
  c++;  

  int tr, tc, i, j, l, row, col, ct, cw, rt, rw;
  const char *tch;
  int rs, cs, re, ce;
  int letter_height, letter_width; 
  
  /* Draw characters on image, in the specified colors */
  /* Adapted from the GIL via disp_lib. */

  int vscale = (int) (_text_height / ((double)TEXT_ROWS) + 0.5);
  int hscale  = (int) (_text_width / ((double)TEXT_COLS) + 0.5);
  letter_width = TEXT_COLS * hscale;
  letter_height = TEXT_ROWS * vscale;

  rs = r;
  cs = c;
  re = rs + letter_height;
  ce = cs + strlen(t) * letter_width;

  tr = r; tc = c; tch = t;
  while ((i = *tch) != 0) {
    if (i == '\n') {tc = c; tr += letter_height;} else
      if (i == '\b') tc -= letter_width; else
        if (i == '\v') tr -= letter_height; else
          if (i == ' ')  tc += letter_width; else {
            if (i == '\021') i = *(++tch);
            row = tr; col = tc;
            for (l = 0; l < 2; l++) {
              for (j = 0; j < 30; j++) {
                if ((j % 6) == 0) {
                  row += vscale; col = tc;
                }
                if (((1 << (29-j)) & letterdefs[2*i+l]) != 0) {
                  for (ct = 0; ct < hscale; ct++) {
                    cw = col + ct;
                    for (rt = 0; rt < vscale; rt++) {
                      rw = row + rt;
                      if (inBounds(cw, rw)) {
                        point(cw, rw, color);
                      }
                    }
                  }
                }
                col += hscale;
              }
            }
            tc += letter_width;
          }
    tch++;
  }
}

void ImgDisplay::Window::text(int x, int y, const char* t,
                              bool centered)
{
  text(x, y, t, _default_color, centered);
}

int ImgDisplay::Window::popSelections(utils::Vector<Selection>& v)
{
  acquireSelectionLock();
  v = _selections;
  _selections.clear();
  releaseSelectionLock();

  return v.numElems();
}

int ImgDisplay::Window::getSelections(utils::Vector<Selection>& v)
{
  acquireSelectionLock();
  v = _selections;
  releaseSelectionLock();

  return v.numElems();
}

void ImgDisplay::Window::clearSelections()
{
  acquireSelectionLock();
  _selections.clear();
  releaseSelectionLock();
}

int ImgDisplay::Window::numSelections() const
{
  return _selections.numElems();
}

void ImgDisplay::Window::select(int x, int y, unsigned int modifiers)
{
  acquireSelectionLock();
  Selection& s = _selections.append();
  s.x = x; s.y = y; s.modifiers = modifiers;
  releaseSelectionLock();
}

