#include <stdio.h>

#include <utils/Output.h>
#include <utils/ConfigFile.h>

#include <ImgDisplay/FileDisplay.h>

FileDisplay::FileDisplay(const char** names, const char* dest_fmt) 
{
  _dest_type = stringToColorType(dest_fmt);

  for (_num_names=0; names[_num_names]; _num_names++);

  _names = new char*[_num_names];
  for (int i=0; i<_num_names; i++) 
    _names[i] = utils::String::copy(names[i]);

  _num_wins = 0;
}

FileDisplay::~FileDisplay()
{
  for (int i=0; i<_num_names; i++) 
    delete [] _names[i];
  delete [] _names;
}

FileWindow* FileDisplay::newFileWindow(int width, int height,
                                       ImgDisplay::ColorType src_type,
                                       ImgDisplay::ColorType dest_type)
{
  return new FileWindow(getCurrentName(),
                        this, width, height, src_type, dest_type);
}
 
ImgDisplay::Window* FileDisplay::createWindow(const char* title,
                                              int width, int height,
                                              ImgDisplay::ColorType src_type)
{
  FileWindow* window;
  if (_dest_type == ImgDisplay::UnknownColorType) 
    window = newFileWindow(width, height, src_type, src_type);
  else
    window = newFileWindow(width, height, src_type, _dest_type);
  
  _num_wins++;

  return window;
}

void FileDisplay::destroyWindow(Window* win)
{
  BufferDisplay::destroyWindow(win);
  _num_wins--;
}

void FileDisplay::destroyAllWindows()
{
  BufferDisplay::destroyAllWindows();
  _num_wins = 0;
}

const char* FileDisplay::getCurrentName() const
{
  if (_num_wins < 0 || _num_wins > _num_names) 
    return "image";
  return _names[_num_wins];
}

FileWindow::FileWindow(const char* base_name,
                       ImgDisplay* display, int width, int height, 
                       ImgDisplay::ColorType src_type,
                       ImgDisplay::ColorType dest_type)
  : BufferWindow(display, width, height,
                 width*getBytesPerPixel(dest_type),
                 src_type, dest_type)
{
  _file_name = base_name;
  int bpp = getBytesPerPixel(dest_type);
  if (bpp == 1)
    _color = false;
  else if (bpp == 3)
    _color = true;
  else {
    _data = NULL;
    return;
  }

  int size = height*width*bpp;
  _data = new unsigned char[size];
  _file_number = 1;
}

bool FileWindow::begin() {
  setDest(_data);
  return true;
}

void FileWindow::end() {
  if (writeFile()) 
    _file_number++;
}

bool FileWindow::writeFile()
{
  char buffer[200];
  snprintf(buffer, 200, "%s%d.pnm", fileName(), fileNumber());

  utils::Output out;
  if (!out.openFile(buffer)) {
    fprintf(stderr, "Cannot open '%s' for output\n", buffer);
    return false;
  }

  if (_color) {
    out.write("P6\n");
    out.write(getWidth());
    out.write(' ');
    out.write(getHeight());
    out.write("\n255\n");
    out.write((unsigned char*) data(), getHeight()*getWidth()*3);
  } else {
    out.write("P5\n");
    out.write(getWidth());
    out.write(' ');
    out.write(getHeight());
    out.write("\n255\n");
    out.write((unsigned char*) data(), getHeight()*getWidth());
  }

  out.closeFile();
  return true;
}
    
ImgDisplay* create_ImgDisplay_file(utils::Generator<ImgDisplay>* gen,
                                   utils::ConfigFile* params,
                                   utils::SymbolTable* globals)
{
  const char* names[11];
  memset(names, 0, 11*sizeof(const char*));
  names[0] = "image";
  params->getStrings("names", names, 10);
  return new FileDisplay(names, params->getString("dest_fmt", "unknown"));
}
