#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>
#include <X11/Xos.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/extensions/xf86dga.h>
#include <X11/keysym.h>

#include <utils/ConfigFile.h>

#include <ImgDisplay/BufferDisplay.h>

class DGAFSWindow;
class DGAFSDisplay : public ImgDisplay {
 public:
  DGAFSDisplay(int debug, bool no_resolution_switch_hack, bool draw_cursor,
               int cursor_size, int cursor_x, int cursor_y);
  virtual ~DGAFSDisplay();

  virtual Window* createWindow(const char* title, int width, int height,
                               ImgDisplay::ColorType);
  virtual void destroyWindow(Window*);
  virtual void destroyAllWindows();
  virtual ColorType defaultWindowColorType() const { return _dest_type; }

  bool begin();
  void end();
  bool waitForSelection(int& x, int& y, unsigned int& modifiers);

  void shutdown();
  bool handle_event();
  void draw_cursor();
  void clear_cursor();
  void cursor_pixel(unsigned char* point, int bpp);

 private:
  //screen geometry
  int _oldScreenX, _oldScreenY; //old phyical screen dimensions
  int _viewableX, _viewableY; //logical dimensions

  //DGA data
  XDGADevice* _dgaDev;
  XDGAMode* _dgamodes;
  Display* _disp;
  DGAFSWindow* _window;
  int _dmode;
  int _onscreenPage;
  int _numModes;
  long _viewportOffset[2];
  //_page0_x and _page0_y are always 0,0
  int _page1_x, _page1_y;
  int _debug;
  ColorType _dest_type;

  //config
  bool _no_resolution_switch_hack;

  bool _draw_cursor;
  int _cursor_x, _cursor_y;
  int _cursor_size;
  struct PixelStore {
    unsigned char* point;
    unsigned char val;
  };
  utils::Vector<PixelStore> _stored_vals;
};

class DGAFSWindow : public BufferWindow
{
 public:
  DGAFSWindow(ImgDisplay* display, int width, int height, 
              int dest_bytes_per_line,
              ImgDisplay::ColorType src_type,ImgDisplay::ColorType dest_type)
    : BufferWindow(display, width, height, dest_bytes_per_line,
                   src_type, dest_type)
  {}

  virtual void end() {
    ((DGAFSDisplay*) display())->end();
  }

  virtual bool begin() {
    return ((DGAFSDisplay*) display())->begin();
  }

  virtual bool waitForSelection(int& x, int& y, unsigned int& modifiers) {
    return ((DGAFSDisplay*) display())->waitForSelection(x, y, modifiers);
  }
};

DGAFSDisplay::DGAFSDisplay(int debug, bool no_resolution_switch_hack,
                           bool draw_cursor, int cursor_size,
                           int cursor_x, int cursor_y)
  : _stored_vals(2*cursor_size)
{
  _draw_cursor = draw_cursor;
  _cursor_size = cursor_size;
  _cursor_x = cursor_x;
  _cursor_y = cursor_y;
  _window = NULL;
  _debug = debug;
  _no_resolution_switch_hack = no_resolution_switch_hack;
  _disp = NULL;
  _dest_type = UnknownColorType;
}

DGAFSDisplay::~DGAFSDisplay()
{
  destroyAllWindows();
}

ImgDisplay::Window* DGAFSDisplay::createWindow(const char* title,
                                               int width, int height,
                                               ImgDisplay::ColorType src_type)
{
  int majorversion, minorversion;
  int eventBase, errorBase;
  int numModes, i;

  if (_window) {
    fprintf(stderr, "Cannot create more than 1 FullscreenDGA window!\n");
    return NULL;
  }

  if (_debug>=2)
    {
      printf ("_no_resolution_switch_hack=%d\n",_no_resolution_switch_hack);
      printf ("_debug=%d\n", _debug);
    }
  _viewableX = width;
  _viewableY = height;

  if (_cursor_x < 0) 
    _cursor_x = _viewableX/2;
  if (_cursor_y < 0) 
    _cursor_y = _viewableY/2;

  //Make sure we are root
  if (geteuid()!=0) {//root is suid 0
    fprintf(stderr, "Must be suid root to use DGA.\n");
    return NULL;
  }
  if (_debug>=1)
    printf("Beginning DGA initialization...\n");

    //Open the display
  if ( _disp!=NULL || ((_disp = XOpenDisplay(getenv("DISPLAY"))) == NULL) ) {
    fprintf(stderr, "Cannot connect to X server.\n");
    return NULL;
  }
  printf("Created %x\n", (unsigned int) _disp);
  if (_debug>=1)
    printf("Opened X display.\n");
  _oldScreenX = DisplayWidth(_disp, DefaultScreen(_disp));
  _oldScreenY = DisplayHeight(_disp, DefaultScreen(_disp));
  int screen_bpp = DefaultDepth(_disp, DefaultScreen(_disp));
  if (_debug>=2)
    printf ("DisplayWidth=%d, DisplayHeight=%d\n",
            _oldScreenX, _oldScreenY);

  //Get DGA version information
  if (!XDGAQueryExtension(_disp, &eventBase, &errorBase)) {
    fprintf(stderr, "XDGAQueryExtension failed. "
            "Unable to query DGA video extension information\n");
    return NULL;
  }
  if (XDGAQueryVersion(_disp, &majorversion, &minorversion)<0) {
    fprintf(stderr, "XDGAQueryVersion failed. Unable to query DGA version.\n");
    return NULL;
  }
  if (majorversion<2) {
    fprintf(stderr, "DGA version too old. We need DGA 2.0 or better.\n");
    return NULL;
  }
  if (_debug>=1)
    printf("DGA major version: %d. DGA minor \
version: %d\n", majorversion, minorversion);

  //Get display modes
  _dgamodes = XDGAQueryModes(_disp, DefaultScreen(_disp), &numModes);
  if (_debug>=3)
    {
      printf("Found %d modes\n", numModes);
      printf("---------------------------------\n");
      printf ("Displaying available DGA modes:\n");
      printf("---------------------------------\n");
      for (i=0;i<numModes;i++) {
        printf("Displaying mode element %d (%s)\
in _dgamodes array.\n", _dgamodes[i].num, _dgamodes[i].name);
        printf("verticalRefresh=%f, imageWidth=%d,\
imageHeight=%d\n",
               _dgamodes[i].verticalRefresh,  
               _dgamodes[i].imageWidth, 
               _dgamodes[i].imageHeight);
        printf("flags: XDGAConcurrentAccess: ");
        if (_dgamodes[i].flags&XDGAConcurrentAccess)
          printf("yes\n"); 
        else printf("no\n");
        printf("XDGASolidFillRect: ");
        if (_dgamodes[i].flags&XDGASolidFillRect)
          printf("yes\n"); 
        else printf("no\n");
        printf("XDGABlitRect: ");
        if (_dgamodes[i].flags&XDGABlitRect)
          printf("yes\n"); 
        else printf("no\n");
        printf("XDGABlitTransRect: ");
        if (_dgamodes[i].flags&XDGABlitTransRect)
          printf("yes\n");
        else printf("no\n");
        printf("XDGAPixmap: ");
        if (_dgamodes[i].flags&XDGAPixmap)
          printf("yes\n");
        else printf("no\n");
        printf("XDGAInterlaced: ");
        if (_dgamodes[i].flags&XDGAInterlaced)
          printf("yes\n");
        else printf("no\n");
        printf("XDGADoublescan: ");
        if (_dgamodes[i].flags&XDGADoublescan)
          printf("yes\n");
        else printf("no\n");
        if (_dgamodes[i].flags&XDGAPixmap)
          printf("pixmapWidth=%d, pixmapHeight=%d\n",
                 _dgamodes[i].pixmapWidth, 
                 _dgamodes[i].pixmapHeight);
        else printf ("pixmapWidth and pixmapHeight: irrelevant because XDGAPixmap is false.\n");
        printf("bytesPerScanline=%d\n", 
               _dgamodes[i].bytesPerScanline);
        printf("byteOrder(0=LE)=%d, depth=%d, bitsPerPixel=%d\n",
               _dgamodes[i].byteOrder, 
               _dgamodes[i].depth, 
               _dgamodes[i].bitsPerPixel);
        //        printf("redMask=0x%x, greenMask=0x%x, blueMask=0x%x\n",
        //          _dgamodes[i].redMask, 
        //          _dgamodes[i].greenMask, 
        //          _dgamodes[i].blueMask);
        printf("visualClass=%d, viewportWidth=%d, viewportHeight=%d\n",
               _dgamodes[i].visualClass,
               _dgamodes[i].viewportWidth,
               _dgamodes[i].viewportHeight);
        printf("xViewportStep=%d, yViewportStep=%d, maxViewportX=%d, maxViewportY=%d\n",
               _dgamodes[i].xViewportStep,
               _dgamodes[i].yViewportStep, 
               _dgamodes[i].maxViewportX,
               _dgamodes[i].maxViewportY);
        printf("viewportFlags=0x%x\n",
               _dgamodes[i].viewportFlags); 
        printf("---------------------------------\n");
      }
    }

  //choose DGA mode
  //n.b. The array starts with 0, but the lowest DGA mode is 1
  if (_no_resolution_switch_hack==true)
    {
      width=_oldScreenX;
      height=_oldScreenY;
    }
  if (_debug>=2)
    {
      if (_no_resolution_switch_hack==true)
        printf("no_resolution_switch_hack is on.\n");
      else
        printf("no_resolution_switch_hack is off.\n");
    }
    
  _dmode=0;
  while ((_dgamodes[_dmode].viewportWidth!=width)||
         (_dgamodes[_dmode].viewportHeight!=height)||
         (_dgamodes[_dmode].depth!=screen_bpp)||
         (_dgamodes[_dmode].imageHeight<_dgamodes[_dmode].viewportHeight*2))
    {
      _dmode++;
      if (_dmode==numModes)
        break;
    }

  if (_dmode==numModes)
    {
      fprintf(stderr,"Can't find a mode with the exact width and height requested, %d by %d, with\n", width, height);
      fprintf(stderr,"a depth of %d, and a viewportHeight of at least %d.\n", 
              screen_bpp, height*2);
      fprintf(stderr,"Trying to find something that will work.\n");

      _dmode=0;
      while ((_dgamodes[_dmode].viewportWidth<width)||
             (_dgamodes[_dmode].viewportHeight<height)||
             (_dgamodes[_dmode].depth<15)||
             (_dgamodes[_dmode].imageHeight<_dgamodes[_dmode].viewportHeight*2))
        {
          _dmode++;
          if (_dmode==numModes) {
            fprintf(stderr, "Can't find a useful mode. Do you have enough video memory for what you requested?\nIf no_resolution_switch_hack is on, you might want to turn it off.\n");
            return NULL;
          }
        }
      fprintf(stderr,"Found a mode that is similar to what you asked for:\n");
      fprintf(stderr,"width and height: %d by %d, with ", 
              _dgamodes[_dmode].viewportWidth, _dgamodes[_dmode].viewportHeight);
      fprintf(stderr,"depth: %d\n", _dgamodes[_dmode].depth);
    }
  if (_debug>=2)
    printf ("_dmode=%d\n", _dmode+1);
  if (XDGAOpenFramebuffer(_disp, DefaultScreen(_disp))<0) {
    fprintf(stderr, "Error with XDGAOpenFramebuffer.\n");
    return NULL;
  }

  int dest_depth = _dgamodes[_dmode].depth;
  if (dest_depth == 8)
    _dest_type = ImgDisplay::Grey8;
  else if (dest_depth == 15)
    _dest_type = ImgDisplay::RGB1555;
  else if (dest_depth == 16)
    _dest_type = ImgDisplay::RGB565;
  else if (dest_depth == 24)
    _dest_type = ImgDisplay::RGB24;

  _window = new DGAFSWindow(this, width, height,
                            _dgamodes[_dmode].bytesPerScanline,
                            src_type, _dest_type);
  if (!_window->converter()) {
    fprintf(stderr, "Unable to initialize ColorConverter class.\n");
    delete _window;
    _window = NULL;
    return NULL;
  }

  _dgaDev = XDGASetMode(_disp, DefaultScreen(_disp), _dmode+1);
  if (_dgaDev->data == NULL) {
    fprintf(stderr, "_dgaDev->data is NULL. Cannot continue.\n");
    return NULL;
  }
  // it actually looks like this doesn't work if I just put in KeyPress,
  // so I throw in KeyRelease and leave KeyPress just to be a little more
  // informative
  XDGASelectInput(_disp, DefaultScreen(_disp), KeyPress | KeyRelease);

  /// set page
  XDGASetViewport(_disp, DefaultScreen(_disp), 0, 0, XDGAFlipImmediate);
  _onscreenPage=0;
  _page1_x=0;
  _page1_y=_dgamodes[_dmode].viewportHeight;
  if (_debug>=1)
    printf("Switching viewport to default viewport.\n");
  while (XDGAGetViewportStatus(_disp, DefaultScreen(_disp))) {}

  if (_debug>=1)
    {
      printf("Creating ColorConverter with\nbytesPerScanline: %d\n_viewableX, _viewableY: (%d, %d)\nbitsPerPixel: %d\n",
             _dgamodes[_dmode].bytesPerScanline,
             _viewableX, _viewableY,
             (_dgamodes[_dmode].bitsPerPixel+7)/8);
    }

  _viewportOffset[0] = ((_dgamodes[_dmode].viewportWidth - _viewableX)/2)*
    ((_dgamodes[_dmode].bitsPerPixel+7)/8) + 
    ((_dgamodes[_dmode].viewportHeight - _viewableY)/2)
    * _dgamodes[_dmode].bytesPerScanline;
  _viewportOffset[1] = (_dgamodes[_dmode].bytesPerScanline*
                        _dgamodes[_dmode].viewportHeight) + _viewportOffset[0];

  unsigned char* w = _dgaDev->data;
  for (int j=0;j<_dgamodes[_dmode].viewportHeight;j++)
    {
      for (int i=0;i<_dgamodes[_dmode].bytesPerScanline;i++)
        {
          *(w++)=0;
        }
    }
  w = _dgaDev->data + _dgamodes[_dmode].bytesPerScanline*
    _dgamodes[_dmode].viewportHeight;
  for (int j=0;j<_dgamodes[_dmode].viewportHeight;j++)
    {
      for (int i=0;i<_dgamodes[_dmode].bytesPerScanline;i++)
        {
          *(w++)=0;
        }
    }

  //Make sure framebuffer is writable
  *_dgaDev->data=0x55;
  if (*_dgaDev->data!=0x55) {
    fprintf(stderr,
            "Unable to write to framebuffer. Possible buggy DGA implementation. Quitting.\n");
    delete _window;
    _window = NULL;
    return NULL;
  }
  *_dgaDev->data=0x66;
  
  if (*_dgaDev->data!=0x66) {
    fprintf(stderr, "Unable to write to framebuffer. Possible buggy DGA implementation. Quitting.\n");
    delete _window;
    _window = NULL;
    return NULL;
  }
    
  if (_debug>=1)
    printf("End Initialization\n");

  return _window;
}

#define XDGAKeyPress 87
#define XDGAKeyRelease 88

// assumes key release is the only event processed
bool DGAFSDisplay::handle_event()
{
  XEvent xev;

  XNextEvent(_disp, &xev);
  if (xev.type != XDGAKeyPress)
    return true;

  XDGAKeyEvent* key_ev = (XDGAKeyEvent*) &xev;
  KeySym ks = XKeycodeToKeysym(_disp, key_ev->keycode, 0);
  if (ks == XK_Escape) {
    return false;
  }

  if (ks == XK_Tab) {
    if (_draw_cursor)
      clear_cursor();
    else
      draw_cursor();
    _draw_cursor = !_draw_cursor;
  }

  if (_draw_cursor && _cursor_x >= 0 && _cursor_y >= 0 &&
      (ks == XK_Left || ks == XK_Right || ks == XK_Up || ks == XK_Down)) {
    clear_cursor();
    int amount;
    if (key_ev->state & ShiftMask)
      amount = 10;
    else if (key_ev->state & ControlMask)
      amount = 50;
    else
      amount = 1;
    switch (ks) {
    case XK_Left:
      _cursor_x -= amount; break;
    case XK_Right:
      _cursor_x += amount; break;
    case XK_Down:
      _cursor_y += amount; break;
    case XK_Up:
      _cursor_y -= amount; break;
    }
    if (_cursor_x < 0)
      _cursor_x = 0;
    if (_cursor_y < 0)
      _cursor_y = 0;
    if (_cursor_x >= _viewableX)
      _cursor_x = _viewableX-1;
    if (_cursor_y >= _viewableY)
      _cursor_y = _viewableY-1;
    draw_cursor();
  } else if (ks == XK_space) {
    unsigned int modifiers = NoModifiers;
    if (key_ev->state & ShiftMask)
      modifiers |= Shift;
    if (key_ev->state & ControlMask)
      modifiers |= Ctrl;
    if ((key_ev->state & Mod1Mask) || (key_ev->state & Mod2Mask) ||
        (key_ev->state & Mod3Mask) || (key_ev->state & Mod4Mask) ||
        (key_ev->state & Mod5Mask))
      modifiers |= Alt;
    _window->select(_cursor_x, _cursor_y, modifiers);
  }
    
  return true;
}

void DGAFSDisplay::cursor_pixel(unsigned char* point, int bpp)
{
  int i;
  int total = 0;
  PixelStore store;
  for (i=0;i<bpp;i++) {
    store.point = point+i;
    store.val = *(store.point);
    total += (int) store.val;
    _stored_vals.append(store);
  }
  total/=bpp;
  int val;
  if (total > 0xf0)
    val = 0;
  else
    val = 0xff;
  memset(point, val, bpp);
}

void DGAFSDisplay::draw_cursor()
{
  if (!_window || !_disp || _cursor_x < 0 || _cursor_y < 0 ||
      _cursor_x >= _viewableX || _cursor_y >= _viewableY)
    return;

  _stored_vals.clear();

  int start = _cursor_x - _cursor_size/2;
  int end = start + _cursor_size;
  int bpp = BufferWindow::getBytesPerPixel(_dest_type);
  if (start < 0)
    start = 0;
  if (end > _viewableX)
    end = _viewableX;
  
  unsigned char* data = _window->getDest();
  unsigned char* row = data + _cursor_y*_dgamodes[_dmode].bytesPerScanline
    + bpp*start;
  int i;
  for (i=start;i<end;i++) {
    cursor_pixel(row, bpp);
    row += bpp;
  }

  start = _cursor_y - _cursor_size/2;
  end = start + _cursor_size;
  if (start < 0)
    start = 0;
  if (end > _viewableY)
    end = _viewableY;
  for (i=start;i<end;i++) {
    if (i!=_cursor_y) {
      row = data + i*_dgamodes[_dmode].bytesPerScanline + bpp*_cursor_x;
      cursor_pixel(row, bpp);
    }
  }
}

void DGAFSDisplay::clear_cursor()
{
  for (int i=0;i<_stored_vals.numElems();i++) {
    *(_stored_vals[i].point) = _stored_vals[i].val;
  }
  _stored_vals.clear();
}

bool DGAFSDisplay::begin()
{
  if (!_disp)
    return false;
    
  while (XPending(_disp)) {
    if (!handle_event()) {
      shutdown();
      return false;
    }
  }

  if (_window) {
    _window->setDest(_dgaDev->data + _viewportOffset[!_onscreenPage]);
    return true;
  }
  _stored_vals.clear();

  return false;
}

void DGAFSDisplay::end()
{
  if (!_disp)
    return;

  if (_draw_cursor) 
    draw_cursor();
    
  _onscreenPage= (!_onscreenPage);

  if (_onscreenPage==0)
    XDGASetViewport(_disp, DefaultScreen(_disp), 0, 0, 
                    XDGAFlipImmediate);
  else
    XDGASetViewport(_disp, DefaultScreen(_disp), 
                    _page1_x, _page1_y, XDGAFlipImmediate);
  while (XDGAGetViewportStatus(_disp, DefaultScreen(_disp))) {}
  //    if (_debug>=2)
  //      printf("SyncWindow: Flipped _onscreenPage to %d\n",
  //      _onscreenPage);
}

void DGAFSDisplay::shutdown()
{
  if (_window)
    _window->setDest(NULL);

  XDGACloseFramebuffer(_disp, DefaultScreen(_disp));
  int i=0;
  while ((_dgamodes[i].viewportWidth!=_oldScreenX)&&
         (_dgamodes[i].viewportHeight!=_oldScreenY)) {
    i++;
  }
  if (_dmode==_numModes)
    fprintf(stderr, "Error resetting display: _dmode==_numModes after loop\n");
  else {
    //set DGA mode to regular screen geometry.
    //unfortunately setmode 0 doesn't always do this.
    if (_debug>=2)
      printf ("Setting mode# %d\n", i+1);
    _dgaDev = XDGASetMode(_disp, DefaultScreen(_disp), i+1);
  }
  XFree(_dgamodes);
  XDGASetMode(_disp, DefaultScreen(_disp), 0); //back to non-DGA mode
  XCloseDisplay(_disp);
  _disp = NULL;
}
    
void DGAFSDisplay::destroyWindow(Window* win)
{
  if (win != _window)
    return;
  if (!_window)
    return;

  shutdown();

  delete _window;
  _window = NULL;
}

void DGAFSDisplay::destroyAllWindows() 
{
  destroyWindow(_window);
}

bool DGAFSDisplay::waitForSelection(int& x, int& y, unsigned int& modifiers)
{
  if (!_window || !_disp)
    return false;

  bool old_draw_cursor = _draw_cursor;
  _draw_cursor = true;
  draw_cursor();

  utils::Vector<ImgDisplay::Selection> v;
  while (handle_event()) {
    if (_window->popSelections(v)) {
      ImgDisplay::Selection& s = v[v.numElems()-1];
      x = s.x;
      y = s.y;
      modifiers = s.modifiers;

      return true;
    }
  }

  _draw_cursor = old_draw_cursor;
  if (_draw_cursor)
    draw_cursor();
  else
    clear_cursor();

  return false;
}    

ImgDisplay* create_ImgDisplay_DGAFS(utils::Generator<ImgDisplay>* gen,
                                    utils::ConfigFile* params,
                                    utils::SymbolTable* globals)
{
  return new DGAFSDisplay(params->getInt("debug", 2),
                          params->getBool("no_resolution_switch_hack", 0),
                          params->getBool("draw_cursor", false),
                          params->getInt("cursor_size", 5),
                          params->getInt("cursor_x", -1),
                          params->getInt("cursor_y", -1));
}
