/* to test on a single machine do the following

   In window A
     iptshmgr

   In window B
     ./server "iptserver: name=server.cfg"

   In window C
     ./iptmodmon "embedded: name=IPTMonitorModule; spec container = iptclient;"

   In window D
     ./config_test "iptclient" string SharedMemory.test.owner 1

*/    

#include <stdio.h>

#include <utils/List.h>
#include <utils/ConfigFile.h>
#include <utils/Output.h>

#include <ConfigSource/Module.h>
#include <ConfigSource/ConfigSource.h>
#include <ConfigSource/Repository.h>

#include <ipt/ipt.h>
#include <ipt/sharedmem.h>

#include "ModuleStatus.h"

struct ModuleSpec {
  ModuleSpec() { shm = NULL; warned = false; }

  IPSharedMemory *shm;
  utils::String module_name;
  bool warned;
};

class IPTMonitorModule : public Module
{
public:
  IPTMonitorModule(const char* spec);
  virtual ~IPTMonitorModule();

  virtual bool initialize(ConfigSource* config, utils::SymbolTable* table);
  virtual bool run();

  bool load_modules();

private:
  IPCommunicator* _communicator;
  utils::ManagedList<ModuleSpec>* _modules;
  double _interval;
  char* _output_value;
  int _size_output_value;
  bool _verbose;
};

IPTMonitorModule::IPTMonitorModule(const char* spec)
  : Module("IPTMonitorModule", spec)
{
  _modules = NULL;
  _size_output_value = 100;
  _output_value = new char[_size_output_value];
}

IPTMonitorModule::~IPTMonitorModule()
{
  delete _modules;
  delete [] _output_value;
}

static void module_change_notification(const char* name, utils::Type type,
                                       void* data, int num_elems, void* cbdata)
{
  ((IPTMonitorModule*) cbdata)->load_modules();
}

bool IPTMonitorModule::initialize(ConfigSource* config,
                                  utils::SymbolTable* table)
{
  _interval = config->getDouble("interval", 0.5);
  _verbose = config->getBool("verbose", false);

  _communicator = 
    IPCommunicator::Communicator(table, 
                                 config->getString("ipt_spec", "unix"));
  if (!_communicator) {
    fprintf(stderr, "IPTMonitorModule: No communicator\n");
    return false;
  }

  if (!load_modules()) 
    return false;
  config->notify("modules", module_change_notification, this);
    
  return true;
}

bool IPTMonitorModule::load_modules()
{
  ConfigSource* config = getConfigSource();

  utils::ManagedList<ModuleSpec>* new_modules =
    new utils::ManagedList<ModuleSpec>;

  // first get the list of modules from the config source
  int num_values = config->numValues("modules");
  const char** module_name_list = new const char*[num_values];
  num_values = config->getStrings("modules", module_name_list, num_values);

  // next go through and make the new list, copying any old shared memory
  // regions from the old list
  for (int i=0;i<num_values;i++) {
    ModuleSpec* mod = new ModuleSpec;
    mod->module_name = module_name_list[i];
    if (_modules) {
      utils::ListIterator<ModuleSpec> iter(*_modules);
      for (ModuleSpec* cur=iter.first(); cur; cur=iter.next()) {
        if (cur->module_name == mod->module_name) {
          mod->shm = cur->shm;
          mod->warned = cur->warned;
          iter.removeCurrent();
          delete cur;
          break;
        }
      }
    }
    new_modules->append(mod);
  }
  delete [] module_name_list;

  // now, go through the old list and close any unaccounted for memory regions
  if (_modules) {
    utils::ListIterator<ModuleSpec> iter(*_modules);
    for (ModuleSpec* cur=iter.first(); cur; cur=iter.next()) {
      if (cur->shm) {
        _communicator->CloseSharedMemory(cur->shm);
      }
    }
    delete _modules;
  }
  _modules = new_modules;
  
  return true;
}

bool IPTMonitorModule::run()
{
  getConfigSource()->sleep(_interval);

  ConfigSource* repo = Repository::instance(getSymbolTable());
  if (!repo) {
    fprintf(stderr, "IPTMonitorModule: No repository\n");
    return true;
  }

  char buffer[1000];
  char struct_name[300];
  utils::ListIterator<ModuleSpec> iter(*_modules);
  utils::ConfigFile result;
  for (ModuleSpec* cur=iter.first(); cur; cur=iter.next()) {
    if (!cur->shm) {
      snprintf(buffer, 1000, "Modules.%s.host", cur->module_name.getString());
      const char* hostname = repo->getString(buffer, "");
      if (hostname && *hostname) {
        snprintf(buffer, 1000, "managed: name='%s_status@%s|1389';",
                 cur->module_name.getString(), hostname);
      } else {
        snprintf(buffer, 1000, "managed: name=%s_status;",
                 cur->module_name.getString());
      }
      cur->shm = _communicator->OpenSharedMemory(buffer, MODULE_STATUS_FORMAT,
                                                 sizeof(ModuleStatusStruct));
      if (!cur->shm) {
        if (!cur->warned)
          fprintf(stderr,
                  "IPTMonitorModule:  Could not open shared memory %s\n",
                  buffer);
        continue;
      }
    }

    ModuleStatusStruct status;
    if (!cur->shm->FormattedData(&status)) {
      if (!cur->warned) {
        fprintf(stderr,
                "IPTMonitorModule:  Problem getting status for %s\n",
                cur->module_name.getString());
      }
      _communicator->CloseSharedMemory(cur->shm);
      cur->shm = NULL;
      continue;
    }
    cur->warned = false;
    snprintf(buffer, 1000, 
             "{ int update_secs = %d; int update_usecs = %d; "
             "  int last_run_secs = %d; int last_run_usecs = %d; "
             "  int status = %d; }",
             status.update_secs, status.update_usecs, 
             status.last_run_secs, status.last_run_usecs,
             (int) status.status);

    snprintf(struct_name, 300, "struct %s", cur->module_name.getString());
    result.set(struct_name, buffer);
  }

  utils::Output output;
  output.setBuffer(_output_value, _size_output_value,
                   utils::Output::standardResize);
  result.write(output);
  output.write('\0');
  void* buf_ptr;
  int size;
  output.getBuffer(buf_ptr, size);
  _output_value = (char*) buf_ptr;
  _size_output_value = size;
  if (_verbose) {
    printf("struct ModuleStatus {\n%s\n}\n", (const char*) buf_ptr);
  }
  repo->set("struct ModuleStatus", (const char*) buf_ptr);
  
  return true;
}

MODULE_MAIN(iptmodmon, IPTMonitorModule);


