#include <stdio.h>

#include <ipt/ipt.h>
#include <ipt/callbacks.h>
#include <ipt/message.h>

#include <utils/ConfigFile.h>
#include <utils/SymbolTable.h>
#include <utils/String.h>

#include <ConfigSource/Remote.h>

declareHandlerCallback(RemoteConfigSource);
implementHandlerCallback(RemoteConfigSource);

ConfigSource* create_ConfigSource_remote(ConfigSourceGenerator*,
                                         utils::ConfigFile* params,
                                         utils::SymbolTable* globals)
{
  IPCommunicator* com =
    IPCommunicator::Communicator(globals, 
                                 params->getString("ipt_spec",
                                                   "unix"));
  if (!com)
    return NULL;
  const char* mod_name = com->ModuleName();

  const char* spec = params->getString("server", "server");
  int wait = params->getBool("wait", true);
  const char* name = params->getString("name", mod_name);
  IPConnection* conn = com->DirectConnect(spec, wait);
  if (!conn) {
    fprintf(stderr, "Could not connect to config server via '%s'\n", spec);
    return NULL;
  }
  RemoteConfigSource* intf = new RemoteConfigSource(com, conn, name);
  if (!intf->valid()) {
    delete intf;
    return NULL;
  }

  return intf;
}

RemoteConfigSource::RemoteConfigSource(IPCommunicator* com, IPConnection* conn,
                                       const char* name)
{
  _communicator = com;
  _source = conn;

  com->RegisterMessage(GET_CONFIGURATION_MSG, GET_CONFIGURATION_FMT);
  com->RegisterMessage(CONFIGURATION_MSG, CONFIGURATION_FMT);
  com->RegisterMessage(ATTACH_CONFIG_VAR_MSG, ATTACH_CONFIG_VAR_FMT);
  com->RegisterMessage(CONFIG_VAR_ATTACHED_MSG, CONFIG_VAR_ATTACHED_FMT);
  com->RegisterMessage(DETACH_CONFIG_VAR_MSG, DETACH_CONFIG_VAR_FMT);
  com->RegisterMessage(CONFIG_VAR_DETACHED_MSG, CONFIG_VAR_DETACHED_FMT);
  com->RegisterMessage(CONFIG_VAR_SET_MSG, CONFIG_VAR_SET_FMT);

  com->RegisterHandler(CONFIG_VAR_SET_MSG,
                       new HandlerCallback(RemoteConfigSource)
                       (this, &RemoteConfigSource::handle_set));

  _name = utils::String::copy(name);
  initialize();
}

bool RemoteConfigSource::initialize()
{
  const char* values = NULL;
  IPMessage* msg =
    _communicator->Query(_source, GET_CONFIGURATION_MSG, (void*) &_name,
                         CONFIGURATION_MSG, 1.0);
  if (!msg || !(values = *(char**) msg->FormattedData()) || !*values ||
      !parse(values)) {
    _name = NULL;
    return false;
  }
  instrument();
  return true;
}

RemoteConfigSource::~RemoteConfigSource()
{
  if (_name)
    delete [] _name;
}

void RemoteConfigSource::handle_set(IPMessage* msg)
{
  ConfigVarSetStruct* cvs = (ConfigVarSetStruct*) msg->FormattedData();
  set(cvs->var_name, cvs->value);
}

bool RemoteConfigSource::attach(utils::Type type, const char* name,
                                void* data, int max_num,
                                ConfigCallback notify, void* cb_data,
                                void* default_data, int num_defaults)
{
  if (!ActiveConfigSource::attach(type, name, data, max_num, notify, cb_data,
                                  default_data, num_defaults))
    return false;
  
  ConfigVarSpecStruct cvs;
  cvs.config_name = _name;
  cvs.var_name = name;
  IPMessage* msg = _communicator->Query(_source, ATTACH_CONFIG_VAR_MSG, &cvs,
                                        CONFIG_VAR_ATTACHED_MSG);
  return msg && (*(int*) msg->FormattedData() != 0);
}

bool RemoteConfigSource::processEvents(double timeout)
{
  _communicator->Idle(timeout);

  return true;
}

bool RemoteConfigSource::refresh()
{
  return initialize();
}
