#include <stdio.h>

#include <ipt/ipt.h>
#include <ipt/callbacks.h>
#include <ipt/message.h>
#include <ipt/server.h>

#include <utils/ConfigFile.h>
#include <utils/ConfigElem.h>
#include <utils/StructElem.h>
#include <utils/SymbolTable.h>
#include <utils/String.h>
#include <utils/Output.h>
#include <utils/Input.h>

#include "messages.h"

#include <ConfigSource/Active.h>
#include <ConfigSource/Repository.h>

class ServerRegistration;

class IPTServerConfigSource : public ActiveConfigSource
{
 public:
  IPTServerConfigSource(IPCommunicator* comm);
  virtual ~IPTServerConfigSource();

  virtual bool processEvents(double timeout=-1);

  void setVerbosity(bool v) { _verbose = v; }

 private:
  void handle_get(IPMessage* msg);
  void handle_get_num_values(IPMessage* msg);
  void handle_set(IPMessage* msg);
  void handle_parse_set(IPMessage* msg);
  void handle_watch(IPMessage* msg);
  void handle_unwatch(IPMessage* msg);

 private:
  IPCommunicator* _communicator;
  utils::ManagedList<ServerRegistration>* _registrations;
  bool _verbose;
};

declareHandlerCallback(IPTServerConfigSource);
implementHandlerCallback(IPTServerConfigSource);

class ServerRegistration 
{
public:
  ServerRegistration(IPTServerConfigSource* source, const char* name,
                     IPCommunicator* comm, IPConnection* dest) {
   _source = source; _name = name; _watch = NULL; _dest = dest; _comm = comm;
   // this mechanism is here to handle 64 bit architectures
   // is there a better way to create a "unique" address?
   _id = int(((unsigned long)this) & 0xffffffff);
   
  }
  ~ServerRegistration() {
    utils::ConfigElem* elem = _source->getFile().lookup(_name.getString());
    if (_source && _watch && elem) {
      _source->unwatch(elem, _watch);
    }
  }

  void setWatch(ActiveElem* watch) { _watch = watch; }
  void callback(const char* name) {
    struct ConfigWatchSet set;

    utils::ConfigElem* elem = _source->getFile().lookup(_name.getString());
    if (!elem) 
      return;

    utils::Output output;
    char* res = new char[50];
    output.setBuffer(res, 50, utils::Output::standardResize);
    elem->writeData(output);
    output.write('\0');
    void* out_buf;
    int size;
    output.getBuffer(out_buf, size);

    set.handle = _id;
    set.type = elem->getTypeId().getKey();
    set.value = (char*) out_buf;
    // printf("Watching handle %d\n", set.handle);

    int send_res = _comm->SendMessage(_dest, CS_WATCH_SET_MSG, &set);

    delete [] (char*) out_buf;
    if (send_res < 0) {
      printf("Invalid watch set removed\n");
      delete this;
    }
  }

  unsigned int getID() const { return _id; }

private:
  IPTServerConfigSource* _source;
  utils::String _name;
  ActiveElem* _watch;
  IPCommunicator* _comm;
  IPConnection* _dest;
  unsigned int _id;
};

#define MAX_PEERS 10

static IPCommunicator* create_server(IPConfigFile* params,
                                     IPSymbolTable* globals)
{
  const char* module_name = (const char*) globals->get("ModuleName");
  if (!module_name)
    module_name = params->getString("ModuleName", "Module");
  const char* message_file = params->getString("message_file", NULL);
  if (message_file && !*message_file)
    message_file = NULL;
  const char* domain_name = params->getString("domain_name", NULL);
  if (domain_name && !*domain_name)
    domain_name = NULL;
  const char* log_file = params->getString("log_file", NULL);
  if (log_file && !*log_file)
    log_file = NULL;

  const char* peer_names[MAX_PEERS];
  const char* peer_hosts[MAX_PEERS];
  int n1 = params->getStrings("peer_names", peer_names, MAX_PEERS, NULL, 0);
  int n2 = params->getStrings("peer_hosts", peer_hosts, MAX_PEERS, NULL, 0);
  int num_peers = MIN(n1,n2);
  IPDomainSpec peers[MAX_PEERS+1];
  for (int i=0;i<num_peers;i++) {
    peers[i].name = (char*) peer_names[i];
    peers[i].host = (char*) peer_hosts[i];
  }
  peers[num_peers].name = peers[num_peers].host = NULL;
    
  IPCommunicator* res =
    new IPServer(module_name, message_file, domain_name, peers, log_file);
  globals->set("Communicator", res, NULL);

  const char* output = params->getString("output", "");
  if (output && *output) 
    res->SetOutput(output);
  
  return res;
}

ConfigSource* create_ConfigSource_iptserver(ConfigSourceGenerator*,
                                            utils::ConfigFile* params,
                                            utils::SymbolTable* globals)
{
  IPCommunicator* server = create_server(params, globals);
  if (!server)
    return NULL;

  IPTServerConfigSource* intf = new IPTServerConfigSource(server);
  if (!intf->open(params->getString("name", "server.conf"))) {
    delete intf;
    return NULL;
  }
  intf->setVerbosity(params->getBool("verbose", false));
      
  Repository::set(globals, intf);
  return intf;
}

IPTServerConfigSource::IPTServerConfigSource(IPCommunicator* com)
{
  _communicator = com;
  _registrations = new utils::ManagedList<ServerRegistration>;

  com->RegisterMessage(CS_GET_MSG, CS_GET_FMT);
  com->RegisterMessage(CS_GOTTEN_MSG, CS_GOTTEN_FMT);
  com->RegisterMessage(CS_GET_NUM_VALUES_MSG, CS_GET_NUM_VALUES_FMT);
  com->RegisterMessage(CS_NUM_VALUES_MSG, CS_NUM_VALUES_FMT);
  com->RegisterMessage(CS_PARSE_SET_MSG, CS_PARSE_SET_FMT);
  com->RegisterMessage(CS_SET_MSG, CS_SET_FMT);
  com->RegisterMessage(CS_WATCH_MSG, CS_WATCH_FMT);
  com->RegisterMessage(CS_WATCHED_MSG, CS_WATCHED_FMT);
  com->RegisterMessage(CS_WATCH_SET_MSG, CS_WATCH_SET_FMT);
  com->RegisterMessage(CS_UNWATCH_MSG, CS_UNWATCH_FMT);
  com->RegisterMessage(CS_STATUS_MSG, CS_STATUS_FMT);

  com->RegisterHandler(CS_GET_MSG, 
                       new HandlerCallback(IPTServerConfigSource)
                       (this, &IPTServerConfigSource::handle_get));
  com->RegisterHandler(CS_GET_NUM_VALUES_MSG, 
                       new HandlerCallback(IPTServerConfigSource)
                       (this, &IPTServerConfigSource::handle_get_num_values));
  com->RegisterHandler(CS_SET_MSG, 
                       new HandlerCallback(IPTServerConfigSource)
                       (this, &IPTServerConfigSource::handle_set));
  com->RegisterHandler(CS_PARSE_SET_MSG, 
                       new HandlerCallback(IPTServerConfigSource)
                       (this, &IPTServerConfigSource::handle_parse_set));
  com->RegisterHandler(CS_WATCH_MSG, 
                       new HandlerCallback(IPTServerConfigSource)
                       (this, &IPTServerConfigSource::handle_watch));
  com->RegisterHandler(CS_UNWATCH_MSG, 
                       new HandlerCallback(IPTServerConfigSource)
                       (this, &IPTServerConfigSource::handle_unwatch));

  instrument();
}

IPTServerConfigSource::~IPTServerConfigSource()
{
  delete _registrations;
  delete _communicator;
}

void IPTServerConfigSource::handle_get(IPMessage* msg)
{
  const char* name = *(char**) msg->FormattedData();

  ConfigReplyValue reply;

  utils::ConfigElem* elem = getFile().lookup(name);
  if (!elem) {
    reply.type = -1;
    reply.value = utils::String::copy("");
  } else {
    utils::Output output;
    char* res = new char[50];
    output.setBuffer(res, 50, utils::Output::standardResize);
    elem->writeData(output);
    output.write('\0');

    void* out_buf;
    int size;
    output.getBuffer(out_buf, size);

    reply.value = (char*) out_buf;
    reply.type = elem->getTypeId().getKey();
  }

  _communicator->Reply(msg, CS_GOTTEN_MSG, (void*) &reply);
  delete [] reply.value;
}

void IPTServerConfigSource::handle_get_num_values(IPMessage* msg)
{
  const char* name = *(char**) msg->FormattedData();

  int reply;
  utils::ConfigElem* elem = getFile().lookup(name);
  if (!elem) 
    reply = -1;
  else
    reply = elem->numValues();

  _communicator->Reply(msg, CS_NUM_VALUES_MSG, (void*) &reply);
}

void IPTServerConfigSource::handle_parse_set(IPMessage* msg)
{
  struct ConfigParseSet* ps = (ConfigParseSet*) msg->FormattedData();

  utils::Managed::deferDeletions();
  int res = (getFile().set(ps->name, ps->value) != NULL);
  _communicator->Reply(msg, CS_STATUS_MSG, (void*) &res);
  utils::Managed::undeferDeletions();
}
  
void IPTServerConfigSource::handle_set(IPMessage* msg)
{
  struct ConfigSet* ps = (ConfigSet*) msg->FormattedData();

  utils::Managed::deferDeletions();

  int res;
  utils::Input input;
  utils::ConfigElem* setter = NULL;
  utils::Type t;

  if (_verbose)
    printf("Setting %s = %s\n", ps->name, ps->value);

  utils::ConfigElem* elem =
    ((utils::StructElem*) getFile().getRoot())->makeElem(ps->name, true);
  if (!elem) {
    res = -1;
    goto sendit;
  }
  elem->ref();

  t = utils::Type(ps->type);
  setter = (utils::ConfigElem*) t.createInstance();
  setter->ref();
  if (!setter ||
      !setter->getTypeId().isDerivedFrom(utils::ConfigElem::getClassTypeId())){
    res = -2;
    goto sendit;
  }

  input.setBuffer((void*) ps->value, strlen(ps->value));
  if (!setter->readData(input)) {
    res = -3;
    goto sendit;
  }

  if (!getFile().set(t, ps->name, setter->getData(), setter->numValues())) {
    res = -4;
    goto sendit;
  }
  res = 0;

 sendit:
  _communicator->Reply(msg, CS_STATUS_MSG, (void*) &res);
  if (setter)
    setter->unref();
  if (elem)
    elem->unref();
  utils::Managed::undeferDeletions();
}

static void watch_cb(const char* name, utils::Type, void*, int, void* cbdata)
{
  ((ServerRegistration*) cbdata)->callback(name);
}

void IPTServerConfigSource::handle_watch(IPMessage* msg)
{
  const char* name = *(char**) msg->FormattedData();

  ConfigWatched reply;
  utils::ConfigElem* elem = getFile().lookup(name);
  if (!elem)
    elem = getFile().makeElem(name);
  if (!elem) {
    reply.handle = -1;
    reply.type = -1;
    reply.value = utils::String::copy("Does not exist");
  } else {
    utils::Output output;
    char* res = new char[50];
    output.setBuffer(res, 50, utils::Output::standardResize);
    elem->writeData(output);
    output.write('\0');

    void* out_buf;
    int size;
    output.getBuffer(out_buf, size);

    reply.value = (char*) out_buf;
    reply.type = elem->getTypeId().getKey();

    ServerRegistration* reg =
      new ServerRegistration(this, name, _communicator, msg->Connection());
    ActiveElem* w = watch(utils::Type::badType(), name,
                          NULL, 0, watch_cb, reg, NULL, 0);
    if (w) {
      reply.handle = reg->getID();
    } else {
      reply.handle = -1;
      reply.type = -1;
      reply.value = utils::String::copy("Can't set watch");
    }
    reg->setWatch(w);
    _registrations->append(reg);
  }
  
  _communicator->Reply(msg, CS_WATCHED_MSG, (void*) &reply);
  delete [] reply.value;
}
  
void IPTServerConfigSource::handle_unwatch(IPMessage* msg)
{
  unsigned int id = *(unsigned int*) msg->FormattedData();
  // check registration list so we don't delete non-existent memory from
  // previous runs of the server
  utils::ListIterator<ServerRegistration> iter(*_registrations);
  for(ServerRegistration* reg=iter.first(); reg; reg=iter.next()) {
    if (reg->getID() == id) {
      iter.removeCurrent();
      delete reg;
      return;
    }
  }
}

bool IPTServerConfigSource::processEvents(double timeout)
{
  return _communicator->Idle(timeout);
}
  
