#include <stdio.h>

#include <ipt/ipt.h>
#include <ipt/callbacks.h>
#include <ipt/message.h>
#include <ipt/connection.h>
#include <ipt/sharedmem.h>

#include <utils/SymbolTable.h>
#include <utils/Generator.h>
#include <utils/ConfigFile.h>
#include <utils/ConfigElem.h>
#include <utils/StructElem.h>
#include <utils/String.h>
#include <utils/Output.h>
#include <utils/Input.h>
#include <utils/Dict.h>
#include <utils/List.h>
#include <utils/Time.h>

#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#include "messages.h"
#include "ModuleStatus.h"

#include <ConfigSource/Active.h>
#include <ConfigSource/Repository.h>
#include <TimeSource/TimeSource.h>

#define MEMORY_MANAGER_PORT 1389

class ClientCallback;
class IPTClientConfigSource : public ConfigSource
{
 public:
  IPTClientConfigSource(const char* module_name,
                        IPCommunicator* comm, IPConnection* conn,
                        utils::ConfigFile* params, utils::SymbolTable* table);
  virtual ~IPTClientConfigSource();

  virtual int get(utils::Type, const char* name, void* data, int max_num=1,
                  void* default_data=NULL, int num_defaults=1);

  virtual bool attach(utils::Type, const char* name, void* data, int max_num=1,
                      ConfigCallback notify=NULL, void* cbdata = NULL,
                      void* default_data=NULL, int num_defaults=1);

  virtual int numValues(const char* name);

  virtual bool set(const char* name, const char* value);
  virtual bool set(utils::Type, const char* name, void* data,
                   int num_values);

  virtual bool processEvents(double timeout=-1) ;
  virtual bool refresh() { return true; }
  virtual void reportState(ModuleState);
  virtual void setStatus(float status) {
    _module_status_struct.status = status;
  }
  virtual void setConfidence(float confidence) {
    _module_status_struct.confidence = confidence;
  }
  virtual void setStatusMessage(const char* msg) {
    _next_status_msg = _cached_status_msg = msg;
  }

  utils::ConfigElem* putData(utils::Type dest_type, 
                             utils::Type src_type, const char* in_value);
  void unwatch(int handle);

 private:
  utils::ConfigElem* getElem(utils::Type);
  void handle_watch_set(IPMessage* msg);

private:
  IPCommunicator* _communicator;
  IPConnection* _server;
  utils::Dict<utils::ConfigElem*> _unpack_elems;
  utils::ManagedList<ClientCallback> _callbacks;
  IPSharedMemory* _module_status;
  ModuleStatusStruct _module_status_struct;
  ModuleState _last_state;
  int _num_cycles_in_average, _max_cycles_in_average;
  float _minimum_status_interval;
  utils::String _next_status_msg, _cur_status_msg, _cached_status_msg;
  utils::Time _last_propagation;

#ifdef HAVE_PTHREAD_H
  pthread_t _block_check_task;
  pthread_mutex_t _mem_mutex;
  double _blocked_time;
  bool _started_cycles;
  bool _block_test_running;
  bool _do_block_test;

  static void* thread_entry(void*);
  void block_check();
#endif
};

declareHandlerCallback(IPTClientConfigSource);
implementHandlerCallback(IPTClientConfigSource);

class ClientCallback {
public:
  ClientCallback(const char* name, IPTClientConfigSource* src,
                 utils::Type out_type, void* data, int max_num,
                 ConfigCallback notify, void* cbdata, int handle) {
    _src = src; _name = name;
    _out_type = out_type; _out_data = data; _out_max_num = max_num;
    _notify_cb = notify; _notify_cb_data = cbdata;
    _handle = handle;
  }
  ~ClientCallback() {
    _src->unwatch(_handle);
  }
  
  void notify(const char* name, utils::Type type, const char* value)
  {
    utils::ConfigElem* elem = _src->putData(_out_type, type, value);
    if (elem) {
      if (_out_data && _out_max_num)
        elem->getValue(_out_type, _out_data, _out_max_num);
      if (_notify_cb) 
        (*_notify_cb)(name, elem->getTypeId(),
                      elem->getData(), elem->numValues(), _notify_cb_data);
    }
  }

  int getHandle() const { return _handle; }
  const char* getName() const { return _name.getString(); }
  

public:
  utils::String _name;
  IPTClientConfigSource* _src;
  utils::Type _out_type;
  void* _out_data;
  int _out_max_num;
  ConfigCallback _notify_cb;
  void* _notify_cb_data;
  int _handle;
};

static utils::SymbolTable* GlobalTable;

static IPSharedMemory*
create_shmem_repository(IPGenerator<IPSharedMemory>* gen,
                        utils::ConfigFile* params, utils::SymbolTable* table)
{
  IPCommunicator* com = (IPCommunicator*) table->get("Communicator");
  if (!com)
    return NULL;

  ConfigSource* repo = Repository::instance(NULL);
  if (!repo) {
    printf("RepositorySharedMemory:  No repository\n");
    return NULL;
  }

  const char* name = params->getString("name");
  if (!*name) {
    printf("RepositorySharedMemory:  requires name field\n");
    return NULL;
  }
  char buffer[500];
  snprintf(buffer, 500, "SharedMemory.%s", name);
 
  utils::ConfigFile mem_struct;
  if (!repo->getStruct(buffer, mem_struct)) {
    printf("RepositorySharedMemory:  no region named %s in repository\n",
           name);
    return NULL;
  }

  const char* owner_module = mem_struct.getString("owner");
  if (!*owner_module) {
    printf("RepositorySharedMemory:  region %s has no owner\n", name);
    return NULL;
  }

  const char* module_name =
    params->getString("module_name",
                      (const char*) GlobalTable->get("ModuleName"));
  if (!*module_name) {
    printf("RepositorySharedMemory: this module must have a name for %s\n", 
           name);
    return NULL;
  }

  char mem_spec[500];
  if (!strcmp(module_name, owner_module)) {
    snprintf(mem_spec, 500, "managed: bool owner=true; string name=%s;\n",
             name);
  } else {
    snprintf(buffer, 500, "Modules.%s", owner_module);
    utils::ConfigFile owner_struct;
    if (!repo->getStruct(buffer, owner_struct)) {
      printf("RepositorySharedMemory: region %s has invalid owner %s\n",
             name, owner_module);
      return NULL;
    }
    const char* machine =
      owner_struct.getString("host", com->ServerHostName());
    int port = mem_struct.getInt("manager_port", MEMORY_MANAGER_PORT);
    snprintf(mem_spec, 500, "managed: string name='%s@%s|%d';", name,
             machine, port);
  }

  utils::ConfigFile mem_spec_struct;
  printf("Requesting memory %s\n", mem_spec);
  if (!mem_spec_struct.parse(mem_spec)) {
    printf("RepositorySharedMemory: could not parse '%s'\n",
           mem_spec);
    return NULL;
  }

  IPSharedMemory* mem =
    com->SharedMemoryGenerator()->interface(&mem_spec_struct, table);
  if (!mem) {
    printf("RepositorySharedMemory: could not initialize shared memory '%s'\n",
           mem_spec);
  }
  return mem;
}

void register_repository_shared_memory(IPCommunicator* com)
{
  com->SharedMemoryGenerator()->registerInterface("repository",
                                                  create_shmem_repository,
                                                  NULL);
}

ConfigSource* create_ConfigSource_iptclient(ConfigSourceGenerator*,
                                            utils::ConfigFile* params,
                                            utils::SymbolTable* globals)
{
  // we do this little dance so we can get the spec, but also
  // try and access ipt_spec.module_name later.  If we don't do it like
  // this, bad things happen when the default is there.
  const char* ipt_spec = params->getString("ipt_spec", "unix");

  IPCommunicator* com = IPCommunicator::Communicator(globals, ipt_spec);
  if (!com)
    return NULL;

  const char* module_def;
  if (!strcmp(ipt_spec, "unix")) {
    module_def = (const char*) globals->get("ModuleName");
  } else {
    module_def = params->getString("string ipt_spec.module_name",
                                   (const char*) globals->get("ModuleName"));
  }
  const char* module_name = params->getString("module_name", module_def);
  globals->set("ModuleName", utils::String::copy(module_name),
               utils::SymbolTable::stringManager, true);
  GlobalTable = globals;

  if (!globals->get("registered_repository_shmem")) {
    register_repository_shared_memory(com);
    globals->set("registered_repository_shmem", (void*) 1);
  }

  const char* spec = params->getString("server", "server");
  int wait = params->getBool("wait", true);
  IPConnection* conn = com->DirectConnect(spec, wait);
  if (!conn) {
    fprintf(stderr, "Could not connect to config server via '%s'\n", spec);
    return NULL;
  }

  IPTClientConfigSource* intf =
    new IPTClientConfigSource(module_name, com, conn, params, globals);

  return intf;
}

IPTClientConfigSource::IPTClientConfigSource(const char* module_name,
                                             IPCommunicator* com,
                                             IPConnection* conn,
                                             utils::ConfigFile* params,
                                             utils::SymbolTable* table)
{
  _communicator = com;
  _server = conn;
  _last_state = ERROR;

  com->RegisterMessage(CS_GET_MSG, CS_GET_FMT);
  com->RegisterMessage(CS_GOTTEN_MSG, CS_GOTTEN_FMT);
  com->RegisterMessage(CS_GET_NUM_VALUES_MSG, CS_GET_NUM_VALUES_FMT);
  com->RegisterMessage(CS_NUM_VALUES_MSG, CS_NUM_VALUES_FMT);
  com->RegisterMessage(CS_PARSE_SET_MSG, CS_PARSE_SET_FMT);
  com->RegisterMessage(CS_SET_MSG, CS_SET_FMT);
  com->RegisterMessage(CS_WATCH_MSG, CS_WATCH_FMT);
  com->RegisterMessage(CS_WATCHED_MSG, CS_WATCHED_FMT);
  com->RegisterMessage(CS_WATCH_SET_MSG, CS_WATCH_SET_FMT);
  com->RegisterMessage(CS_UNWATCH_MSG, CS_UNWATCH_FMT);
  com->RegisterMessage(CS_STATUS_MSG, CS_STATUS_FMT);

  com->RegisterHandler(CS_WATCH_SET_MSG, 
                       new HandlerCallback(IPTClientConfigSource)
                       (this, &IPTClientConfigSource::handle_watch_set));


  _module_status = NULL;
  if (params->getBool("report_status", true)) {
    // printf("Reporting status %s\n", module_name);
    if (*module_name) {
      _minimum_status_interval = params->getDouble("minimum_status_interval",
                                                   0.2);
      _max_cycles_in_average = params->getInt("max_cycles_in_average", 100);
      Repository::set(table, this);
      char buffer[300];
      snprintf(buffer, 300, 
               "managed: string name=%s_status; bool owner=true;",
               module_name);
      for (int i=0;i<5;i++) {
        _module_status =
          com->OpenSharedMemory(buffer, MODULE_STATUS_FORMAT,
                                sizeof(ModuleStatusStruct) +
                                MAX_STATUS_MESSAGE_SIZE + sizeof(int));
        if (_module_status)
          break;
        fprintf(stderr, "IPTClientConfigSource::IPTClientConfigSource: "
                "Could not open %s, retry %d/5 in 1 second\n", buffer,
                i+1);
        utils::Time::sleep(1.0);
      }
      memset(&_module_status_struct, 0, sizeof(_module_status_struct));
      _num_cycles_in_average = 0;
      reportState(NOT_RUNNING);
    }
  }

#ifdef HAVE_PTHREAD_H  
  _do_block_test = params->getBool("do_block_test", false);
  _blocked_time = params->getDouble("blocked_time", 1.0);
  _block_test_running = false;
  _started_cycles = false;
#endif
}

IPTClientConfigSource::~IPTClientConfigSource()
{
  utils::DictIterator<utils::ConfigElem*> iter(_unpack_elems);
  for (utils::ConfigElem* elem=iter.first(); elem; elem=iter.next()) 
    elem->unref();

#ifdef HAVE_PTHREAD_H
  if (_do_block_test && _block_test_running) {
    pthread_mutex_destroy(&_mem_mutex);
    _block_test_running = false;
    pthread_cancel(_block_check_task);
  }
#endif  
}

#ifdef HAVE_PTHREAD_H
void* IPTClientConfigSource::thread_entry(void* data)
{
  ((IPTClientConfigSource*) data)->block_check();
  return NULL;
}

void IPTClientConfigSource::block_check()
{
  while (_block_test_running) {
    utils::Time::sleep(_blocked_time);
    if (_started_cycles) {
      utils::Time now = TimeSource::now();
      utils::Time last = utils::Time(_module_status_struct.last_run_secs,
                                     _module_status_struct.last_run_usecs);
      if ((now - last).getValue() > _blocked_time) {
        reportState(BLOCKED);
      }
    }
  }
}
#endif

utils::ConfigElem* IPTClientConfigSource::getElem(utils::Type t)
{
  utils::ConfigElem* elem;

  if (_unpack_elems.find(t.getKey(), elem))
    return elem;

  elem = (utils::ConfigElem*) t.createInstance();
  if (!elem)
    return NULL;

  if (!elem->getTypeId().isDerivedFrom(utils::ConfigElem::getClassTypeId())) {
    delete elem;
    return NULL;
  }

  elem->ref();
  _unpack_elems.enter(t.getKey(), elem);

  return elem;
}

utils::ConfigElem*
IPTClientConfigSource::putData(utils::Type out_type, utils::Type in_type,
                               const char* in_value)
{
  utils::ConfigElem* elem = getElem(in_type);
  if (!elem) 
    elem = getElem(out_type);
  if (!elem) {
    printf("Invalid type '%s'\n", out_type.getName().getString());
    return NULL;
  }

  utils::Input input;
  input.setBuffer((void*) (char*) in_value, strlen(in_value));
  if (!elem->readData(input)) {
    printf("Problem parsing '%s' as '%s'\n", (char*) in_value,
           out_type.getName().getString());
    return NULL;
  }
  return elem;
}

#define QUERY_TIMEOUT 5.0

int IPTClientConfigSource::get(utils::Type out_type, const char* name,
                               void* data, int max_num,
                               void* default_data, int num_defaults)
{
  utils::ConfigElem* elem=NULL;

  IPMessage* repl = _communicator->Query(_server, CS_GET_MSG, &name,
                                         CS_GOTTEN_MSG, QUERY_TIMEOUT);
  if (repl) {
    ConfigReplyValue* cr = (ConfigReplyValue*) repl->FormattedData();
    if (cr->type < 0) {
      if (*cr->value)
        printf("IPTClient: Error from server when getting '%s': %s\n", name, 
               cr->value);
    } else {
      elem = putData(out_type, cr->type, cr->value);
    }
    delete repl;
  } else {
    elem = getElem(out_type);
    if (elem) {
      if (!elem->setValue(out_type, default_data, num_defaults))
        return 0;
    } 
  }

  if (elem)
    return elem->getValue(out_type, data, max_num);
  else 
    return 0;
}

int IPTClientConfigSource::numValues(const char* name)
{
  int num;
  if (_communicator->
      QueryFormatted(_server, CS_GET_NUM_VALUES_MSG, (void*) &name,
                     CS_NUM_VALUES_MSG, (void*) &num, QUERY_TIMEOUT))
    return num;
  else
    return 0;
}

bool IPTClientConfigSource::processEvents(double timeout)
{
  return _communicator->Idle(timeout);
}

bool IPTClientConfigSource::set(const char* name, const char* value)
{
  ConfigParseSet ps;
  ps.name = (char*) name;
  ps.value = (char*) value;
  int status;
  if (_communicator->QueryFormatted(_server, CS_PARSE_SET_MSG, &ps,
                                    CS_STATUS_MSG, &status,
                                    QUERY_TIMEOUT) && status > 0)
    return true;
  else
    return false;
}  

bool IPTClientConfigSource::set(utils::Type out_type, const char* name,
                                void* data, int num_values)
{
  utils::ConfigElem* elem = getElem(out_type);
  if (!elem)
    return false;
  if (!elem->setValue(out_type, data, num_values))
    return false;
  
  utils::Output output;
  char* value = new char[50];
  output.setBuffer(value, 50, utils::Output::standardResize);
  elem->writeData(output);
  output.write('\0');
  int nbytes;
  void* out_buf;
  output.getBuffer(out_buf, nbytes);
  value = (char*) out_buf;

  ConfigSet s;
  s.type = out_type.getKey();
  s.name = (char*) name;
  s.value = value;
  int status;
  bool res = (_communicator->QueryFormatted(_server, CS_SET_MSG, &s,
                                            CS_STATUS_MSG, &status,
                                            QUERY_TIMEOUT)
              && status > 0);

  delete [] value;
  return res;
}

bool IPTClientConfigSource::attach(utils::Type type, const char* name,
                                   void* data, int max_num,
                                   ConfigCallback notify, void* cbdata,
                                   void* default_data, int num_defaults)
{
  utils::ConfigElem* elem = NULL;
  ClientCallback* callback = NULL;
  IPMessage* repl = _communicator->Query(_server, CS_WATCH_MSG, &name,
                                         CS_WATCHED_MSG,
                                         QUERY_TIMEOUT);
  if (repl) {
    ConfigWatched* w = (ConfigWatched*) repl->FormattedData();
    //    printf("Watching handle %d\n", w->handle);
    if (w->handle > 0) {
      callback = new ClientCallback(name, this, type, data, max_num,
                                    notify, cbdata, w->handle);
      elem = putData(type, w->type, w->value);
    } else {
      fprintf(stderr,
              "IPTClientConfigSource: Could not attach to %s because '^%s'\n",
              name, w->value);
    }
  } else {
    if (!elem && default_data)
      elem = getElem(type);
    if (elem && !elem->setValue(type, default_data, num_defaults))
      elem = NULL;
  }

  if (elem && data)
    elem->getValue(type, data, max_num);

  if (elem != NULL && callback) {
    _callbacks.append(callback);
    return true;
  }
  return false;
}

void IPTClientConfigSource::handle_watch_set(IPMessage* msg)
{
  ConfigWatchSet* s = (ConfigWatchSet*) msg->FormattedData();

  utils::ListIterator<ClientCallback> iter(_callbacks);
  for (ClientCallback* cb = iter.first(); cb; cb = iter.next()) {
    if (cb->getHandle() == s->handle) {
      cb->notify(cb->getName(), s->type, s->value);
    }
  }
}

void IPTClientConfigSource::unwatch(int handle)
{
  // printf("Unwatching handle %d\n", handle);
  _communicator->SendMessage(_server, CS_UNWATCH_MSG, &handle);
}

void IPTClientConfigSource::reportState(ModuleState state)
{
  //  printf("Report state %p\n", _module_status);
  if (!_module_status)
    return;

  // get current time
  utils::Time now = TimeSource::now();
  long secs, usecs;
  now.getValue(secs, usecs);

#ifdef HAVE_PTHREAD_H
  // lock in case the blocking thread is reporting state
  pthread_mutex_lock(&_mem_mutex);
#endif  

  // propagate status if the state has changed or the message has changed
  bool propagate_status = false;
  const char* next_msg = "No message";
  if (_last_state != state) {
    // if next message has been set since the last state change use it
    if (_cur_status_msg != _next_status_msg) {
      next_msg = _next_status_msg.getString();
      _cur_status_msg = _next_status_msg;
    } else { // otherwise set the message from the state
      switch (state) {
      case NOT_RUNNING:
        next_msg = "Module not running";
        break;
      case INITIALIZING:
        next_msg = "Module initializing";
        break;
      case INITIALIZING_DISPLAY:
        next_msg = "Module initializing display";
        break;
      case INITIALIZED:
        next_msg = "Module initialized";
        break;
      case RUNNING:
        // Coming out of BLOCKED is a special case where we might simply
        // want to restore the old cached message
        // Empty cache message means nothing was ever set, so use the
        // default message - kind of a hack which eliminates the possiblity
        // of the user setting an empty message.
        if (_last_state == BLOCKED && _cached_status_msg.getLength())
          next_msg = _cached_status_msg.getString();
        else
          next_msg = "Module Running";
        break;
      case PAUSED:
        next_msg = "Module Paused";
        break;
      case BLOCKED:
        next_msg = "Module Blocked";
        break;
      case ERROR:
        next_msg = "Module Error";
        break;
      }
      _next_status_msg = next_msg;
    }
    _cur_status_msg = next_msg;
    propagate_status = true;
  } else if (_cur_status_msg != _next_status_msg) {
    _cur_status_msg = _next_status_msg;
    propagate_status = true;
  }

#ifdef HAVE_PTHREAD_H
  // start block test if state enters initialized, as there are some weird
  // bugs interacting with prioritized threads (see state_sense.cc) if
  // the block test thread is started earlier than this
  if (state == INITIALIZED) {
    _started_cycles = true;
    if (_do_block_test && !_block_test_running) {
      pthread_mutex_init(&_mem_mutex, NULL);
      _block_test_running = true;
      pthread_create(&_block_check_task, NULL, thread_entry, this);
    }
  } else
#endif
  if (state == RUNNING) {
    // and update average cycle time
    if (_num_cycles_in_average) {
      utils::Time last(_module_status_struct.last_run_secs,
                       _module_status_struct.last_run_usecs);
      double elapsed = now - last;
      float n = _num_cycles_in_average-1;
      _module_status_struct.avg_cycle_time =
        (n*_module_status_struct.avg_cycle_time + elapsed)/(n+1.0);
      if (_num_cycles_in_average < _max_cycles_in_average)
        _num_cycles_in_average++;
    } else
      _num_cycles_in_average = 1;
      
    // if running, update run times
    _module_status_struct.last_run_secs = secs;
    _module_status_struct.last_run_usecs = usecs;
  } else {
    // otherwise clear out any running related info, as long as not blocked
    // (BLOCKED is sort of a special "sub type" of RUNNING
    if (state != BLOCKED) {
      _module_status_struct.avg_cycle_time = 0;
      _module_status_struct.confidence = _module_status_struct.status = 0;
      _num_cycles_in_average = 0;
    }
  }

  // set up generic info
  _module_status_struct.state = (int) state;
  _module_status_struct.update_secs = secs;
  _module_status_struct.update_usecs = usecs;
  _module_status_struct.msg = _cur_status_msg.getString();

  // propagate state, if something important changed or we have excceeded 
  // minimum interval
  if (propagate_status ||
      (now - _last_propagation).getValue() > _minimum_status_interval) {
    _module_status->PutFormattedData(&_module_status_struct);
    _last_propagation = now;
  }
  _last_state = state;
#ifdef HAVE_PTHREAD_H
  pthread_mutex_unlock(&_mem_mutex);
#endif  
}

