#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <utils/ConfigFile.h>
#include <utils/StructElem.h>
#include <utils/SymbolTable.h>
#include <utils/Input.h>

#include <ConfigSource/ConfigSource.h>
#include <ConfigSource/Module.h>
#include <TimeSource/TimeSource.h>

// Convenience functions (can be overridden as well)

int ConfigSource::getInt(const char* name, int def)
{
  int data;
  if (get(utils::ConfigFile::intType(), name, &data, 1, &def) > 0)
    return data;
  else
    return def;
}

float ConfigSource::getFloat(const char* name, float def)
{
  float data;
  if (get(utils::ConfigFile::floatType(), name, &data, 1, &def) > 0)
    return data;
  else
    return def;
}

double ConfigSource::getDouble(const char* name, double def)
{
  double data;
  if (get(utils::ConfigFile::doubleType(), name, &data, 1, &def) > 0)
    return data;
  else
    return def;
}

char ConfigSource::getChar(const char* name, char def)
{
  char data;
  if (get(utils::ConfigFile::charType(), name, &data, 1, &def) > 0)
    return data;
  else
    return def;
}

bool ConfigSource::getBool(const char* name, bool def)
{
  bool data;
  if (get(utils::ConfigFile::boolType(), name, &data, 1, &def) > 0)
    return data;
  else
    return def;
}

const char* ConfigSource::getString(const char* name, const char* def)
{
  const char* data;
  if (get(utils::ConfigFile::stringType(), name, &data, 1, &def) > 0)
    return data;
  else
    return def;
}

int ConfigSource::getInts(const char* name, int* data, int max_num,
                          int* default_data, int num_defaults)
{
  return get(utils::ConfigFile::intType(), name, data, max_num,
             default_data, num_defaults);
}

int ConfigSource::getFloats(const char* name, float* data, int max_num,
                            float* default_data, int num_defaults)
{
  return get(utils::ConfigFile::floatType(), name, data, max_num,
             default_data, num_defaults);
}

int ConfigSource::getDoubles(const char* name, double* data, int max_num,
                             double* default_data, int num_defaults)
{
  return get(utils::ConfigFile::doubleType(), name, data, max_num,
             default_data, num_defaults);
}

int ConfigSource::getChars(const char* name, char* data, int max_num,
                           char* default_data, int num_defaults)
{
  return get(utils::ConfigFile::charType(), name, data, max_num,
             default_data, num_defaults);
}

int ConfigSource::getBools(const char* name, bool* data, int max_num,
                           bool* default_data, int num_defaults)
{
  return get(utils::ConfigFile::boolType(), name, data, max_num,
             default_data, num_defaults);
}

int ConfigSource::getStrings(const char* name, const char** data, int max_num,
                             const char** default_data, int num_defaults)
{
  return get(utils::ConfigFile::stringType(), name, data, max_num,
             default_data, num_defaults);
}

int ConfigSource::getStructs(const char* name, utils::ConfigElem** data,
                             int max_num)
{
  return get(utils::StructElem::getClassTypeId(), name, data, max_num);
}

utils::ConfigFile* ConfigSource::getSubFile(const char* name)
{
  utils::ConfigElem* elem;
  if (!getStructs(name, &elem)) 
    return NULL;

  return new utils::ConfigFile(elem);
}

bool ConfigSource::getStruct(const char* name, utils::ConfigFile& file)
{
  utils::ConfigElem* elem;
  if (!getStructs(name, &elem)) 
    return false;

  file.initRoot((utils::StructElem*) elem);
  return true;
}

bool ConfigSource::notify(const char* name, ConfigCallback cb, void* cb_data)
{
  return attach(utils::ConfigFile::boolType(), name, NULL, 0, cb,
                cb_data, NULL);
}

bool ConfigSource::attachInt(const char* name, int* data, int def,
                             ConfigCallback notify, void* cb_data)
{
  return attach(utils::ConfigFile::intType(), name, data, 1, notify, cb_data,
                &def);
}

bool ConfigSource::attachFloat(const char* name, float* data, float def,
                               ConfigCallback notify, void* cb_data)
{
  return attach(utils::ConfigFile::floatType(), name, data, 1,
                notify, cb_data, &def);
}

bool ConfigSource::attachDouble(const char* name, double* data, double def,
                                ConfigCallback notify, void* cb_data)
{
  return attach(utils::ConfigFile::doubleType(), name, data, 1, notify,
                cb_data, &def);
}

bool ConfigSource::attachChar(const char* name, char* data, char def, 
                              ConfigCallback notify, void* cb_data)
{
  return attach(utils::ConfigFile::charType(), name, data, 1,
                notify, cb_data, &def);
}

bool ConfigSource::attachBool(const char* name, bool* data, bool def,
                              ConfigCallback notify, void* cb_data)
{
  return attach(utils::ConfigFile::boolType(), name, data, 1,
                notify, cb_data, &def);
}

bool ConfigSource::attachString(const char* name, const char** data,
                                const char* def, ConfigCallback notify,
                                void* cb_data)
{
  return attach(utils::ConfigFile::stringType(), name, data, 1,
                notify, cb_data, &def);
}

bool ConfigSource::attachInts(const char* name, int* data, int max_num,
                              ConfigCallback notify, void* cb_data, 
                              int* default_data, int num_defaults)
{
  return attach(utils::ConfigFile::intType(), name, data, max_num,
                notify, cb_data, default_data, num_defaults);
}

bool ConfigSource::attachFloats(const char* name, float* data, int max_num,
                                ConfigCallback notify, void* cb_data, 
                                float* default_data, int num_defaults)
{
  return attach(utils::ConfigFile::floatType(), name, data, max_num,
                notify, cb_data, default_data, num_defaults);
}

bool ConfigSource::attachDoubles(const char* name, double* data, int max_num,
                                 ConfigCallback notify, void* cb_data, 
                                 double* default_data, int num_defaults)
{
  return attach(utils::ConfigFile::doubleType(), name, data, max_num,
                notify, cb_data, default_data, num_defaults);
}

bool ConfigSource::attachChars(const char* name, char* data, int max_num,
                               ConfigCallback notify, void* cb_data, 
                               char* default_data, int num_defaults)
{
  return attach(utils::ConfigFile::charType(), name, data, max_num,
                notify, cb_data, default_data, num_defaults);
}

bool ConfigSource::attachBools(const char* name, bool* data, int max_num,
                               ConfigCallback notify, void* cb_data, 
                               bool* default_data, int num_defaults)
{
  return attach(utils::ConfigFile::boolType(), name, data, max_num,
                notify, cb_data, default_data, num_defaults);
}

bool ConfigSource::attachStrings(const char* name, 
                                 const char** data, int max_num,
                                 ConfigCallback notify, void* cb_data, 
                                 const char** default_data, int num_defaults)
{
  return attach(utils::ConfigFile::stringType(), name, data, max_num,
                notify, cb_data, default_data, num_defaults);
}

bool ConfigSource::setInt(const char* name, int data) 
{
  return set(utils::ConfigFile::intType(), name, &data, 1);
}

bool ConfigSource::setFloat(const char* name, float data)
{
  return set(utils::ConfigFile::floatType(), name, &data, 1);
}

bool ConfigSource::setDouble(const char* name, double data)
{
  return set(utils::ConfigFile::doubleType(), name, &data, 1);
}

bool ConfigSource::setChar(const char* name, char data)
{
  return set(utils::ConfigFile::charType(), name, &data, 1);
}

bool ConfigSource::setString(const char* name, const char* data)
{
  return set(utils::ConfigFile::stringType(), name, &data, 1);
}

bool ConfigSource::setBool(const char* name, bool data)
{
  return set(utils::ConfigFile::boolType(), name, &data, 1);
}

bool ConfigSource::setInts(const char* name, int* data, int num_values)
{
  return set(utils::ConfigFile::intType(), name, &data, num_values);
}

bool ConfigSource::setFloats(const char* name, float* data, int num_values)
{
  return set(utils::ConfigFile::floatType(), name, &data, num_values);
}

bool ConfigSource::setDoubles(const char* name, double* data, int num_values)
{
  return set(utils::ConfigFile::doubleType(), name, &data, num_values);
}

bool ConfigSource::setChars(const char* name, char* data, int num_values)
{
  return set(utils::ConfigFile::charType(), name, &data, num_values);
}

bool ConfigSource::setStrings(const char* name, const char** data,
                              int num_values)
{
  return set(utils::ConfigFile::stringType(), name, &data, num_values);
}

bool ConfigSource::setBools(const char* name, bool* data, int num_values)
{
  return set(utils::ConfigFile::boolType(), name, &data, num_values);
}

bool ConfigSource::sleep(double timeout)
{
  timeout -= 0.002; // shave a little off for overhead
  utils::Time start = utils::Time::getRealTimeOfDay();
  bool res = false;
  while (timeout > 0) {
    bool cur_res = processEvents(timeout);
    res = res || cur_res;
    utils::Time now = utils::Time::getRealTimeOfDay();
    timeout -= (now-start).getValue();
    start = now;
  }

  return res;
}

extern "C" {

IF_ConfigSource CS_create(const char* spec, UTL_SymbolTable globals)
{
  return ConfigSource::create(spec, globals);
}

bool CS_set(IF_ConfigSource cs, const char* name, const char* value)
{
  return cs->set(name, value);
}

int CS_get(IF_ConfigSource cs, int type, const char* name,
           void* data, int max_num, void* default_data, int num_defaults)
{
  return cs->get(type, name, data, max_num, default_data, num_defaults);
}

int CS_getInt(IF_ConfigSource cs, const char* name, int def)
{
  return cs->getInt(name, def);
}

float CS_getFloat(IF_ConfigSource cs, const char* name, float def)
{
  return cs->getFloat(name, def);
}

double CS_getDouble(IF_ConfigSource cs, const char* name, double def)
{
  return cs->getDouble(name, def);
}

char CS_getChar(IF_ConfigSource cs, const char* name, char def)
{
  return cs->getChar(name, def);
}

const char* CS_getString(IF_ConfigSource cs, const char* name,const char* def)
{
  return cs->getString(name, def);
}

bool CS_getBool(IF_ConfigSource cs, const char* name, bool def)
{
  return cs->getBool(name, def);
}


int CS_getInts(IF_ConfigSource cs, const char* name, int* data, int max_num,
               int* default_data, int num_defaults)
{
  return cs->getInts(name, data, max_num, default_data, num_defaults);
}

int CS_getFloats(IF_ConfigSource cs, const char* name,
                 float* data, int max_num,
                 float* default_data, int num_defaults)
{
  return cs->getFloats(name, data, max_num, default_data, num_defaults);
}

int CS_getDoubles(IF_ConfigSource cs, const char* name,
                  double* data, int max_num,
                  double* default_data, int num_defaults)
{
  return cs->getDoubles(name, data, max_num, default_data, num_defaults);
}

int CS_getChars(IF_ConfigSource cs, const char* name, char* data, int max_num,
                char* default_data, int num_defaults)
{
  return cs->getChars(name, data, max_num, default_data, num_defaults);
}

int CS_getStrings(IF_ConfigSource cs, const char* name,
                  const char** data, int max_num,
                  const char** default_data, int num_defaults)
{
  return cs->getStrings(name, data, max_num, default_data, num_defaults);
}

int CS_getBools(IF_ConfigSource cs, const char* name, bool* data, int max_num,
                bool* default_data, int num_defaults)
{
  return cs->getBools(name, data, max_num, default_data, num_defaults);
}

bool CS_attachInt(IF_ConfigSource cs, const char* name, int* data, int def,
                  ConfigCallback notify, void* cb_data)
{
  return cs->attachInt(name, data, def, notify, cb_data);
}

bool CS_attachFloat(IF_ConfigSource cs, const char* name, float* data,
                    float def, ConfigCallback notify, void* cb_data)
{
  return cs->attachFloat(name, data, def, notify, cb_data);
}

bool CS_attachDouble(IF_ConfigSource cs, const char* name, double* data,
                     double def, ConfigCallback notify, void* cb_data)
{
  return cs->attachDouble(name, data, def, notify, cb_data);
}

bool CS_attachChar(IF_ConfigSource cs, const char* name, char* data, char def,
                   ConfigCallback notify, void* cb_data)
{
  return cs->attachChar(name, data, def, notify, cb_data);
}

bool CS_attachString(IF_ConfigSource cs, const char* name, const char** data,
                     const char* def, ConfigCallback notify, void* cb_data)
{
  return cs->attachString(name, data, def, notify, cb_data);
}

bool CS_attachBool(IF_ConfigSource cs, const char* name, bool* data, bool def,
                   ConfigCallback notify, void* cb_data)
{
  return cs->attachBool(name, data, def, notify, cb_data);
}


bool CS_attachInts(IF_ConfigSource cs, const char* name,
                   int* data, int max_num, ConfigCallback notify,
                   void* cb_data, int* default_data, int num_defaults)
{
  return cs->attachInts(name, data, max_num, notify, cb_data, 
                        default_data, num_defaults);
}

bool CS_attachFloats(IF_ConfigSource cs, const char* name, float* data,
                     int max_num, ConfigCallback notify, void* cb_data,
                     float* default_data, int num_defaults)
{
  return cs->attachFloats(name, data, max_num, notify, cb_data,
                          default_data, num_defaults);
}

bool CS_attachDoubles(IF_ConfigSource cs, const char* name, double* data,
                      int max_num, ConfigCallback notify, void* cb_data,
                      double* default_data, int num_defaults)
{
  return cs->attachDoubles(name, data, max_num, notify, cb_data,
                           default_data, num_defaults);
}

bool CS_attachChars(IF_ConfigSource cs, const char* name, char* data,
                    int max_num, ConfigCallback notify, void* cb_data,
                    char* default_data, int num_defaults)
{
  return cs->attachChars(name, data, max_num, notify, cb_data,
                         default_data, num_defaults);
}

bool CS_attachStrings(IF_ConfigSource cs, const char* name, const char** data,
                      int max_num, ConfigCallback notify, void* cb_data,
                      const char** default_data, int num_defaults)
{
  return cs->attachStrings(name, data, max_num, notify, cb_data,
                           default_data, num_defaults);
}

bool CS_attachBools(IF_ConfigSource cs, const char* name, bool* data,
                    int max_num, ConfigCallback notify, void* cb_data,
                    bool* default_data, int num_defaults)
{
  return cs->attachBools(name, data, max_num, notify, cb_data,
                         default_data, num_defaults);
}

}

Module::Module(const char* name, const char* spec_string)
{
  _cycle_number = 0;
  _name = name;
  _symbol_table = NULL;
  _config_src = NULL;
  _movie_buf = NULL;
  if (!spec_string)
    return;

  utils::Input::clearDirectories();
  const char* data_path = getenv("DATA_PATH");
  if (data_path) {  // customize file path for Input operations
    utils::Input::addEnvDirectoriesLast("DATA_PATH");
  } else {  // default path is .:$DATA_DIR:$LOCAL_CONFIG_DIR:$CONFIG_DIR:$HOME
    utils::Input::addDirectoryFirst(".");
    utils::Input::addEnvDirectoriesLast("DATA_DIR");
    utils::Input::addEnvDirectoriesLast("LOCAL_CONFIG_DIR");
    utils::Input::addEnvDirectoriesLast("CONFIG_DIR");
    utils::Input::addEnvDirectoriesLast("HOME");
  }
  utils::Input config_input;
  char s[300];
  const char* config_spec;
  if (config_input.openFile(spec_string, true)) {
    if (snprintf(s, 300, "active: name=%s;", config_input.getCurFileName())>0){
      config_spec = s;
      config_input.closeFile();
    } else {
      fprintf(stderr, "Module: filename '%s' too long\n", spec_string);
      config_input.closeFile();
      return;
    }
  } else
    config_spec = spec_string;

  _symbol_table = new utils::SymbolTable();
  _symbol_table->set("ModuleName", name);
  
  _config_src = ::ConfigSource::create(config_spec, _symbol_table);
  if (!_config_src) {
    delete _symbol_table;
    _symbol_table = NULL;
    fprintf(stderr, "Module: Invalid spec '%s'\n", config_spec);
    return;
  }
  _config_src->ref();
}

Module::~Module()
{
  utils::ListIterator<utils::Managed> iter(_intfs);
  for (utils::Managed* cur = iter.first(); cur; cur = iter.next()) {
    cur->unref();
  }

  if (_config_src) {
    _config_src->reportState(ConfigSource::NOT_RUNNING);
    _config_src->unref();
  }
  delete _symbol_table;
  _config_src = NULL;
  _symbol_table = NULL;
  delete [] _movie_buf;
}

void Module::mainLoop()
{
  if (!startup())
    return;
  
  while (cycle())
    ;
}

bool Module::startup()
{
  if (!_config_src)
    return false;
  
  _config_src->attachBool("running", &_running, true);
  _config_src->attachBool("paused", &_paused, false);
  _display = _config_src->getBool("display", false);
  _cycle_time = _config_src->getFloat("cycle_time", 0);
  _idle_time = _config_src->getFloat("idle_time", 0.1);
  _display_interval = _config_src->getFloat("display_interval", 0.0333);

  const char* spec = _config_src->getString("time_source_spec");
  if (*spec) {
    TimeSource* source = TimeSource::create(spec, _symbol_table);
    if (source)
      TimeSource::setDefaultTimeSource(source);
  }

  _config_src->reportState(ConfigSource::INITIALIZING);
  if (!initialize(_config_src, _symbol_table))
    return false;

  if (_display) {
    _config_src->reportState(ConfigSource::INITIALIZING_DISPLAY);
    _display = initializeDisplay(_config_src);
  }
  _config_src->reportState(ConfigSource::INITIALIZED);

  _timing = _config_src->getBool("timing", true);
  if (_timing) {
    _timing_periods = _config_src->getInt("timing_periods", 100);
    _elapsed = 0;
  }
  _timing_details = _config_src->getBool("timing_details", false);
  if (_timing_details) {
    _n_run = _n_display = 0;
    _elapsed_run = _elapsed_display = 0;
  }
    
  utils::ConfigFile movie_params;
  _making_movie = false;
  _movie_buf = NULL;
  if (_config_src->getStruct("movie_params", movie_params)) {
    _making_movie = movie_params.getBool("enable", true);
    if (!_making_movie)
      return true;
    _movie_step = movie_params.getFloat("step", 0.033);
    _movie_script = movie_params.getString("script", "");
    if (_movie_script.getLength() == 0) {
      _making_movie = false;
      return true;
    }

    _movie_time_key = TimeSource::getSettingKey();
    if (_movie_time_key < 0) {
      fprintf(stderr, "Module: Cannot acquire time key for movie making\n");
      return true;
    }
    
    _movie_buf = new char[_movie_script.getLength() + 20];
  }

  return true;
}

bool Module::cycle()
{
  if (_paused) {
    printf("Pausing module\n");
  }
  while (_paused && _running) {
    _config_src->reportState(ConfigSource::PAUSED);
    _config_src->processEvents(_idle_time);
    if (_display) {
      if (!displayIdle())
        return false;
    }
  }
  if (!_running) 
    return false;

  utils::Time pre_run;
  if (_timing_details)
    pre_run = utils::Time::getRealTimeOfDay();
  if (!run())
    return false;

  utils::Time post_run;
  if (_timing_details) {
    post_run = utils::Time::getRealTimeOfDay();
    _elapsed_run += (post_run - pre_run).getValue();
    _n_run++;
    if (!(_n_run % _timing_periods)) {
      printf("Average run time for %d runs is %f\n", _timing_periods, 
             _elapsed_run/_timing_periods);
      _n_run = 0;
      _elapsed_run = 0;
    }
  }

  if (_display) {
    if (_display_interval > 0) {
      // note: use raw time of day because we are interested in real
      // elapsed time for this.
      utils::Time t = utils::Time::getRealTimeOfDay();
      if ((t - _last_display).getValue() > _display_interval) {
        if (!display())
          return false;
        _last_display = t;
      } else
        if (!displayIdle())
          return false;
    } else
      if (!display())
        return false;

    if (_timing_details) {
      _elapsed_display +=
        (utils::Time::getRealTimeOfDay() - pre_run).getValue();
      _n_display++;
      if (!(_n_display % _timing_periods)) {
        printf("Average display time for %d displays is %f\n",
               _timing_periods, _elapsed_display/_timing_periods);
        _n_display = 0;
        _elapsed_display = 0;
      }
    }
  }

  _config_src->reportState(ConfigSource::RUNNING);
  if (_timing) {
    utils::Time now = utils::Time::getRealTimeOfDay();
    if (_cycle_number) {
      _elapsed += (now-_start).getValue();
      if (!(_cycle_number % _timing_periods)) {
        printf("Average cycle time for %d cycles is %f\n",
               _timing_periods,
               _elapsed/_timing_periods);
        _elapsed = 0;
      }
    }

    _start = now;
  }

  _cycle_number++;

  if (_making_movie) {
    sprintf(_movie_buf, _movie_script.getString(), _cycle_number);
    printf("Saving: %s\n", _movie_buf);
    // some day when my software rules the world, this could be a security
    // problem, as it might mean a smart person could configure their
    // system to rm -rf the world
    system(_movie_buf);
    printf("Incrementing %f\n", _movie_step);
    TimeSource::setCurTime(TimeSource::now() + utils::Time(_movie_step),
                           _movie_time_key);
  }

  _config_src->processEvents(0.0);

  // if we are not making a movie and have a non-zero cycle time
  if (!_making_movie && _cycle_time) {
    // if we have not taken up too much time
    utils::Time now = TimeSource::now();
    double elapsed;
    if (_last_cycle.isZero())
      elapsed = 0;
    else
      elapsed = (now - _last_cycle).getValue();      
    if (elapsed < _cycle_time) {
      // try and advance time by the proper amount to have the right cycle time
      TimeSource* source = TimeSource::getDefaultTimeSource();
      // this should sleep for a real clock, and bump the clock forward
      // for a fake clock
      if (!source || !source->advanceTime(_cycle_time - elapsed))
        utils::Time::sleep(_cycle_time-elapsed);
    }
    _last_cycle = TimeSource::now();
  }
  return true;
}
  
bool Module::isDisplaying() const { return _display; }
bool Module::isRunning() const { return _running; }
bool Module::isPaused() const { return _paused; }

void Module::unrefIntf(utils::Managed* intf)
{
  if (intf) {
    _intfs.remove(intf);
    intf->unref();
  }
}
  
