#include <stdio.h>

#include <utils/Time.h>
#include <utils/String.h>
#include <ConfigSource/Module.h>
#include <ConfigSource/ConfigSource.h>

#include <VehPoseSource/VehPoseSource.h>
#include <VehPoseDest/VehPoseDest.h>

class VehPoseRelayModule : public Module
{
public:
  VehPoseRelayModule(const char* spec) : Module("VehPoseRelay", spec) {}
  virtual ~VehPoseRelayModule() {}
  
  virtual bool initialize(ConfigSource* config, utils::SymbolTable* table) {
    const char* spec;
    spec = config->getString("veh_pose_source_spec", "default");
    _veh_pose_source = create<VehPoseSource>(spec);
    if (!_veh_pose_source) {
      printf("Invalid veh pose in specification '%s'\n", spec);
      return false;
    }
  
    spec = config->getString("veh_pose_dest_spec");
    _veh_pose_dest = create<VehPoseDest>(spec);
    if (!_veh_pose_dest) {
      printf("Invalid veh pose output specification '%s', ignoring\n");
    }

    _exit_on_no_data = config->getBool("exit_on_no_data", false);
    _print_data = config->getBool("print_data", false);
    _sleep_time = config->getFloat("sleep_time", 0.0);

    return true;
  }

  virtual bool run() {
    utils::Time t;
    VehPose pose;
    if (_veh_pose_source->getCurPose(t, pose, true)) {
      if (_veh_pose_dest)
        _veh_pose_dest->outputPose(t, pose);
      if (_print_data) {
        printf("%f: Got %f %f\n", (double) t, pose.pos.x, pose.pos.y);
      }
    } else {
      if (_exit_on_no_data)
        return false;
      printf("Error getting veh_pose\n");
      return true;
    }

    if (_sleep_time)
      getConfigSource()->sleep(_sleep_time);

    return true;
  }

private:
  VehPoseSource* _veh_pose_source;   ///< the veh_pose input
  VehPoseDest* _veh_pose_dest;       ///< the veh_pose output
  bool _exit_on_no_data;      ///< true if an error in input is fatal
  bool _print_data;           ///< true if we output some debugging info
  float _sleep_time;          ///< if non-zero, sleep this many seconds

};

MODULE_MAIN(veh_pose_relay, VehPoseRelayModule);
