/** \file VehPoseSource.h
    API header file for for the VehPoseSource reconfigurable interface 
    \ref VehPoseGroup "example".
*/

#ifndef VEH_POSE_SOURCE_INTF_H
#define VEH_POSE_SOURCE_INTF_H

#include <utils/Time.h>
#include <utils/Interface.h>
#include <VehPoseDest/VehPose.h>

/** \class VehPoseSource

    The VehPoseSource interface example.

    An abstract interface to 3D road information given by an STL vector of 3D 
    points

    \ingroup VehPoseGroup
 */
UTILS_INTERFACE(VehPoseSource) {
 public:
  /** Empty virtual destructor for VehPoseSource.
      For C++ syntax reasons, it is a good idea to explicitly define a virtual
      destructor, even if empty like this one, to any abstract class like this.
   */
  virtual ~VehPoseSource() {}

  /** Get the vehicle pose at a given time

      Returns the vehicle pose at the given time.  The implementation will
      attempt to interpolate or extrapolate from known poses to get the pose at
      the desired time.

      \param[in] time The time for which to get the vehicle pose.
      \param[out] pose The resulting vehicle pose
      \return \p true if the routine was able to obtain a vehicle pose at
              the desired time, \p false if not.
   */
  virtual bool getPose(utils::Time time, VehPose& pose) = 0;

  /** Get the current vehicle pose

      Returns the latest vehicle pose along with the time at which it was
      collected.  The call can be blocking, i.e., we block until the next
      unread vehicle pose is acquired, but it is usually polling, i.e., it
      returns immediately with the most recent vehicle pose

      \param[out] time The time at which the road information was collected. (0
                        if the road is invalid)
      \param[out] pose The vehicle pose
      \param[in] blocking  If \p true, then the call is blocking, if \p false,
                           it is polling.
      \return \p true if "new" data has been acquired successfully, \p false
              on an invalid read or if no new road data has been acquired
              since the last read.  Note an invalid read can be verified by
              checking to see if \p time.isZero() is true.
  */
  virtual bool getCurPose(utils::Time& time,
                          VehPose& pose, bool blocking = false);

  
  /** Convenience routine for interpolating between and extrapolating sensor
      poses. 

      Takes prev_pose and next_pose, and then interpolation factor t, where t=0
      means prev_pose and t=1 means next_pose, and puts interpolated result in
      sensor_pose.  If t > 1, then we do an extrapolation.

      \param[in] prev_pose The beginning pose
      \param[in] next_pose The end pose
      \param[in] t The distance between the beginning and end pose to
                   interpolate, with 0 being the beginning and 1 being the end
      \param[out] sensor_pose The interpolated pose
  */
  static void interpolate(const VehPose& prev_pose, 
                          const VehPose& next_pose, double t,
                          VehPose& sensor_pose);

  // declare the standard interface static methods
  UTILS_INTF_DECL(VehPoseSource);
};
  
#endif
