/** \file
    Implements the FakeVehPoseSource interface instance
    \ingroup VehPoseGroup
 */
#include <iostream>

#include <utils/ConfigFile.h>
#include <TimeSource/TimeSource.h>

#include "VehPoseSource.h"

/** Fake interface definition for VehPoseSource.
    Does a simple 2D motion simulation to fake a vehicle pose source
*/
class FakeVehPoseSource : public VehPoseSource {
 public:
  /// \copydoc VehPoseSource::getPose
  virtual bool getPose(utils::Time time, VehPose& pose);

  /// Initialization routine
  bool init(utils::ConfigFile& params);

 private:
  double _radius;
  double _speed;
  utils::Time _start_time;
};

/// The required creation function for the "fake" tag
VehPoseSource* create_VehPoseSource_fake(VehPoseSourceGenerator*,
                                   utils::ConfigFile* params,
                                   utils::SymbolTable* globals)
{
  FakeVehPoseSource* intf = new FakeVehPoseSource();
  if (!intf->init(*params)) {
    delete intf;
    return NULL;
  }
  return intf;
}

bool FakeVehPoseSource::init(utils::ConfigFile& params)
{
  _radius = params.getDouble("radius", 1000);
  _speed = params.getDouble("speed", 1);
  _start_time = TimeSource::now();
  
  return true;
}

bool FakeVehPoseSource::getPose(utils::Time time, VehPose& pose)
{
  // do simple calculation of vehicle position
  utils::Time elapsed = time - _start_time;
  double theta = fmod(_speed/_radius*elapsed.getValue(), 2.0*M_PI);
  pose.pos.x = sin(theta)*_radius;
  pose.pos.y = _radius - cos(theta)*_radius;
  pose.pos.z = 0;
  pose.ori = utils::Rotation(utils::Vec3d(0, 0, 1), theta);

  return true;
}

