#include <stdio.h>

#include <utils/Time.h>
#include <utils/String.h>
#include <ConfigSource/Module.h>
#include <ConfigSource/ConfigSource.h>

#include <RoadSource/RoadSource.h>
#include <RoadDest/RoadDest.h>

class RoadRelayModule : public Module
{
public:
  RoadRelayModule(const char* spec) : Module("RoadRelay", spec) {}
  virtual ~RoadRelayModule() {}
  
  virtual bool initialize(ConfigSource* config, utils::SymbolTable* table) {
    const char* spec;
    spec = config->getString("road_source_spec", "default");
    _road_source = create<RoadSource>(spec);
    if (!_road_source) {
      printf("Invalid road in specification '%s'\n", spec);
      return false;
    }
  
    spec = config->getString("road_dest_spec");
    _road_dest = create<RoadDest>(spec);
    if (!_road_dest) {
      printf("Invalid road output specification '%s', ignoring\n");
    }

    _exit_on_no_data = config->getBool("exit_on_no_data", false);
    _print_data = config->getBool("print_data", false);
    _sleep_time = config->getFloat("sleep_time", 0.0);

    return true;
  }

  virtual bool run() {
    utils::Time t;
    if (_road_source->getPoints(t, _points)) {
      if (_road_dest)
        _road_dest->outputPoints(t, _points);
      if (_print_data) {
        printf("%f: Got %d road points\n", (double) t, _points.size());
      }
    } else {
      if (_exit_on_no_data)
        return false;
      printf("Error getting road\n");
      return true;
    }

    if (_sleep_time)
      getConfigSource()->sleep(_sleep_time);

    return true;
  }

private:
  RoadSource* _road_source;   ///< the road input
  RoadDest* _road_dest;       ///< the road output
  bool _exit_on_no_data;      ///< true if an error in input is fatal
  bool _print_data;           ///< true if we output some debugging info
  std::vector<utils::Vec3d> _points;  ///< point cache
  float _sleep_time;          ///< if non-zero, sleep this many seconds

};

MODULE_MAIN(road_relay, RoadRelayModule);
