/** \file
    Defines the RoadPlayer class
    \ingroup RoadGroup
*/
#ifndef ROAD_PLAYER_H
#define ROAD_PLAYER_H

#include <vector>
#include <utils/Player.h>
#include <TimeSource/TimeSource.h>
#include <RoadDest/RoadStructs.h>

namespace utils {
  class ConfigFile;
  class Vec3d;
  class SymbolTable;
};

/** Standalone class for playing road information.
    This class lets you open a structured canned data file and get data
    from it

    \ingroup RoadGroup
*/
class RoadPlayer {
 public:
  /** Open a file for playing.
      \param[in] params Configuration file including the file name and other 
                        standard TimeSource::PlayerManager parameters
      \param[in] globals The global symbol table                        
      \return \p true on success, \p false on failure
  */
  bool open(utils::ConfigFile& params, utils::SymbolTable* globals);

  /** Advance to the next set of road points
      \return \p true for success, \p false for failure
  */
  bool advance();

  /** Get the current set of road points tagged with a time.
      \param[out] time Time the road points were detected
      \param[out] points Detected 3D road points
      \return \p true for success, \p false for failure
  */
  bool getPoints(utils::Time& time, std::vector< utils::Vec3d > & points);

  /** Get the next set of road points tagged with a time.
      This could have a side effect of advancing the file pointers
      \param[out] time Time the road points were detected
      \param[out] points Detected 3D road points
      \param[in] blocking If \p false we will poll the manager to see if we
                          need to advance
      \return \p true for success, \p false for failure
  */
  bool nextPoints(utils::Time& time, std::vector< utils::Vec3d > & points,
                  bool blocking=true);

  /** Provides access to the underlying canned data manager. */
  TimeSource::PlayerManager* getManager() { return &_mgr; }

 private:
  TimeSource::PlayerManager _mgr;  ///< the player manager 
  utils::Player* _player;          ///< the actual player
  utils::PlayElem* _play_elem;     ///< bookkeeping element for player

  RoadDataStruct _input_area; ///< Packaged road information
  utils::Time _play_time;     ///< Timestamp for last read road information
};

#endif

