/** \file
    Implements the RoadPlayer class
    \ingroup RoadGroup
 */
#include <utils/Linear.h>

#include "RoadPlayer.h"

bool RoadPlayer::open(utils::ConfigFile& params,
                      utils::SymbolTable* globals)
{
  // setup the player manager
  // the player manager takes care of all of the interactions between the
  // data file and the time source
  if (!_mgr.open("Road.rad", params, globals))
    return false;

  // get the actual canned data reader from the manager
  _player = _mgr.getPlayer();

  // clear the input area
  memset(&_input_area, 0, sizeof(_input_area));

  // check versions
  int major_version = _mgr.getPlayer()->getHeader().
    getInt("int DataFormat.version_major", 1);
  int minor_version = _mgr.getPlayer()->getHeader().
    getInt("int DataFormat.version_minor", 1);
  // note, you don't have to simply reject other version, you can
  // try and adapt here
  if (major_version != 1 && minor_version != 0) {
    printf("RoadPlayer::init:  Cannot read version %d.%d\n",
           major_version, minor_version);
    return false;
  }

  // Tell the player to expect the points.  We store a reference in _play_elem
  // so we can have the player properly manage the memory associated with the
  // points.
  _play_elem = _player->expect("points", ROAD_DATA_FMT, &_input_area);

  // Ready the player for action
  return _player->setup();
}

bool RoadPlayer::advance()
{
  // advance the file pointer, if necessary, 
  // and cache the last read time for later
  // the player manager takes care of all the necessary interactions with
  // time, i.e., does reading advance time, or do we observe time to see
  // where to read.
  return _mgr.next(_play_time);
}

bool RoadPlayer::getPoints(utils::Time& time,
                           std::vector<utils::Vec3d> & points)
{
  // transfer input area points
  points.clear();
  for (int i=0;i<_input_area.num_points;i++) {
    RoadDataPoint& pt = _input_area.points[i];
    points.push_back(utils::Vec3d(pt.x, pt.y, pt.z));
  }

  // release point memory back to the player
  _player->release(_play_elem);

  // and set time from cached value
  time = _play_time;

  return true;
}

bool RoadPlayer::nextPoints(utils::Time& time,
                            std::vector<utils::Vec3d> & points,
                            bool blocking)
{
  if (blocking || (!blocking && _mgr.poll())) {
    if (!advance())
      return false;
  }
  return getPoints(time, points);
}
  
  
