/** \file
    Implements the OutputRoadSource interface instance
    \ingroup RoadGroup
*/
#include <iostream>

#include <utils/ConfigFile.h>
#include <utils/Generator.h>
#include <TimeSource/TimeSource.h>
#include <RoadDest/RoadDest.h>

#include "RoadSource.h"

/** Container interface definition for RoadSource that transparently redirects
    to a RoadDest.
*/
class OutputRoadSource : public RoadSource {
 public:
  OutputRoadSource();
  virtual ~OutputRoadSource();

  /// \copydoc RoadSource::getPoints
  virtual bool getPoints(utils::Time& time,
                         std::vector<utils::Vec3d>& points,
                         bool blocking = true);

  /// Initialization routine
  bool init(utils::ConfigFile& params,
            RoadSourceGenerator* gen, utils::SymbolTable* globals);

 private:
  RoadSource* _contained;
  RoadDest* _output;
};

/// The required creation function for the "output" tag
RoadSource* create_RoadSource_out(RoadSourceGenerator* gen,
                                  utils::ConfigFile* params,
                                  utils::SymbolTable* globals)
{
  OutputRoadSource* intf = new OutputRoadSource();
  if (!intf->init(*params, gen, globals)) {
    delete intf;
    return NULL;
  }
  return intf;
}

/// Additional creation function for the "logger" tag which will create
/// a logger output.
RoadSource* create_RoadSource_logger(RoadSourceGenerator* gen,
                                     utils::ConfigFile* params,
                                     utils::SymbolTable* globals)
{
  // pose output specification inherited from params for the most part
  utils::ConfigFile output_params;
  utils::ConfigFile::copy(*params, output_params);
  output_params.setString("tag", "logger");  // set up logger tag
  output_params.set("contained", "{}"); //we clear contained to avoid confusion
  
  // get the contained specification
  utils::ConfigFile contained_params;
  params->getStruct("contained", contained_params);

  // now create the output instance parameters
  utils::ConfigFile final_params;
  final_params.setString("tag", "output");
  final_params.setStruct("contained", contained_params);
  final_params.setStruct("output", output_params);

  OutputRoadSource* intf = new OutputRoadSource();
  if (!intf->init(final_params, gen, globals)) {
    delete intf;
    return NULL;
  }
  return intf;
}

/// Additional creation function for the "logger" tag which will create
/// a logger output.
RoadSource* create_RoadSource_shmemPublish(RoadSourceGenerator* gen,
                                           utils::ConfigFile* params,
                                           utils::SymbolTable* globals)
{
  // pose output specification inherited from params for the most part
  utils::ConfigFile output_params;
  utils::ConfigFile::copy(*params, output_params);
  output_params.setString("tag", "shmem");  // set up shmem tag
  output_params.set("contained", "{}"); //we clear contained to avoid confusion
  
  // get the contained specification
  utils::ConfigFile contained_params;
  params->getStruct("contained", contained_params);

  // now create the output instance parameters
  utils::ConfigFile final_params;
  final_params.setString("tag", "output");
  final_params.setStruct("contained", contained_params);
  final_params.setStruct("output", output_params);

  OutputRoadSource* intf = new OutputRoadSource();
  if (!intf->init(final_params, gen, globals)) {
    delete intf;
    return NULL;
  }
  return intf;
}

OutputRoadSource::OutputRoadSource()
{
  _contained = NULL;
  _output = NULL;
}

OutputRoadSource::~OutputRoadSource()
{
  if (_contained)
    _contained->unref();
  if (_output)
    _output->unref();
}

bool OutputRoadSource::init(utils::ConfigFile& params,
                            RoadSourceGenerator* gen,
                            utils::SymbolTable* globals)
{
  // create the contained interface
  _contained = gen->interface(params.getString("contained"), globals);
  if (!_contained) {
    cerr << "OutputRoadSource::init: could not create contained\n";
    return false;
  }
  _contained->ref();

  _output = RoadDest::generate(params.getString("output", "default"), globals);
  if (!_output) {
    cerr << "OutputRoadSource::init: could not create output\n";
    return false;
  }
  _output->ref();
  
  return true;
}

bool OutputRoadSource::getPoints(utils::Time& time,
                               std::vector<utils::Vec3d>& points,
                               bool blocking)
{
  if (_contained->getPoints(time, points, blocking)) {
    _output->outputPoints(time, points);
    return true;
  } 
  return false;
}
