#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <stdlib.h>
#include <pthread.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/user.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <netinet/in.h>

#if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
/* union semun is defined by including <sys/sem.h> */
#else
/* according to X/OPEN we have to define it ourselves */
union semun {
  int val;                    /* value for SETVAL */
  struct semid_ds *buf;       /* buffer for IPC_STAT, IPC_SET */
  unsigned short int *array;  /* array for GETALL, SETALL */
  struct seminfo *__buf;      /* buffer for IPC_INFO */
};
#endif

#include <utils/ConfigFile.h>
#include <utils/StructElem.h>
#include <utils/Vector.h>
#include <utils/StringDict.h>
#include <utils/String.h>
#include <utils/Time.h>
#include <utils/formatting/Format.h>
#include <utils/formatting/FormatParser.h>
#include <utils/formatting/FormatEqualsAction.h>
#include <utils/formatting/PackAction.h>
#include <utils/formatting/UnpackAction.h>
#include <utils/formatting/DeleteAction.h>

#include <ipt/ipt.h>
#include <ipt/connection.h>
#include <ipt/callbacks.h>
#include <ipt/primcomm.h>
#include <ipt/message.h>
#include <ipt/posixmem.h>

#include "internal_messages.h"

class IPManagerSharedMemory : public IPClientSharedMemory {
public:
  IPManagerSharedMemory(IPCommunicator* com, const char* fmt, int max_size)
    : IPClientSharedMemory(IPC_PRIVATE, IPC_PRIVATE, 1.0, com, fmt, max_size)
  {
    if (_shm)
      bzero(_shm, MaxSize());
    semctl(_sem_id, 2, SETVAL, 1); // set up mutual exclusion semaphore
  }
  virtual ~IPManagerSharedMemory() {
    if (shmID() != -1) {
      /* detach and clear shared memory */
      if (_shm && (-1 == shmdt ((void *)_shm))) 
        perror ("shmdt");
      if (-1 == shmctl (shmID(), IPC_RMID, NULL)) 
        perror ("shmctl");
    }
    if (semID() != -1) {
      union semun un;
      un.val = 0;
      /* clear semaphore */
      if (-1 == semctl (semID(), 1, IPC_RMID, un)) 
        perror ("semctl");
    }
    setIDs(-1,-1);
  }
  virtual int PutData(int size) {
    if (size != 0)
      return 0;
    if (_sem_id == -1 || _shm_id == -1 || size > MaxSize())
      return 0;

    static sembuf op_lock[3] = {
      { 0, 0, 0 },   // wait for clients to say clear to write
      { 1, 1, SEM_UNDO },   // indicate we are writing
      { 2, -1, SEM_UNDO },  // grab condition mutex
    };
    static sembuf op_unlock[1] = {
      { 1, -1, SEM_UNDO},  // finished writing
    };
    static sembuf op_release_mutex[1] = {
      { 2, 1, SEM_UNDO }
    };
    static sembuf op_signal[2] = {
      { 2, 1, SEM_UNDO },
      { 3, -1, 0 }
    };

    while (1) {
      if (semop(_sem_id, op_lock, 3) == -1) {
        perror("owner lock semop");
        return 0;
      } else
        break;
    }

    *((int*)_shm) = -1;
    *(((int*)_shm)+1) = 0;

    if (semop(_sem_id, op_unlock, 1) == -1) {
      perror("owner unlock semop");
      return 0;
    }

    int val = semctl(_sem_id, 3, GETVAL);
    if (val < 0) {
      perror("get val");
      return 0;
    }
    if (val > 0) {
      if (semop(_sem_id, op_signal, 2) == -1) {
        perror("signal");
        return 0;
      }
    } else {
      if (semop(_sem_id, op_release_mutex, 1) == -1) {
        perror("release mutex");
        return 0;
      }
    }

    return 1;
  }
};

// gives a safety margin of real max size of 65000
#define UDP_MAXSIZE 64000
#define UDP_BUFFER_MAX 65000

static void rthandle(int)
{
}

struct ShmemSpec {
  IPString name;
  int sem_id, shm_id;
  IPSharedMemory* shmem;
  IPConnection* owner;
  IPConnection* remote;
  int last_tag;
  int sequence_num;
  bool use_tcp;
};

struct RemoteClientSpec {
  ShmemSpec* shmem;
  IPConnection* connection;
  int socket_fd;
  struct sockaddr_in addr;
  int last_tag;
  IPConnection* tcp_dest;
};

class MemPool {
public:
  MemPool(IPCommunicator* comm, int udp_port, int sig_num);
  ~MemPool();

  int hasNewData() const;
  void remove(const char* name);
  int signalNum() const { return _sig_num; }

  void monitorMemory();   // to be used in a sub thread

private:
  void cleanup();
  bool output_packet(RemoteClientSpec&);
  void mem_activity(IPConnection*);
  void sock_activity(IPConnection*);
  void owned_init_handler(IPMessage*);
  void client_init_handler(IPMessage*);
  void close_handler(IPMessage*);
  void remote_client_handler(IPMessage*);
  void remote_client_canceler(IPMessage*);
  void message_handler(IPMessage*);
  ShmemSpec* make_shmem_spec(const char* name, bool use_tcp, const char* fmt,
                             IPPosixSharedMemory* shm,
                             IPConnection* owner, IPConnection* remote);
  void scan_owners(IPTimer*);

  void process_data(int seqnum,const char* name, unsigned char* data, int len);

private:
  IPCommunicator* _comm;
  IPConnection* _memconn;
  IPConnection* _sockconn;
  int _serve_sock;
  int _serve_port;
  IPStringDict<ShmemSpec*> _pool;
  IPVector<RemoteClientSpec> _client_list;
  IPVector<ShmemSpec*> _owned_list;
  int _flag_sem_id;
  unsigned char* _output;
  int _out_size;
  int _sequence_num;
  pid_t _pid;
  int _sig_num;
  unsigned char _input[UDP_BUFFER_MAX];
};

declareConnectionCallback(MemPool);
implementConnectionCallback(MemPool);
declareHandlerCallback(MemPool);
implementHandlerCallback(MemPool);
declareTimerCallback(MemPool);
implementTimerCallback(MemPool);

class MemConnection : public IPConnection {
public:
  MemConnection(const char* name, IPCommunicator* comm, MemPool* pool)
    : IPConnection("Mem", name, "Unknown", comm, -1) {
    _pool = pool;
  }

  virtual int Active() { return true; }
  virtual int Send(IPMessage*) { return 0; }
  virtual IPMessage* Receive() { return NULL; }
  virtual int DataAvailable() { return _pool->hasNewData(); }

private:
  MemPool* _pool;
};

MemPool::MemPool(IPCommunicator* comm, int udp_port, int sig_num) :
  _client_list(10), _owned_list(10)
{
  _comm = comm;
  _sequence_num = 0;
  _pid = getpid();
  _sig_num = sig_num;

  _memconn = new MemConnection("MemChecker", comm, this);
  comm->AddAdministrativeConnection(_memconn, 
                                    new ConnectionCallback(MemPool)
                                    (this, &MemPool::mem_activity));

  _serve_sock = socket(AF_INET, SOCK_DGRAM, 0);
  struct sockaddr_in addr;
  _serve_port = udp_port;
  addr.sin_port = htons(udp_port);
  addr.sin_family = AF_INET;
  if (!lookup_machine_name(comm->ThisHost(), &addr.sin_addr.s_addr)) {
    printf("Problem lookup own host name '%s'\n", comm->ThisHost());
    addr.sin_addr.s_addr = INADDR_ANY;
  }

  if (bind(_serve_sock,(struct sockaddr*)&addr, sizeof(struct sockaddr_in))<0){
    perror("udp bind");
    cleanup();
    return;
  }

  _sockconn = comm->
    AddAdministrativeConnection(_serve_sock,
                                new ConnectionCallback(MemPool)
                                (this, &MemPool::sock_activity));

  

  comm->RegisterMessage(IPTShmRemoteClientInitMsgNum,
                        IPT_SHM_REMOTE_CLIENT_INIT_MSG,
                        IPT_SHM_REMOTE_CLIENT_INIT_FORM);
  comm->RegisterMessage(IPTShmRemoteClientRespMsgNum,
                        IPT_SHM_REMOTE_CLIENT_RESP_MSG,
                        IPT_SHM_REMOTE_CLIENT_RESP_FORM);
  comm->RegisterMessage(IPTShmRemoteClientCancelMsgNum,
                        IPT_SHM_REMOTE_CLIENT_CANCEL_MSG,
                        IPT_SHM_REMOTE_CLIENT_CANCEL_FORM);
  comm->RegisterMessage(IPTShmMessageMsgNum,
                        IPT_SHM_MESSAGE_MSG, IPT_SHM_MESSAGE_FORM);

  comm->RegisterHandler(IPT_SHM_OWNED_MEMORY_INIT_MSG,
                        new HandlerCallback(MemPool)
                        (this,&MemPool::owned_init_handler));
  comm->RegisterHandler(IPT_SHM_CLIENT_MEMORY_INIT_MSG,
                        new HandlerCallback(MemPool)
                        (this,&MemPool::client_init_handler));
  comm->RegisterHandler(IPT_SHM_MEMORY_CLOSE_MSG,
                        new HandlerCallback(MemPool)
                        (this,&MemPool::close_handler));
  comm->RegisterHandler(IPT_SHM_REMOTE_CLIENT_INIT_MSG,
                        new HandlerCallback(MemPool)
                        (this,&MemPool::remote_client_handler));
  comm->RegisterHandler(IPT_SHM_REMOTE_CLIENT_CANCEL_MSG,
                        new HandlerCallback(MemPool)
                        (this,&MemPool::remote_client_canceler));
  comm->RegisterHandler(IPT_SHM_MESSAGE_MSG,
                        new HandlerCallback(MemPool)
                        (this,&MemPool::message_handler));

  comm->AddTimer(1.0,
                 new TimerCallback(MemPool)(this,&MemPool::scan_owners));

  _out_size = 1000;
  _output = new unsigned char[_out_size];

  _flag_sem_id = semget(IPC_PRIVATE, 1, IPC_CREAT|0666);
  union semun un;
  un.val = 1;
  if (semctl(_flag_sem_id, 0, SETVAL, un)==-1) {
    perror("semctl");
    return;
  }
}

void delete_shmem_spec(const char*, ShmemSpec* spec)
{
  delete spec;
}

MemPool::~MemPool()
{
  _pool.applyToAll(delete_shmem_spec);
  cleanup();
  semctl (_flag_sem_id, 0, IPC_RMID, NULL);
}

void MemPool::cleanup()
{
  if (_serve_sock) {
    close(_serve_sock);
    _serve_sock = 0;
    _comm->RemoveAdministrativeConnection(_memconn);
    _comm->RemoveAdministrativeConnection(_sockconn);
    _memconn = _sockconn = NULL;
  }
}

void MemPool::owned_init_handler(IPMessage* msg)
{
  IPTShmMemorySpecStruct res;
  ShmemSpec* spec;
  IPPosixSharedMemory* shmem;

  IPTShmMemoryInitStruct* smi = (IPTShmMemoryInitStruct*) msg->FormattedData();

  if (_pool.find(smi->name, spec)) {
    if (spec->owner && msg->Connection() != spec->owner &&
        spec->owner->Active()) {
      printf("Cannot open pool %s by %s, %s (%d) still owns it\n",
             smi->name, msg->Connection()->Name(), spec->owner->Name(),
             spec->owner->Active());
      res.shm_id = res.sem_id = -4;
      goto sendit;
    }
    spec->owner = msg->Connection();

    if (_owned_list.find(spec) < 0)
      _owned_list.append(spec);

    IPFormat* fmt = _comm->FormatParser()->parseString(smi->fmt);
    if (fmt)
      fmt->ref();
    int eq = IPFormatEqualsAction::equals(fmt, spec->shmem->Formatter());
    if (fmt)
      fmt->unref();
    if (!eq || smi->max_size > spec->shmem->MaxSize()) {
      printf("Warning: owned size %d, format %s overriding initial for memory named %s\n",
             smi->max_size, smi->fmt, smi->name);
      remove(smi->name);
    } else {
      for (int i=0;i<_client_list.numElems();i++) {
        if (_client_list[i].shmem == spec)
          _client_list[i].last_tag = -1;
      }
      res.shm_id = spec->shm_id;
      res.sem_id = spec->sem_id;
      goto sendit;
    }
  }

  shmem = new IPManagerSharedMemory(_comm, smi->fmt, smi->max_size);
  spec = make_shmem_spec(smi->name, smi->use_tcp, smi->fmt, shmem,
                         msg->Connection(), NULL);
  if (!spec) {
    res.shm_id = -2;
    res.shm_id = -2;
  } else {
    _owned_list.append(spec);
    res.shm_id = spec->shm_id;
    res.sem_id = spec->sem_id;
  }

 sendit:
  res.flag_sem_id = _flag_sem_id;
  _comm->Reply(msg, IPT_SHM_MEMORY_INITIALIZED_MSG, &res);
}

ShmemSpec* MemPool::
make_shmem_spec(const char* name, bool use_tcp,
                const char* fmt, IPPosixSharedMemory* shmem,
                IPConnection* owner, IPConnection* remote)
{
  if (!shmem->Valid()) {
    delete shmem;
    return NULL;
  }
  if (shmem->MaxSize() > UDP_MAXSIZE){
    printf("%s propagated via TCP\n", name);
    use_tcp = true;
  }
  
  ShmemSpec* spec = new ShmemSpec;
  _comm->UseSharedMemory(shmem);
  spec->shmem = shmem;
  spec->shm_id = shmem->shmID();
  spec->sem_id = shmem->semID();
  spec->name = name;
  spec->owner = owner;
  spec->remote = remote;
  spec->use_tcp = use_tcp;
  spec->last_tag = spec->sequence_num = -1;
  _pool.enter(name, spec);

  return spec;
}

void MemPool::client_init_handler(IPMessage* msg)
{
  IPTShmMemorySpecStruct res;
  ShmemSpec* spec;
  IPPosixSharedMemory* shmem;
  IPString name;
  char* stripped_name;
  char* host;
  IPConfigFile params;

  IPTShmMemoryInitStruct* smi = (IPTShmMemoryInitStruct*) msg->FormattedData();

  stripped_name = (char*) smi->name;  // going to stomp on smi->name
  host = strchr(stripped_name, '@');
  if (host) {
    *host = '\0';
    host++;
  }

  if (_pool.find(stripped_name, spec)) {
    IPFormat* fmt = _comm->FormatParser()->parseString(smi->fmt);
    if (fmt)
      fmt->ref();
    int eq = IPFormatEqualsAction::equals(fmt, spec->shmem->Formatter());
    if (fmt)
      fmt->unref();
    if (!eq || smi->max_size > spec->shmem->MaxSize()) {
      if (!eq) {
        printf("Format does not match\n");
      } else {
        printf("Size %d bigger than max %d\n", smi->max_size,
               spec->shmem->MaxSize());
      }
      res.shm_id = res.sem_id = -7;
      goto sendit;
    } else {
      res.shm_id = spec->shm_id;
      res.sem_id = spec->sem_id;
      goto sendit;
    }
  }

  char* port;
  char common_req_host[300];
  char common_this_host[300];
  if (host) {
    port = strrchr(host, '|');
    if (!port) {
      res.shm_id = res.sem_id = -8;
      goto sendit;
    }
    *port = '\0';
    port++;
    if (!get_common_machine_name(host, common_req_host, 300)) {
      fprintf(stderr, "Cannot lookup requesting host name %s\n", host);
      res.shm_id = res.sem_id = -8;
      goto sendit;
    }
        
    host = common_req_host;
  } 

  get_common_machine_name(_comm->ThisHost(), common_this_host, 300);
  if (!host || !strcmp(host, common_this_host)) {
    IPPosixSharedMemory* shmem =
      new IPManagerSharedMemory(_comm, smi->fmt, smi->max_size);
    spec = make_shmem_spec(stripped_name, smi->use_tcp, smi->fmt,
                           shmem, NULL, NULL);
    if (!spec) {
      res.shm_id = -2;
      res.shm_id = -2;
    } else {
      res.shm_id = spec->shm_id;
      res.sem_id = spec->sem_id;
    }
    goto sendit;
  }


  int p;
  if (sscanf(port, "%d", &p) != 1) {
    res.shm_id = res.sem_id = -8;
    goto sendit;
  }

  char common_name[300];
  if (!get_common_machine_name(host, common_name, 300)) {
    printf("Bad host %s\n", host);
    res.shm_id = res.sem_id = -15;
    goto sendit;
  }
  name = common_name;
  name += port;
  IPConnection* other;
  params.set("string tag", "TCP");
  params.set("string name", name.getString());
  params.set("string host", host);
  params.set("int port", port);
  // printf("Shared memory initiating to '%s', '%s'\n", name.getString(), port);
  other = _comm->LookupConnection(name.getString());
  // printf("\tLooking up result %x\n", (unsigned int) other);
  if (!other || !other->Active()) {
    other = _comm->DirectConnect(&params, IPT_OPTIONAL);
    if (!other || !other->Active()) {
      printf("Failed to connect to %s on %s\n", port, name.getString());
      res.shm_id = res.sem_id = -9;
      goto sendit;
    }
  }

  IPTShmRemoteMemoryInitStruct req;
  req.port = _serve_port;
  req.use_tcp = smi->use_tcp;
  req.max_size = smi->max_size;
  req.fmt = smi->fmt;
  req.name = stripped_name;
  req.dest = _comm->ThisHost();
  IPMessage* resp;
  resp = _comm->Query(other, IPT_SHM_REMOTE_CLIENT_INIT_MSG, &req,
                      IPT_SHM_REMOTE_CLIENT_RESP_MSG);
  int remote_port;
  remote_port = (resp ? *(int*) resp->FormattedData() : -1);
  delete resp;
  if (remote_port < 0) {
    res.shm_id = res.sem_id = -10;
    goto sendit;
  }

  shmem = new IPOwnedSharedMemory(IPC_PRIVATE, IPC_PRIVATE, 
                                  _comm, smi->fmt, smi->max_size);
  spec = make_shmem_spec(stripped_name, smi->use_tcp, smi->fmt,
                         shmem, other, other);
  if (!spec) 
    res.shm_id = res.sem_id = -2;
  else {
    res.sem_id = spec->sem_id;
    res.shm_id = spec->shm_id;
  }

 sendit:
  res.flag_sem_id = _flag_sem_id;
  _comm->Reply(msg, IPT_SHM_MEMORY_INITIALIZED_MSG, &res);
}

void MemPool::remote_client_handler(IPMessage* msg)
{
  int res;
  ShmemSpec* spec;
  IPPosixSharedMemory* shmem;

  IPTShmRemoteMemoryInitStruct* smi =
    (IPTShmRemoteMemoryInitStruct*) msg->FormattedData();

  if (!_pool.find(smi->name, spec)) {
    shmem = new IPManagerSharedMemory(_comm, smi->fmt, smi->max_size);
    spec = make_shmem_spec(smi->name, smi->use_tcp, smi->fmt, shmem,
                           NULL, NULL);
  } else {
    IPFormat* fmt;
    fmt = _comm->FormatParser()->parseString(smi->fmt);
    if (fmt)
      fmt->ref();
    int eq;
    eq = IPFormatEqualsAction::equals(fmt, spec->shmem->Formatter());
    if (fmt)
      fmt->unref();
    if (!eq || smi->max_size > spec->shmem->MaxSize()) {
      res = -7;
      goto sendit;
    }
  }
    
  // printf("Initializing remote client %s\n", smi->name);
  RemoteClientSpec cspec;
  cspec.shmem = spec;
  cspec.connection = msg->Connection();
  cspec.socket_fd = socket(AF_INET, SOCK_DGRAM, 0);
  cspec.addr.sin_port = htons(smi->port);
  cspec.addr.sin_family = AF_INET;
  cspec.last_tag = -1;
  cspec.tcp_dest = msg->Connection();
  lookup_machine_name(smi->dest, &cspec.addr.sin_addr.s_addr);
  _client_list.append(cspec);

  res = _serve_sock;

 sendit:
  _comm->Reply(msg, IPT_SHM_REMOTE_CLIENT_RESP_MSG, &res);
}

void MemPool::close_handler(IPMessage* msg)
{
  const char* name = *(char**) msg->FormattedData();

  ShmemSpec* spec;
  if (!_pool.find(name, spec)) 
    return;

  if (spec->owner != msg->Connection())
    return;

  remove(name);
}

void MemPool::remote_client_canceler(IPMessage* msg)
{
  const char* name = *(char**) msg->FormattedData();

  ShmemSpec* spec;
  if (!_pool.find(name, spec)) 
    return;

  for (int i=0;i<_client_list.numElems();i++) {
    RemoteClientSpec& rs = _client_list[i];
    if (!strcmp(name, rs.shmem->name.getString()) && 
        msg->Connection() == rs.connection) {
      _client_list.remove(i);
      return;
    }
  }
}

int MemPool::hasNewData() const
{
  int tag;
  for (int i=0;i<_owned_list.numElems();i++) {
    ShmemSpec* spec = _owned_list[i];
    tag = spec->shmem->UpdatedTag();
    if (tag != spec->last_tag) {
      return 1;
    }
  }

  return 0;
}

void MemPool::remove(const char* name)
{
  ShmemSpec* spec;
  if (!_pool.find(name, spec))
    return;

  int i;
  for (i=_owned_list.numElems()-1;i>=0;i--) 
    if (_owned_list[i] == spec) 
      _owned_list.remove(i);
  for (i=_client_list.numElems()-1;i>=0;i--) 
    if (_client_list[i].shmem == spec) 
      _client_list.remove(i);
  _pool.remove(name);
  _comm->CloseSharedMemory(spec->shmem);
}

void MemPool::mem_activity(IPConnection*)
{
  // update tags
  int tag;
  int i;
  for (i=0;i<_owned_list.numElems();i++) {
    ShmemSpec* spec = _owned_list[i];
    if (!spec->shmem->Update())
      continue;
    spec->last_tag = spec->shmem->Tag();
  }

  IPVector<int> dead;
  for (i=0;i<_client_list.numElems();i++) {
    RemoteClientSpec& spec = _client_list[i];
    tag = spec.shmem->last_tag;
    if (tag != spec.last_tag) {
      if (!output_packet(spec)) {
        dead.append(i);
      }
      spec.last_tag = tag;
    }
  }

  for (i=dead.numElems()-1;i>=0;i--) 
    _client_list.remove(dead[i]);
}

bool MemPool::output_packet(RemoteClientSpec& spec)
{
  int align, byte_order;
  unsigned char* data;
  int size;

  if (!spec.shmem->shmem->Data(size, data, align, byte_order))
    return true;

  if (spec.shmem->use_tcp) {
    if (!spec.tcp_dest) {
      fprintf(stderr, "MemPool::output_packet: no remote for %s\n",
              spec.shmem->name.getString());
      return false;
    }
    IPTShmMessageStruct sm;
    sm.len = size;
    sm.data = data;
    sm.name = (char*) spec.shmem->name.getString();
    if (!_comm->SendMessage(spec.tcp_dest, IPT_SHM_MESSAGE_MSG, &sm)) {
      fprintf(stderr, "MemPool::output_packet: error sending message\n");
      return false;
    }
    // printf("Propagated %s (%d) via tcp\n", sm.name, sm.len);
  } else {
    const char* name = spec.shmem->name.getString();
    int name_len = strlen(name)+1;
    if (name_len > 256) {
      printf("Cannout output message with long name, '%s'\n", name);
      return true;
    }
    
    int buffer_size = size + name_len + sizeof(unsigned int);
    if (buffer_size > _out_size) {
      _out_size = buffer_size;
      delete [] _output;
      _output = new unsigned char[_out_size];
    }

    _sequence_num++;
    int sn = htons(_sequence_num);
    bcopy(&sn, _output, sizeof(unsigned int));
    bcopy(name, _output+sizeof(int), name_len);
    bcopy(data, _output+name_len+sizeof(int), size);

    if (sendto(spec.socket_fd, _output, buffer_size, 0,
               (sockaddr*) &spec.addr, sizeof(struct sockaddr_in)) < 0) {
      perror("sendto");
      return false;
    }
  }
  return true;
}

void MemPool::monitorMemory()
{
  static struct sembuf serv_wait[1] = {
    { 0, 0, 0 } /* wait for clients to indicate new memory*/
  };
  static struct sembuf serv_unlock[1] = {
    { 0, 1, 0 }  /* reset flag */
  };

  while (semop(_flag_sem_id, serv_wait, 1) != -1) {
    if (kill(_pid, _sig_num) == -1)
      break;
    if (semop(_flag_sem_id, serv_unlock, 1) == -1) {
      break;
    }
  }
}

void MemPool::process_data(int seqnum, const char* name,
                           unsigned char* data, int len)
{
  ShmemSpec* spec;
  if (!_pool.find(name, spec))
    return;

  if (!spec->remote)
    return;

  if (spec->sequence_num > seqnum &&
      (((unsigned long) spec->sequence_num) - ((unsigned long) (seqnum))
       < ((unsigned long) (0x7FFF)))) {
    printf("Rejecting out of sequence packet %d %d\n",
           spec->sequence_num, seqnum);
    return;
  }
  spec->sequence_num = seqnum;

  unsigned char* area = spec->shmem->GetOutputArea();
  if (!area)
    return;

  // first fix byte order issues
  IPFormat* fmt = spec->shmem->Formatter();
  if (fmt && spec->remote->ByteOrder() != BYTE_ORDER) {
    // this could get awfully inefficient, could either do some more
    // caching, or, create a convert byteorder in place
    // action that could do this without going through the intermediate
    // data at all.
    void* fmtdata = IPUnpackAction::unpack(fmt, data, len,
                                        spec->remote->ByteOrder(), ALIGN);
    spec->shmem->PutFormattedData(fmtdata);
    IPDeleteAction::deleteData(fmt, fmtdata, data, len);
  } else {
    if (len > spec->shmem->MaxSize())
      len = spec->shmem->MaxSize();
    bcopy(data, area, len);
    spec->shmem->PutData(len);
  }
}

void MemPool::sock_activity(IPConnection*)
{
  int len = recvfrom(_serve_sock, _input, UDP_BUFFER_MAX,
                            0, NULL, NULL);
  if (len < 0) {
    perror("recvfrom");
    return;
  }

  int seqnum = ntohs(*(int*) _input);
  const char* name = (const char*) _input+sizeof(int);
  int name_len = strlen(name)+1;
  unsigned char* data = (unsigned char*) name+name_len;
  len -= sizeof(int) + name_len;

  process_data(seqnum, name, data, len);
}

void MemPool::message_handler(IPMessage* msg)
{
  IPTShmMessageStruct* sm = (IPTShmMessageStruct*) msg->FormattedData();

  // printf("Received tcp propagation of %s %d\n", sm->name, sm->len);

  process_data(0, sm->name, sm->data, sm->len);
}

void MemPool::scan_owners(IPTimer*)
{
  for (int i=0;i<_owned_list.numElems();i++) {
    if (!_owned_list[i]->owner->Active()) {
      _owned_list[i]->shmem->PutData(0);
    }
  }
}

IPCommunicator* ipt;

void cntrl_c_handler(int)
{
  ipt->Finish();
}

void* mem_monitor(void* arg)
{
  MemPool* pool = (MemPool*) arg;
  sigset_t sigset;
  sigemptyset(&sigset);
  sigaddset(&sigset, pool->signalNum());
  sigaddset(&sigset, SIGINT);
  pthread_sigmask(SIG_BLOCK, &sigset, NULL);

  while (!ipt->Finished())
    pool->monitorMemory();

  return NULL;
}

int main(int argc, const char** argv)
{
  IPConfigFile config;
  
  if (argc > 1) {
    if (!config.open(argv[1])) {
      printf("Problem loading configuration file %s, using defaults\n",
             argv[1]);
    }
  }

  char buf[200];
  sprintf(buf, "iptshmgr%d", getpid());
  const char* mod_name = config.getString("module_name", buf);
  ipt =
    IPCommunicator::Create(mod_name,
                           config.getString("ipt_spec", "unix: port=1389;"));
  if (!ipt) {
    printf("Problem creating communicator '%s', '%s'\n", mod_name, 
           config.getString("ipt_spec", "unix: port=1389;"));
  }

  int sig = config.getInt("signal", SIGRTMIN);
  signal(sig, rthandle);

  MemPool* pool = new MemPool(ipt, config.getInt("udp_port", 1389),
                              config.getInt("sig_num", sig));
  signal(SIGINT, cntrl_c_handler);

  pthread_t mem_thread;
  pthread_create(&mem_thread, NULL, mem_monitor, (void*) pool);

  ipt->MainLoop();

  pthread_cancel(mem_thread);
  delete pool;
  delete ipt;
}

