#include <stdio.h>

#include <ConfigSource/ConfigSource.h>
#include <ConfigSource/Repository.h>
#include <StatusSource/StatusSource.h>
#include <TimeSource/TimeSource.h>

#include "RepoMonInterface.h"

#include <Fl/Fl_Box.h>
#include <Fl/Fl_Value_Output.h>
#include <Fl/fl_draw.h>

class MonitorIntf {
public:
  MonitorIntf(const char* name);
  ~MonitorIntf();

  bool initialize(StatusSource* source);

  void update();

public:
  Fl_Pack* group;
  Fl_Box* status;
  Fl_Box* time_output;
  Fl_Box* msg_output;
  
  utils::String time_value;
  utils::String msg_value;

private:
  utils::String _name;
  StatusSource::ModuleStatus* _module_status;
  utils::Time _last_run_change, _last_update_change;
  utils::Time _last_run, _last_update;
};

RepoMon::RepoMon(const char* spec)
  : Module("RepoMon", spec)
{
  _repomon_intf = NULL;
  _finished = false;
  _hosts = NULL;

  clear_modules();
}

RepoMon::~RepoMon()
{
  clear_modules();

  delete _repomon_intf;
}

bool RepoMon::
initialize(ConfigSource* config,utils::SymbolTable* table)
{
  const char* spec;

  spec = config->getString("repository_spec");
  _repository = create<ConfigSource>(spec);
  if (!_repository) {
    fprintf(stderr, "RepoMon: No repository\n");
    return false;
  }
  Repository::set(table, _repository);

  spec = config->getString("status_source_spec");
  _status_source = create<StatusSource>(spec);
  if (!_status_source) {
    fprintf(stderr,
            "RepoMon: Spec '%s', creates no valid status_source\n", spec);
    return false;
  }

  _interval = config->getDouble("interval", 0.1);

  _repomon_intf = new RepoMonInterface(this);
  _repomon_intf->module_pack->spacing(5);

  load_modules();

  _repomon_intf->window->show();

  return true;
}

void RepoMon::finish()
{
  _finished = true;
}

void RepoMon::refresh_config()
{
  _repository->set("refresh", "1");
}

bool RepoMon::run()
{
  std::list<MonitorIntf*>::iterator i;
  for (i=_modules.begin(); i != _modules.end(); i++) {
    (*i)->update();
  }
  
  Fl::check();
  _repository->sleep(_interval);

  return !_finished;
}

void RepoMon::clear_modules()
{
  delete [] _hosts;
  _hosts = NULL;

  if (_repomon_intf) {
    Fl_Widget* child;
    while ((child = _repomon_intf->module_pack->child(1))) {
      _repomon_intf->module_pack->remove(child);
      delete child;
    }
  }

  std::list<MonitorIntf*>::iterator i;
  for (i=_modules.begin(); i != _modules.end(); i++) {
    delete (*i);
  }
  _modules.clear();
}

void RepoMon::load_modules()
{
  clear_modules();

  int num_hosts = _repository->numValues("string Hosts.hosts");
  printf("%d hosts\n", num_hosts);
  if (!num_hosts) {
    _repomon_intf->module_pack->redraw();
    return;
  }

  int i, j;
  const char** hosts = new const char*[num_hosts];
  _repository->getStrings("string Hosts.hosts", hosts, num_hosts);
  _hosts = new utils::String[num_hosts];
  for (i=0;i<num_hosts;i++)
    _hosts[i] = hosts[i];
  delete [] hosts;

  const char *module_names[100];

  Fl_Box* host_box;
  int w=0, h=0;
  fl_font(FL_HELVETICA_ITALIC, 14);

  char buffer[500];
  for (i=0;i<num_hosts;i++) {
    fl_measure(_hosts[i].getString(), w, h);
    host_box = new Fl_Box(0, 0, w, h+5);
    host_box->label(_hosts[i].getString());
    host_box->labelfont(FL_HELVETICA_ITALIC);
    host_box->labelsize(14);
    host_box->align(FL_ALIGN_LEFT | FL_ALIGN_BOTTOM | FL_ALIGN_INSIDE);
    _repomon_intf->module_pack->add(host_box);
    snprintf(buffer, 500, "string Hosts.%s.modules", _hosts[i].getString());
    int num_modules = _repository->getStrings(buffer, module_names, 100);
    for (j=0;j<num_modules;j++) {
      if (!strncmp(module_names[j], "iptshmgr", 7))
        continue;
      snprintf(buffer, 500, "bool Modules.%s.is_module", module_names[j]);
      bool is_module = _repository->getBool(buffer, true);
      if (!is_module) {
        printf("Warning: %s is not a module, skipping it\n", module_names[j]);
        continue;
      }
        
      MonitorIntf* intf = new MonitorIntf(module_names[j]);
      _modules.push_back(intf);
      _repomon_intf->module_pack->add(intf->group);
    }

    std::list<MonitorIntf*>::iterator iter;
    for (iter=_modules.begin(); iter != _modules.end(); iter++) {
      (*iter)->initialize(_status_source);
    }
  }

  _repomon_intf->module_pack->redraw();
}

MonitorIntf::MonitorIntf(const char* name)
{
  Fl_Box* box;
  int w = 0;
  int h = 0;
  fl_font(FL_HELVETICA, 12);
  fl_measure(name, w, h);

  _name = name;

  group = new Fl_Pack(0, 0, 100, h);
  group->type(Fl_Pack::HORIZONTAL);
  new Fl_Box(0,0, 5, h);
  status = new Fl_Box(0,0, h, h);
  status->box(FL_FLAT_BOX);
  status->color(FL_BLACK);
  box = new Fl_Box(0, 0, 120, h);
  box->label(_name.getString());
  box->align(FL_ALIGN_LEFT | FL_ALIGN_INSIDE | FL_ALIGN_CLIP);
  box->labelfont(FL_HELVETICA_BOLD);
  box->labelsize(12);

  time_output = new Fl_Box(0, 0, 375, h);
  time_output->box(FL_FLAT_BOX);
  time_output->label(time_value.getString());
  time_output->labelfont(FL_HELVETICA);
  time_output->labelsize(11);
  time_output->align(FL_ALIGN_LEFT | FL_ALIGN_INSIDE | FL_ALIGN_CLIP);

  msg_output = new Fl_Box(0, 0, 270, h);
  msg_output->box(FL_FLAT_BOX);
  msg_output->label(msg_value.getString());
  msg_output->labelfont(FL_HELVETICA_ITALIC);
  msg_output->labelsize(11);
  msg_output->align(FL_ALIGN_LEFT | FL_ALIGN_INSIDE | FL_ALIGN_CLIP);

  group->end();
}

bool MonitorIntf::initialize(StatusSource* status_source)
{
  _module_status = status_source->moduleStatus(_name.getString());

  _last_update_change = _last_run_change = TimeSource::now();

  if (!_module_status) {
    fprintf(stderr, "MonitorIntf: cannot create monitor for '%s'\n",
            _name.getString());
    return false;
  }
  return true;
}

MonitorIntf::~MonitorIntf()
{
}

void MonitorIntf::update()
{
  StatusSource::Status mod_status;

  if (!_module_status) {
    status->color(FL_BACKGROUND_COLOR);
    time_value = msg_value = "";
  } else if (!_module_status->getStatus(mod_status, STATUS_NONBLOCKING)) {
    status->color(FL_BLACK);
    time_value = msg_value = "";
  } else if (mod_status.state == ConfigSource::NOT_RUNNING) {
    status->color(FL_RED);
    time_value = msg_value = "";
  } else {
    utils::Time now = TimeSource::now();
    double delta_run_local, delta_run_real;
    if (mod_status.last_run != _last_run) {
      delta_run_local = 0;
      _last_run_change = now;
      _last_run = mod_status.last_run;
    } else {
      delta_run_local = (double) (now - _last_run_change);
    }
    delta_run_real = (double) (now - _last_run);

    double delta_update_local, delta_update_real;
    if (mod_status.last_update != _last_update) {
      delta_update_local = 0;
      _last_update_change = now;
      _last_update = mod_status.last_update;
    } else {
      delta_update_local = (double) (now - _last_update_change);
    }
    delta_update_real = (double) (now - _last_update);
      
    char buffer[400];
    snprintf(buffer, 400,
             "Run %5.2f (%5.2f), Update %5.2f (%5.2f), Cycle %5.3f"
             " Status %5.2f, Conf %5.2f",
             delta_run_local, delta_run_real,
             delta_update_local, delta_update_real, mod_status.avg_cycle_time,
             mod_status.status, mod_status.confidence);
    time_value = buffer;
    if (mod_status.msg)
      msg_value = mod_status.msg;
    else
      msg_value = "";

    switch (mod_status.state) {
    case ConfigSource::INITIALIZING:
    case ConfigSource::INITIALIZING_DISPLAY:
      status->color(FL_BLUE);
      break;
    case ConfigSource::PAUSED:
      status->color(FL_DARK_YELLOW);
      break;
    case ConfigSource::BLOCKED:
      status->color(FL_YELLOW);
      break;
    case ConfigSource::RUNNING:
      if (delta_run_local < 2.0)
        status->color(FL_GREEN);
      else if (delta_update_local < 2.0)
        status->color(FL_YELLOW);
      else
        status->color(FL_RED);
      break;
    default:
      status->color(FL_BLACK);
      break;
    }
  }

  time_output->label(time_value.getString());
  msg_output->label(msg_value.getString());
  group->redraw();
}

MODULE_MAIN(repomon, RepoMon);


