%module _ConfigSource
%{
#include <utils/Input.h>
#include <utils/Output.h>
#include <utils/SpecElem.h>
#include <ConfigSource/ConfigSource.h>

typedef utils::SymbolTable SymbolTable;
typedef utils::SymbolTable ConfigFile;

class PyNotifier {
public:
  PyNotifier(PyObject* method) { _method = NULL; setMethod(method); }
  ~PyNotifier() { Py_XDECREF(_method); }
  
  void execute(const char* name, utils::Type type, void* data, int num_elems)
  {
    PyObject* result;
    if (PyCallable_Check(_method)) {
        PyObject* arglist = Py_BuildValue("(s)", name);
        result = PyObject_CallObject(_method, arglist);
        Py_DECREF(arglist);
    } else
        result = PyObject_CallMethod(_method, "execute", "(s)", name);
    if (!result) {
        PyErr_Print();
    } else
        Py_DECREF(result);
  }

  void setMethod(PyObject* method) {
    if (method == _method)
        return;
    Py_XDECREF(_method);
    _method = method;
    Py_XINCREF(_method);
  }

  static void notify(const char* name, utils::Type type,
                     void* data, int num_elems, void* cbdata) {
    ((PyNotifier*) cbdata)->execute(name, type, data, num_elems);
  }

private:
  PyObject* _method;
};

%}

%init %{
   const char* data_path = getenv("DATA_PATH");
   if (data_path) {  // customize file path for Input operations
     utils::Input::addEnvDirectoriesLast("DATA_PATH");
   } else {  // default path is .:$DATA_DIR:$LOCAL_CONFIG_DIR:$CONFIG_DIR:$HOME
     utils::Input::addDirectoryFirst(".");
     utils::Input::addEnvDirectoriesLast("DATA_DIR");
     utils::Input::addEnvDirectoriesLast("LOCAL_CONFIG_DIR");
     utils::Input::addEnvDirectoriesLast("CONFIG_DIR");
     utils::Input::addEnvDirectoriesLast("HOME");
   }
   utils::SpecElem::initClass();
%}

%typemap(python, except) bool {
    $function
    if (!$source) {
        PyErr_SetString(PyExc_RuntimeError, "ConfigSource error");
        return NULL;
    }
}

%typemap(python, except) ConfigSource* {
    $function
    if (!$source) {
        PyErr_SetString(PyExc_RuntimeError, "Invalid specification");
        return NULL;
    }
}

%typemap(python, except) ConfigFile* {
    $function
    if (!$source) {
        PyErr_SetString(PyExc_RuntimeError, "No such structure");
        return NULL;
    }
}

%typemap(python, in) PyObject* {
    $target = $source;
}

// When we add a method that returns a python object,
// it should be returned as a python object
%typemap(python, out) PyObject* {
    $target = $source;
}

%import SymbolTable.i

%name(_ConfigSource) class ConfigSource {
public:
  static ConfigSource* generate(const char* spec, SymbolTable* table);
  static ConfigSource* create(const char* spec, SymbolTable* table);

  void ref();
  void unref();

  int numValues(const char* name);

  // convenience methods for getting singleton parameters
  int getInt(const char* name, int def=0);
  float getFloat(const char* name, float def=0.0);
  double getDouble(const char* name, double def=0.0);
  char getChar(const char* name, char def='\0');
  const char* getString(const char* name, const char* def = "");
  int getBool(const char* name, bool def=false);
  ConfigFile* getSubFile(const char* name);

  int processEvents(double timeout=-1);
  bool refresh();
  %name(parseSet) bool set(const char* name, const char* value);

  %addmethods {
    PyObject* getInts(const char* name) {
      int n = self->numValues(name);
      PyObject* res = PyTuple_New(n);
      if (!n)
        return res;
      int* data = new int[n];
      int actual_n = self->getInts(name, data, n);
      if (actual_n != n) {
        Py_DECREF(res);
        PyErr_SetString(PyExc_RuntimeError, "Error getting values");
        return NULL;
      }
      for (int i=0;i<n;i++) 
        PyTuple_SET_ITEM(res, i, PyInt_FromLong(data[i]));
      delete [] data;

      return res;
    }

    PyObject* getFloats(const char* name) {
      int n = self->numValues(name);
      PyObject* res = PyTuple_New(n);
      if (!n)
        return res;
      double* data = new double[n];
      int actual_n = self->getDoubles(name, data, n);
      if (actual_n != n) {
        Py_DECREF(res);
        PyErr_SetString(PyExc_RuntimeError, "Error getting values");
        return NULL;
      }
      for (int i=0;i<n;i++) 
        PyTuple_SET_ITEM(res, i, PyFloat_FromDouble(data[i]));
      delete [] data;

      return res;
    }

    PyObject* getStrings(const char* name) {
      int n = self->numValues(name);
      if (n < 0) {
        PyErr_SetString(PyExc_RuntimeError, "Error getting values");
        return NULL;
      }
      PyObject* res = PyTuple_New(n);
      if (!n)
        return res;
      const char** data = new const char*[n];
      int actual_n = self->getStrings(name, data, n);
      if (actual_n != n) {
        Py_DECREF(res);
        PyErr_SetString(PyExc_RuntimeError, "Error getting values");
        return NULL;
      }
      for (int i=0;i<n;i++) 
        PyTuple_SET_ITEM(res, i, PyString_FromString(data[i]));
      delete [] data;

      return res;
    }

    PyObject* set(const char* name, PyObject* val) {
      PyObject* repr;
      bool res;
      if (!PyString_Check(val) && PySequence_Check(val)) {
        utils::Output output;
        output.setBuffer(NULL, 0, utils::Output::standardResize);
        int len = PySequence_Length(val);
        PyObject* item;
        for (int i=0; i<len; i++) {
          item = PySequence_GetItem(val, i);
          repr = PyObject_Repr(item);
          output.write(PyString_AS_STRING(repr));
          if (i!=len-1)
            output.write(' ');
          Py_DECREF(repr);
          Py_DECREF(item);
        }
        output.write('\0');

        void* buf_ptr;
        int size;
        output.getBuffer(buf_ptr, size);
        res=self->set(name, (const char*) buf_ptr);
        delete (char*) buf_ptr;
      } else {
        repr = PyObject_Repr(val);
        res=self->set(name, PyString_AS_STRING(repr));
        Py_DECREF(repr);
      }

      if (!res) {
        PyErr_SetString(PyExc_RuntimeError, "Unable to set");
        return NULL;
      }
      Py_INCREF(Py_None);
      return Py_None;
    }

    PyObject* notify(const char* name, PyObject* method) {
      PyNotifier* notifier = new PyNotifier(method);
      if (!self->notify(name, PyNotifier::notify, notifier)) {
        PyErr_SetString(PyExc_RuntimeError, "Problem setting up notification");
        return NULL;
      }
      Py_INCREF(Py_None);
      return Py_None;
    }
  }      
};



