import os
import socket
import sys
import string
import time
import WinSender
import ConfigSource
import MgrDisp
import SymbolTable
import StatusSource

class RepoMan:
  def __init__(self, spec=None, display=None, no_display=0):
    if spec is None:
      try:
        spec = sys.argv[1]
      except:
        spec = "repoman.cfg"
      # switch comments to turn verbose IPT stuff on and off
      spec = 'iptserver: name=%s; ModuleName=RepoMan; output=none;' % spec
      #spec = 'iptserver: name=%s; ModuleName=RepoMan;' % spec

    self.managed = []
      
    self.logging = 0
    self.finished = 0
    self.symbol_table = SymbolTable.SymbolTable()
    self.server = ConfigSource.ConfigSource(spec, self.symbol_table);
    self.status_source = StatusSource.StatusSource_create("repository",
                                                          self.symbol_table)
    self.status_source.ref()
    self.server.ref()
    self.hosts = {}
    self.num_hosts = 0
    self.overridden = {}
    self.shell = "/bin/tcsh"
    self.update_time = self.server.getDouble("update_time", 0.1)

    if not no_display and \
       not display and not self.server.getBool("no_display", 0):
      try:
        display = os.environ['DISPLAY']
      except KeyError:
        display = None
    else:
      display = None

    if display:
      self.disp = MgrDisp.MgrDisp(self)
      self.sender = WinSender.WinSender(display)
    else:
      self.disp = None
      self.sender = None

    try:
      cur_host = os.environ['IPTHOST']
      self.host = cur_host
    except KeyError:
      self.host = socket.gethostname()
    self.host = socket.gethostbyname(self.host)
    self.generated_configs = []
    self.hosts = {}
    self.hosts[self.host] = {}

  def __del__(self):
    print "Deleting"
    self.status_source.unref()
    self.server.unref()

  def sendString(self, window, s):
    if self.sender:
      self.sender.sendString(window, s)

  def start_all(self, name):
    if self.in_restart_process:
      return
    self.in_restart_process = 1
    self.setup()
    time.sleep(1)
    self.run()
    self.in_restart_process = 0

  def stop_all(self, name):
    print "*****Stopping everything currently running"
    if self.in_restart_process:
      return
    self.in_restart_process = 1
    self.invoke("finish")
    time.sleep(1)
    self.invoke("stop")
    self.in_restart_process = 0

  def restart(self, name):
    if self.in_restart_process:
      return
    self.in_restart_process = 1
    self.total_restart()
    self.in_restart_process = 0

  def refresh(self, name):
    print "Refreshing configuration files"
    self.refresh_config()

  def make_creation_string(self, m, params):
    create_string = '%s(name="%s", ' % (params.get('class', 'Module'), m)

    val = params.geometry
    if val and val != '':
      create_string = create_string + 'geom="%s",' % val

    val = params.font
    if val and val != '':
      create_string = create_string + 'font="%s",' % val

    val = params.dir
    if val and val != '':
      if val[0] == '$':
        parts = string.split(val, '/')
        val = os.environ[parts[0][1:]] + '/' + string.join(parts[1:], '/')
      create_string = create_string + 'dir="%s",' % val

    val = params.xterm_options
    if val and val != '':
      create_string = create_string + 'xterm_options="%s",' % val

    val = params.preamble
    if val and val != '':
      create_string = create_string + 'preamble="%s",' % val

    val = params.command
    if val and val != '':
      create_string = create_string + 'command="%s",' % val

    # note, we require a host, if none is provided, use the local host name
    val = params.get("host", self.host)
    if not val or val == '':
      val = self.host
    create_string = create_string + 'host="%s",' % val

    val = params.user
    if val and val != '':
      create_string = create_string + 'user="%s",' % val

    val = params.redirect_output
    if val and val != '':
      create_string = create_string + "redirect_output='%s'," % val  
    val = params.stop_all
    if val and val != '':
      create_string = create_string + "stop_all='%s'," % val  

      
    val = params.embed_command_in
    if val and val != '':
      create_string = create_string + 'embed_command_in="%s",' % val

    create_string = create_string + ')'
    return create_string

  def initialize(self, active_modules=None):
    self.server.set("string stop_all", 0)
    self.server.notify("string stop_all", self.stop_all)
    self.server.set("bool start_all", 0)
    self.server.notify("bool start_all", self.start_all)
    self.server.set("bool restart", 0)
    self.server.notify("bool restart", self.restart)
    self.server.set("bool refresh", 0)
    self.server.notify("bool refresh", self.refresh)
    self.in_restart_process = 0

    self.generated_configs = []
    if active_modules is None:
      active_modules = self.server.getStrings("active_modules");
      active_modules = list(active_modules)
      active_modules = active_modules + sys.argv[2:]
    ubiquitous = []
    for m in active_modules:
      if self.overridden.has_key(m):
        continue
      
      good = 1
      for e in self.managed:
        if m == e.name:
          good = 0
          break

      if not good:
        continue
      
      print "Trying", m

      self.shell = self.server.getString("string shell", '/bin/tcsh -l')

      try:
        params = self.server.getSubFile("Modules.%s" % m)
      except RuntimeError:
        raise RuntimeError, "Problem accessing module parameters for %s" % m

      # set the active flag:  this is used by RepoSub to indicate a module
      # that does not have to be started
      if not params.active:
        params.set("bool active", "true")
        self.generated_configs.append(('bool Modules.%s.active' % m, 'true'))

      override = params.get("override_module", "")
      if override:
        if type(override) != type(()):
          override = [ override ]
        for o in override:
          self.overridden[o] = m
          print "Overriding", o, m

      m = eval(self.make_creation_string(m, params))
      self.manage(m)

      m.restart_time = float(params.get("float restart_time", 0.0))
      m.initialize_time = float(params.get("float initialize_time", 15.0))
      #print m.name, m.restart_time, m.initialize_time

      memory_regions = params.get("owned_memory")
      if type(memory_regions) != type(()):
        if memory_regions is None:
          memory_regions = []
        else:
          memory_regions = [ memory_regions ]
      m.memory_region_names = memory_regions;

      if not self.hosts.has_key(m.host):
        self.hosts[m.host] = {}

      if int(params.get("bool ubiquitous", 0)):
        m.client_index = 1
        m.client_geom = params.client_geom
        m.client_inc = params.client_inc
        m.client_spec = params.get("string client_spec")
        m.client_command = params.get("string client_command", params.command)
        ubiquitous.append(m)

      required = params.get("required_modules")
      if required:
        if type(required) != type(()):
          required = [ required ]
        for e in required:
          try:
            e = self.overridden[e]
          except KeyError:
            pass
          active_modules.append(e)

    y = 0;
    for h in self.hosts.keys():
      print "Host", h
      self.manage(self.memManager(repr(y), h, None))
      y = y + 100;

      for u in ubiquitous:
        if u.host != h:
          name = u.name + '_' + h
          name = string.join(string.split(name, '.'), '_')

          # make sure we make a record of where this is running, specifically
          # to keep the repository shared memory stuff happy
          if u.dir:
            mconfig = ('struct Modules.%s' % name,
                       '{string host=%s; string command=%s; string dir=%s;}' %
                       (h, u.client_command, u.dir))
          else:
            mconfig = ('struct Modules.%s' % name,
                       '{string host=%s; string command=%s;}' %
                       (h, u.client_command))
          self.generated_configs.append(mconfig)
          self.server.parseSet(mconfig[0], mconfig[1])

          # make sure we set up parameters correctly
          config = ('struct ' + name, u.client_spec % u.host)
          self.generated_configs.append(config)
          self.server.parseSet(config[0], config[1])

          m = u.__class__(name=name, host=h,
                          geom=u.client_geom%(u.client_index*u.client_inc),
                          font=u.font, dir=u.dir, 
                          preamble=u.preamble, command=u.client_command,
                          xterm_options=u.xterm_options,redirect_output=u.redirect_output)
          self.manage(m)
          
          u.client_index = u.client_index + 1

    # Any overridden modules are marked as active, not because they are
    # but so RepoSub will not bother to start them as dependencies
    for m in self.overridden.keys():
      key = 'bool Modules.%s.active' % m
      active = int(self.server.getBool(key, 0))
      if not active:
        self.server.set(key, 1)
        self.generated_configs.append((key, 'true'))

    self.setup_memory_from_modules()

  def manage(self, *windows):
    for w in windows:
      self.managed.append(w)
      w.manage(self)

  def startup(self):
    if not self.disp:
      print "No display, not starting up windows"
      return
    self.invoke("create", manager=0)
    need_wait=0
    for w in self.managed:
      if w.startup():
        need_wait = 1

    return need_wait

  def set_generated_config_elements(self):
    for config in self.generated_configs:
      self.server.parseSet(config[0], config[1])
    self.setup_memory_from_modules()

  def setup_memory_from_modules(self):
    for m in self.managed:
      try:
        params = self.server.getSubFile("Modules.%s" % m.name)
        # reset host to default if not set already
        params.get("host", self.host)
        # make sure is_module is set
        is_module = int(params.get("is_module", 1))
        if is_module==1:
          is_module = issubclass(m.__class__, Module)
        self.server.parseSet("struct Modules.%s.is_module" % m.name,
                             "%d" % is_module)
      except:
        pass
      
      for r in m.memory_region_names:
        var_name = "string SharedMemory.%s.owner" % r
        mem_name = self.server.getString(var_name)
        if len(mem_name) == 0:
          self.server.parseSet(var_name, m.name)

      host_list = self.hosts[m.host]
      if not host_list.has_key(m.name):
        host_list[m.name] = 1

    # now construct the host entries to let external tool find out what should
    # be running on what machines
    host_names = []
    for h in self.hosts.keys():
      name = string.join(string.split(h, '.'), '_')
      host_names.append(name)
      self.server.parseSet("string Hosts.%s.modules" % name,
                           string.join(self.hosts[h].keys()) + ";")

    print "Setting up hosts", self.hosts

    self.server.parseSet("string Hosts.hosts", string.join(host_names))

  def refresh_config(self):
    self.server.refresh()
    self.set_generated_config_elements()

  def setup(self):
    self.refresh_config();
    self.invoke("setup", manager=0)
    
  def run(self):
    self.refresh_config();
    self.invoke("run", manager=0)
    
  def shutdown(self):
    for w in self.windows:
      w.status = 0
    if self.sender:
      self.sender.destroyAll()

  def destroy(self):
    self.shutdown()

  def invoke(self, routine, manager=1):
    # need protection here from infinite recursion
    if manager:
      try:
        apply(getattr(self.__class__, routine), [self])
        return
      except AttributeError:
        pass

    for window in self.managed:
      window.invoke(routine)

  def quit(self):
    self.finished = 1

  def serve(self):
    while not self.finished:
      if self.disp:
        self.disp.update()
      self.update()
      self.server.processEvents(self.update_time)

  def update(self):
    # for each managed module 
    for m in self.managed:
      # that has a restart time and has a running status
      if m.restart_time > 0 and m.status >= 3:
        # make sure we get a connection to the module status
        if m.module_status is None:
          # note:  host can get unset by all the clearing upon run/setup
          #        so we reset it here to be safe
          self.server.parseSet('string Modules.%s.host' % m.name, m.host);
          m.module_status = self.status_source.moduleStatus(m.name)
          # it may not be ready, i.e., the mem. manager may still be starting
          if m.module_status.this == 'NULL':
            m.module_status = None
            continue
        # get the module status
        mstatus = m.module_status.getStatus()
        # make sure we don't thrash, i.e., don't start monitoring until at
        # least m.restart_time seconds have passed since the run start
        now = time.time()
        if now - m.start_run < m.restart_time:
          continue
        # print m.name, mstatus
        # by default assume we are not running
        not_running = 1
        # and assume the time that matters is the restart time
        critical_time = m.restart_time
        if mstatus:
          last_update = mstatus[0]
          not_running = (mstatus[2] == 'NotRunning')
          # if we are initializing, the time that matters is the
          # initialize_time, which allows for a long initialization
          if mstatus[2] == 'Initializing':
            critical_time = m.initialize_time
        else:
          last_update = 0
        if not_running:
          # Make sure we at least start running within m.restart_time seconds
          # Note a side effect for now is that if we go to not-running from
          # running we will initiate a restart immediately because start_run
          # is so far in the past
          last_update = m.start_run
        else:
          # if we get any kind of response from a module, then it at least is
          # starting up
          m.retries = 0
        if not m.first_report:
          # if first time, just set let variables be set up
          if last_update > 0:
            m.first_report = 1
        else:
          # if we have had 5 unsuccessful retries, lets try restarting the
          # whole bloody system over again
          if m.retries >= 5:
            print "*****Total failure with module %s, restarting whole system" % m.name
            self.total_restart()
            return

          # otherwise, restart if the time since last update exceeds critical
          if now-last_update > critical_time:
            m.restart()

  def total_restart(self):
    print "*****Finishing*****"
    self.invoke("finish")
    time.sleep(1)
    print "*****Stopping*****"
    self.invoke("stop")
    time.sleep(1)
    print "*****Destroying*****"
    self.invoke("destroy")
    time.sleep(1)
    print "*****Creating*****"
    self.startup();
    time.sleep(1)
    print "*****Setup*****"
    self.setup()
    time.sleep(1)
    print "*****Run*****"
    self.run()

  def vehInterface(self, y, hostname, clientof):
    return StateSensor(geom="40x5-300+"+y, host=hostname, font='6x10',
                       preamble="setenv IPTMACHINE " + hostname + "\n",
                       xterm_options='-xrm "*Page:1 0 " -iconic',
                       clientof=clientof)

  def memManager(self, y, hostname, clientof):
    return MemManager(geom="40x5-0+"+y, host=hostname, font='6x10',
                      preamble="setenv IPTMACHINE " + hostname + "\n",
                      xterm_options='-xrm "*Page:1 0" -iconic')

  def addHost(self, hostname, clientof=None):
    if not hostname:
      return
    if self.hosts.has_key(hostname):
      return
    self.hosts[hostname] = clientof
    y = repr(self.num_hosts*100)
    self.manage(self.memManager(y, hostname, clientof),
                self.vehInterface(y, hostname, clientof))
    self.num_hosts = self.num_hosts+1

  def collect_run(self, collect_module):
    pass

  def collect_finish(self, collect_module):
    pass

  def registerClass(self, c):
    sys.modules[RepoMan.__module__].__dict__[c.__name__] = c

class Basic:
  def __init__(self, geom=None, name='Basic', font = None, xterm_options=None,
               dir=None, preamble=None, command=None, host=None, user=None,redirect_output=None):
    self.geom = geom
    self.name = name
    self.dir = dir
    self.preamble=preamble
    self.redirect_output=redirect_output
    if xterm_options:
      self.xterm_options=xterm_options
    else:
      self.xterm_options=''
    self.manager = None
    self.window = None
    if not command:
      self.command = './' + name
    else:
      self.command = command
    self.font = font
    self.status = 0
    self.host = socket.gethostbyname(host)
    self.memory_region_names = []
    self.restart_time = -1
    self.retries = 0
    self.user = user

  def startup(self):
    if not self.window:
      self.confirm()
      return 1
    else:
      return 0

  def stop(self):
    if not self.status:
      return
    self.manager.sendString(self.window, '\003')
    self.status = 2

  def finish(self):
    self.stop()

  def restart(self):
    print "Restarting", self.name, self.retries+1
    self.stop()
    # add a little sleep to the restart to prevent thrashing
    self.manager.sendString(self.window, '\nsleep 1\n')
    # try and catch disconnect events
    self.manager.server.processEvents(0.1)
    self.setup()
    self.run()
    self.retries = self.retries + 1

  def invoke(self, routine):
    try:
      apply(getattr(self.__class__, routine), [self])
      return 1
    except AttributeError:
      return 0

  def create(self):
    if self.confirm(0):
      self.status=1
      return
    
    if not self.name:
      raise RuntimeError, "Can't create unnamed window"
    if not self.geom:
      raise RuntimeError, "Window %s has no geometry" % self.name
    if self.font:
      font = ' -fn ' + self.font
    else:
      font = ''
    try:
      dest_host = socket.gethostbyname(self.host)
    except TypeError:
      print "****Warning:  No host for %s" % self.name
      dest_host = None
    if self.host and \
       (dest_host != "127.0.0.1") and \
       (dest_host != \
       socket.gethostbyname(socket.gethostname())):
      if self.user:
        exec_opt = ' -e ssh -4 -t %s@%s %s' % (self.user, self.host,
                                               self.manager.shell)
      else:
        exec_opt = ' -e ssh -4 -t %s %s' % (self.host, self.manager.shell)
    else:
      exec_opt = ' -e %s' % self.manager.shell
    self.window = self.manager.sender.create(self.name,
      '-geom ' + self.geom + font + ' ' + self.xterm_options + exec_opt, 0)
    self.status = 1

  def manage(self, mgr):
    self.manager = mgr

  def setup(self):
    if not self.status:
      return
    if not self.confirm():
      print "Could not confirm %s for setup" % self.name
      return
    if self.dir:
      self.manager.sendString(self.window, 'cd '+self.dir+'\n')
    if self.preamble:
      self.manager.sendString(self.window, self.preamble)
    self.status = 2

  def destroy(self):
    if not self.status:
      return
    if self.confirm(0):
      if self.window:
        self.manager.sendString(self.window, '\003\003\003\004\004\004')
        self.manager.sender.destroyWindow(self.window)
    self.status = 0

  def confirm(self, blocking=1):
    if blocking:
      self.window = self.manager.sender.waitForWindow(self.name)
      if self.window is None:
        print "******Problem starting up window %s******" % self.name
        print "******Fatal error*************"
        sys.exit();
    else:
      self.window = self.manager.sender.getWindow(self.name)

    if not self.window:
      self.status = 0

    return self.window

  def run(self):
    if not self.status:
      return

    self.first_report = 0
    self.manager.sendString(self.window, self.command+'\n')
    self.status = 3

class Module(Basic):
  def __init__(self, geom=None, name='Module', font = None, xterm_options='',
               dir=None, preamble=None, command=None, host=None, user=None,
               redirect_output=None, embed_command_in=None,stop_all=0):
    Basic.__init__(self, geom, name, font, xterm_options, dir, preamble,
                  command, host, user,redirect_output)
    self.embed_command_in = embed_command_in
    self.module_status = None
    self.start_run = 0
    self.stop_all = stop_all
  def setBool(self, tag, value):
    self.manager.server.set('bool %s.%s' % (self.name, tag), value)

  def manage(self, mgr):
    Basic.manage(self, mgr)

  def run(self):
    if not self.status:
      return

    self.first_report = 0

    self.setBool('running', 1)

    print 'struct %s' % self.name, \
      self.manager.server.getSubFile(self.name).output()

    if self.manager.logging:
      self.manager.sendString(self.window,
                                     'date >> %s.log\n' % self.name)

    ipt_spec = "{unix: module_name=%s; host_name=%s;}" \
               % (self.name, self.manager.host)
    cmd = \
    '%s "embedded: name=%s; spec container {iptclient: do_block_test=true; spec ipt_spec %s}"'\
        % (self.command, self.name, ipt_spec)
    if self.embed_command_in:
      cmd = self.embed_command_in % cmd
    if self.manager.logging:
      self.manager.sendString(self.window, '%s >> %s.log\n'
                                     % (cmd, self.name))
    else:
      self.manager.sendString(self.window, cmd+'\n')
                                     
    self.status = 3
    # add 5 to give things time to start up, i.e., by adding 5 seconds to start
    # run, then we make the monitoring system wait an extra 5 seconds before
    # nuking it.
    self.start_run = time.time() + 5

  def finish(self):
    self.status = 2
    self.start_run = 0
    self.setBool('running', 0)

  def pause(self):
    if self.status < 3:
      return
    self.setBool('paused', 1)

  def unpause(self):
    self.setBool('paused', 0)

class InitModule(Module):
  def run(self):
    self.setBool('running', 1)
    pass

  def setup(self):
    Module.setup(self)
    if not self.status:
      return
    Module.run(self)

  def finish(self):
    pass

class StateSensor(InitModule):
  memory_managed=0
  def __init__(self, geom=None, font=None, xterm_options='', \
               host=None, user=None, clientof=None, preamble=None):
    self.cfg_string = None
    if not host or not clientof:
      name = 'state_sense'
    else:
      name = 'state_sense.'+host
      name = string.join(string.split(name, '.'), '_')
      if StateSensor.memory_managed:
        self.cfg_string = 'spec state_sense_spec { remote: retry_times=3; retry_wait=0.25; host=%s; } spec full_state_out_spec {array:  spec current_spec { }  spec shmem_spec { managed: string name=VehStateArray; bool owner=true; } }' % clientof
      else:
        self.cfg_string = 'spec state_sense_spec { remote: retry_times=3; retry_wait=0.25; host=%s; }' % clientof
      
    Module.__init__(self, geom=geom, name=name, font=font, \
                    xterm_options=xterm_options,
                    command='state_sense',
                    host=host, user=user, preamble=preamble,redirect_output=redirect_output)

  def setup(self):
    if self.cfg_string:
      self.manager.server.parseSet('struct ' + self.name, self.cfg_string)
    InitModule.setup(self)

class InitBasic(Basic):
  def run(self):
    pass

  def finish(self):
    pass

  def setup(self):
    Basic.setup(self)
    if not self.status:
      return
    Basic.run(self)
    
class CollectInitModule(Module):
  def run(self):
    self.setBool('running', 1)
    pass

  def setup(self):
    Module.setup(self)
    if not self.status:
      return
    Module.run(self)
    self.crunch_files = \
                      self.manager.server.getStrings("Modules.%s.crunch_files"
                                                     % self.name)
    self.copy_files = \
                    self.manager.server.getStrings("Modules.%s.copy_files"
                                                   % self.name)
    self.manager.collect_run(self)
 
    
  def finish(self):
      self.manager.collect_finish(self)
      pass

class CollectModule(Module):
  def run(self):
    Module.run(self)
    self.crunch_files = \
                      self.manager.server.getStrings("Modules.%s.crunch_files"
                                                     % self.name)
    self.copy_files = \
                    self.manager.server.getStrings("Modules.%s.copy_files"
                                                   % self.name)
    self.manager.collect_run(self)

  def finish(self):
    Module.finish(self)
    self.manager.collect_finish(self)

class MemManager(InitBasic):
  def __init__(self, geom=None, font = None, xterm_options='', host=None,
               preamble=None):
    if host:
      name = 'iptshmgr.'+host
    else: 
      name = 'iptshmgr'

    InitBasic.__init__(self,geom=geom, font=font, xterm_options=xterm_options,\
                       host=host, name=name, command='iptshmgr',
                       preamble=preamble)

  def manage(self, mgr):
    InitBasic.manage(self, mgr)
    if not self.host:
      self.host = mgr.host
                         
if __name__=='__main__':
  mgr = RepoMan()

  mgr.initialize()
  mgr.startup()
  mgr.serve()
