/* -*-c++-*-
=========================================================================
=                                                                       =
=                                GLUT_WINDOW.H                          =
=                                                                       =
=========================================================================

This describes a basic glut based window base class.

Target: UNIX/Generic

Designed by:    Chris Urmson
Written by:     Chris Urmson
Date:           June 2002

Copyright 2002 Carnegie Mellon University
$Author: jayg $
$Date: 2004/04/26 19:58:43 $
$Header: /IUS/jeep/cvsroot/ModUtils/GLUtils/glut_window.h,v 1.1 2004/04/26 19:58:43 jayg Exp $
$Id: glut_window.h,v 1.1 2004/04/26 19:58:43 jayg Exp $
$Log: glut_window.h,v $
Revision 1.1  2004/04/26 19:58:43  jayg
First version of GLUtils ripped from Redteam

Revision 1.9  2003/11/17 05:18:11  curmson
Added some accessor functions to allow non descendant users to tweak mouse and
keyboard input.

Revision 1.8  2003/11/10 06:22:53  curmson
Changes that unify the 3D point data type

Revision 1.7  2003/11/09 19:48:36  mkj
fixed Fl GL init incompatiblities

Revision 1.6  2003/11/05 22:40:23  mkj
headerfile missing

Revision 1.5  2003/11/05 22:37:24  mkj
mightnot work yet

Revision 1.4  2003/10/24 14:51:01  wjh
Fred is an idiot

Revision 1.3  2003/10/24 14:35:35  wjh
glut_window.h: remove "Common/" from #include
MakeTail: move the location of added libraries kludge

Revision 1.2  2003/10/01 22:47:58  kp
PGMImage is updated to take const char* rather than just char* in order to
be compatible with module code

glut_window.h is changed to make display() virtual
glut_window.cc is not changed

Revision 1.1  2003/09/06 22:25:04  curmson
First commit of gl utils in this working directory structure.

Revision 1.1  2003/02/25 14:18:39  curmson
First commit of the new gl_utils library.  This code is somewhat cleaner than the old glHelpers code and is more powerful.

It supports new features including:
        multiple windows
        lighting
        object registration
        ability to set z up or z down orientation

Revision 1.6  2002/11/20 16:21:09  curmson
Added code to draw text at arbitrary locations in the 3d scene.

Revision 1.5  2002/09/17 13:45:17  curmson
Added the ability to register objects for drawing, thus elliminating the need to subclass glut window in some cases.

Revision 1.4  2002/07/24 18:40:21  curmson
Modified openGL camera positioning routines to render with the correct coordinate system orientations (i.e. X forward, Y to the right, Z down).

Revision 1.3  2002/07/01 21:59:26  curmson
Added more comments and restructured initialization to avoid some warning messages on certain systems.

Revision 1.2  2002/06/26 20:28:24  curmson
removed dependance on share_cmu

Revision 1.1  2002/06/26 01:01:24  curmson
This is a first commit of the util_open_gl module.  It provides a simplified interface to glut and the openGL, and simplifies the task of creating a quick debugging UI.



 ------------------------------------------------------------------------*/
#ifndef GLUT_WINDOW
#define GLUT_WINDOW

#include <GL/gl.h>
#include <GL/glut.h>

#include <GL/glu.h>
#include <list>
#include <GLUtils/gl_object.h>

#define GL_UP_KEY 101    // the up arrow key
#define GL_DOWN_KEY 103  // the down arrow key
#define GL_LEFT_KEY 100  // the left arrow key
#define GL_RIGHT_KEY 102 // the right arrow key
#define GL_PUP_KEY 104   // the page up key
#define GL_PDN_KEY 105   // the page down key
#define GL_HOME_KEY 106
#define GL_END_KEY 107
#define GL_MAX_GLUT_WINDOWS 256 // the maximum number of glut windows one process can start

/**
   This class wraps alot of the basic glut functionality, and provides some
   basic setup behavior.*/
class Glut_Window {
  /**
     which button is pressed*/
  int _button_down;
  /**
     used to deal with mouse movement*/
  int _mouse_start_x;
  /**
     used to deal with mouse movement*/
  int _mouse_start_y;
  /**
     used to deal with mouse movement*/
  int _mouse_start_view_D, _mouse_start_view_Az, _mouse_start_view_El;

  /**
     used to determine wether Z should be rendered up or down
     default is false (i.e. z down)
  */
  bool _z_up;

  /**
     variables used for windows management*/
  static Glut_Window * _windows[GL_MAX_GLUT_WINDOWS];
  static int _num_windows;

//    static Glut_Window * _window;    
protected:


  /** This is the unique glut window ID */
  int window_ID; 
  /** this is the width and height of the window*/
  int width, height;

  /**
     the elevation of the camera*/
  double view_elevation;
  /**
     the azimuth of the camera */
  double view_azimuth;
  /** 
      the distance the camera is from the center of rotation*/
  double view_distance;
  /**
     the location of the center of rotation of the camera*/
  double view_x;
  /**
     the location of the center of rotation of the camera*/
  double view_y;
  /**
     the location of the center of rotation of the camera*/
  double view_z; 
  /**
     this is the distance the near clipping plane is from the camera, 
     it defaults to 0.5 units*/
  double clip_near;
  /**
     this is the distance the far clipping plane is from the camera, 
     it defaults to 500.0 units*/
  double clip_far;
  /**
     this is the angular field of view of the camera, it defaults to 30 deg
     it is specified in DEGREES
  */
  double fov_y;
  
  /**
     This is the distance the center of the view moves when an arrow key is 
     pressed.  It defaults to 0.05 units
   */
  double step_size;

  /**
     This is a multiplier used to set the rate at which mouse motion zooms the 
     view in and out*/
  double zoom_scale;

  /**
     These are static member functions that provide an interface to the glut 
     callbacks.  These call the appropriate virtual function call */
  static void _display(void);
  static void _reshape(int w,int h);
  static void _mouse(int button, int state, int x, int y);
  static void _key(unsigned char key,int x, int y);
  static void _specialKey(int key,int x, int y);
  static void _mouseMotion(int x, int y);
  static void _timer(int value);

  /**
     This contains a list of objects that have been registered with the window
     to be drawn*/
  list<GL_Object *> _reg_objects;

  /**
     This function draws all of the registered objects*/
  void _draw_registered(void);
public:
  static bool _initialized;
  /**
     This creates a new window and sets up the handlers etc. 
     if useTimer is set, a timer is also setup.*/
  Glut_Window(char *title, int width, int height);

  virtual ~Glut_Window();
  /**
     any process specific initialization can be performed here.  This is the last thing called in the GlutWindow constructor */
  virtual void init(void) {};

  /**
     This returns the width of the glut window*/
  int get_width(void) {return width;};

  /**
     This returns the height of the glut window*/
  int get_height(void) {return height;};

  int get_button(void) {return _button_down;};

  /**
     The program will not return from this function call- all windows created so far will start executing
  */
  void run(void) { glutMainLoop();};

  /**
     This will setup the camera (call positionCamera()), draw all of the registered object (call _draw_registered() ) and then call draw()
  */
  virtual void display(void);

  /**
     This tells the system that we want this window to be redrawn*/
  void post_redisplay(void) {
    int temp_ID = glutGetWindow();
    glutSetWindow(window_ID); 
    glutPostRedisplay();
    glutSetWindow(temp_ID);
  };

  /** 
      This will cause the timer_func function to be called with the given timerId after the given number of milliseconds.  Multiple timers can be set, providing they utilize different timer ids.  timer id should be an unsigned int in the range of 0 to 255.*/
  void register_timer(unsigned int delay, unsigned char timer_ID);

  /**
     This draws text in the scene, in the current pixel frame*/ 
  void draw_text_in_scene(char *txt, float r=0.5, float g=0.5, float b=0.5,
                          void *font = GLUT_BITMAP_HELVETICA_10);

  /**
     This will draw the given string to the screen at the location given by
     x,y- (0,0 is the bottom left of the window, 1,1 is the top right).
     the color defaults to a gray
     font can be one of the GLUT_BITMAP_*
  */
  void draw_text(char *txt,float x,float y, float r=0.5,float g=0.5, float b=0.5, void *font=GLUT_BITMAP_HELVETICA_10);

  /**
     These functions toggle wether +ve z points up or down.
   */
  void set_z_up(void) {_z_up=true;};
  void set_z_down(void) {_z_up=false;};
  bool zIsUp(void) {return _z_up;};

  /** 
      This adjusts the step made by the arrow keys*/
  void set_step_size(double ss) {
    step_size = ss;
  };
  
  /**
     This sets the distance to the far clipping plane*/
  void set_far_clip(double farClip) {
    clip_far = farClip;
  };
  
  /**
     This sets the distance to the near clipping plane*/
  void set_near_clip(double nearClip) {
    clip_near = nearClip;
  }

  /**
     This sets the rate of zoom when the mouse is used to adjust the
  viewing distance*/
  void set_zoom_scale(double zs) {
    zoom_scale = zs;
  }

  /**
     These hide and show the window*/
  void hide(void);
  void show(void);
  /**
     This moves the window around the screen*/
  void position_window(int x,int y);
  /**
     This changes the title of the window */
  void set_title(char * title);

  /**
     This adds an object to the list of objects that will be drawn in each
     display pass.*/
  void register_object(GL_Object * obj);
  /**
     This removes every instance of the given object from the display list */
  void unregister_object(GL_Object *obj);


  /**
     override this function to draw objects */
  virtual void draw(void) {};
  /**
     This performs generic matrix setup operations, you shouldn't need
     to override this unless you want to render something without perspective
     @see fov_y, near_clip, far_clip
  */
  virtual void reshape(int w,int h);
  /**
     This is used in conjunction with mouse_motion to provide the camera 
     movement.  Moving the mouse with the left button down rotates the camera 
     around a fixed point (specified by view_x, view_y, view_z).  Moving the 
     mouse with the right button down zooms the camera in and out

     If you would like to perform other operations with the mouse, override
     this function, you may want to consider calling Glut_Window::mouse() 
     in that routine to maintain the mouse interface.
     @see special_key, position_camera
  */
  virtual void mouse(int button, int state, int x, int y);

  /**
     This routine is called whenever a mouse movement event has occured.
     If you are overriding it, consider calling Glut_Window::mouse_motion()
     in your descendant classes, to maintain the mouse interface
     @see mouse
  */
  virtual void mouse_motion(int x, int y);

  /**
     Override this function to cause action something to occur when a key
     is pressed
  */
  virtual void key(unsigned char key,int x, int y);

  /**
     This function is used to perform actions when one of the non-ascii keys
     are presed.  In the base class, the arrow keys are used to move the 
     center of the field of view around in the plane, while PAGE_UP & 
     PAGE_DOWN are used to change the Z value of the center of view
     @see mouse, position_camera
  */
  virtual void special_key(int key,int x, int y);
  
  /**
     Overide this function to change the location from which the scene is 
     rendered.  It currently uses information from the mouse and the special 
     keyboard keys to set its direction
     @see mouse, position_camera
  */
  virtual void position_camera(void);

  /**
     Override this function to have something occur after a set amount of time.
     Multiple timers can be set, when the timer "goes off" the timer_ID of that
     timer is passed as the argument to timer_func
     @see register_timer
  */
  virtual void timer_func(unsigned char timer_ID) {}; 
};


class Glut_Lit_Window: public Glut_Window {

protected:
  GLfloat _def_specular[4];
  GLfloat _def_shininess[1];
  GLfloat _def_light_pos[4];
  GLfloat _white_light[4];

public:
  /** 
      This performs the the same function as Glut_Window's version, but it 
      also sets up lighting*/
  virtual void position_camera(void);

  Glut_Lit_Window(char *title, int width, int height);


};

#endif





