/* -*-c++-*-
=========================================================================
=                                                                       =
                               GL_OBJECT_.H                            
=                                                                       =
=========================================================================

Contains basic gl drawable object classes.
 
Target: VxWorks/UNIX/Generic

Designed by:    Chris Urmson
Written by:     Chris Urmson
Date:           June 2002

(C)  2002 Carnegie Mellon University
$Author: jayg $
$Date: 2004/04/26 19:58:43 $
$Header: /IUS/jeep/cvsroot/ModUtils/GLUtils/gl_object.h,v 1.1 2004/04/26 19:58:43 jayg Exp $
$Id: gl_object.h,v 1.1 2004/04/26 19:58:43 jayg Exp $
$Log: gl_object.h,v $
Revision 1.1  2004/04/26 19:58:43  jayg
First version of GLUtils ripped from Redteam

Revision 1.5  2003/11/28 05:51:14  curmson
Added code to draw lozenges (i.e. waypoint corridors) and added a couple of macros to simplify using r3Vector with GL code.

Revision 1.4  2003/11/17 05:19:36  curmson
Added a number of objects for rendering texture mapped stuff.

Revision 1.3  2003/11/09 19:48:36  mkj
fixed Fl GL init incompatiblities

Revision 1.2  2003/11/09 04:23:09  curmson
Made destructor virtual

Revision 1.1  2003/09/06 22:25:04  curmson
First commit of gl utils in this working directory structure.

Revision 1.2  2003/03/26 19:12:22  curmson
fixed an unitialized variable in gl_object

Revision 1.1  2003/02/25 14:18:39  curmson
First commit of the new gl_utils library.  This code is somewhat cleaner than the old glHelpers code and is more powerful.

It supports new features including:
        multiple windows
        lighting
        object registration
        ability to set z up or z down orientation

Revision 1.4  2002/11/20 16:21:07  curmson
Added code to draw text at arbitrary locations in the 3d scene.

Revision 1.3  2002/10/14 19:05:09  curmson
Minor changes:
 added an init check in glut_window and added start and stop list calls to gl_display_list_object

Revision 1.2  2002/07/01 21:59:24  curmson
Added more comments and restructured initialization to avoid some warning messages on certain systems.

Revision 1.1  2002/06/26 01:01:19  curmson
This is a first commit of the util_open_gl module.  It provides a simplified interface to glut and the openGL, and simplifies the task of creating a quick debugging UI.



-------------------------------------------------------------------------*/



#ifndef GL_OBJECT
#define GL_OBJECT
#include <GL/gl.h>
#include <GL/glut.h>
#include <utils/Linear.h>

typedef utils::Vec3d r3Vector;
typedef utils::Vec3d SensorPoint3D;
typedef r3Vector POSITION_TYPE, *POSITION_PTR;

#define PI M_PI
#define RAD_TO_DEG(r)   ((r)*180.0 / PI)
#define DEG_TO_RAD(d)   ((d)*PI / 180.0)

/**
 * @defgroup gl_objects GL_Objects
 * These are generally useful gl drawing objects
 *
 */

/** convenience defines for using r3Vector types with GL*/
#define glVertexR3(a) glVertex3d((a).x,(a).y,(a).z)
#define glTranslateR3(a) glTranslatef((a).x,(a).y,(a).z);




/**
   this is a pure virtual base class that can be overridden to contain whatever
   3D data is required
   @ingroup gl_objects
*/
class GL_Object {
  POSITION_TYPE _location;
  bool _should_draw;
public:
  GL_Object(void) { _should_draw=true; _location.x = _location.y = _location.z = 0;};
  virtual ~GL_Object() {};
  /**
     This moves the drawing frame to the location of the object and then calls 
     the draw_object member function*/
  void draw(void);

  /**
     This actually draws the given object
     This is the function that should be overridden by descendant classes
  */
  virtual void draw_object(void)=0;
  /**
     This sets the position of the object*/
  void set_position(POSITION_TYPE loc) {_location= loc;};
  void set_position(float x, float y, float z) {
    _location.x=x;
    _location.y = y;
    _location.z = z;};

  /**
     This moves posit of the object*/
  void offset(POSITION_TYPE offset) {
    _location+offset;};
  void offset(float dx, float dy, float dz) {
    _location.x+=dx;
    _location.y+=dy;
    _location.z+=dz;};
  /**
     This returns the current position of the object*/
  POSITION_TYPE get_position(void) {return _location;};

  /**
     This causes the object to not be drawn*/
  void hide(void) {_should_draw=false;};
  /**
     This causes the object to be drawn, this is the default condition*/
  void show(void) {_should_draw=true;};

  bool is_visible(void) {return _should_draw;};

  /**
     This draws text at the current origin in the scene.*/
  void draw_text(char *txt, float r=0.5,float g=0.5,float b =0.5,
                 void *font = GLUT_BITMAP_HELVETICA_10) {
    glColor3f(r,g,b);
    glRasterPos3f(0,0,0);
    char *ch;
    for(ch= txt; *ch; ch++) {
      glutBitmapCharacter(font, (int)*ch);
    }
  };

};




/**
   This class can be used as a base class for rendering complicated objects.
   Complex shapes can be stored by calling the drawing routine between glNewList(_display_list,GL_COMPILE); and glEndList(); calls.
   @ingroup gl_objects
*/
class GL_Display_List_Object: public GL_Object {
protected:
  int _display_list;
public:
  GL_Display_List_Object(void);
  void start_list(void) {glNewList(_display_list,GL_COMPILE);};
  void end_list(void) {glEndList();};
  virtual void draw_object(void);
};

/**
   A simple RGB origin marker
   A red,green and blue correspond to X,Y,Z axes.
   @ingroup gl_objects
*/
class GL_Origin : public GL_Object{
  float _length;
public:
  /**
     creates a new drawable origin, the length of the axies will be
     length units*/
  GL_Origin(float length=1.0) {this->_length= length;};
  void draw_object(void);
};


/**
   This class draws a hash mark of the given color
   @ingroup gl_objects
*/
class GL_Hash: public GL_Object {
  float _size,_s2;
  float _R,_G,_B;
public:
/**
   This creates a new hashmark.
   @param size the length of the hash
   @param R the intensity of red used to render the hash (0-1.0)
   @param G the intensity of green used to render the hash (0-1.0)
   @param B the intensity of blue used to render the hash (0-1.0)
   
*/    GL_Hash(float size, float R, float G, float B);
  void draw_object(void);
};

/**
   This draws a regular grid that can be used to represent a ground plane.
   @ingroup gl_objects
*/
class GL_Ground_Plane: public GL_Object {
  float _R,_G,_B;
  float _lineSpacing;
  int _numCells;
public:
/**
   This creates the ground plane with the given parameters
   @param line_spacing the distance between grid lines
   @param num_cells the number of cells in each dimension
   @param R the intensity of red used to render the grid (0-1.0)
   @param G the intensity of green used to render the grid (0-1.0)
   @param B the intensity of blue used to render the grid (0-1.0)
*/
  GL_Ground_Plane(float line_spacing,int num_cells,
                  float R, float G, float B);
  void draw_object(void);
};

#endif





