#ifndef GLUTDISPLAY_H
#define GLUTDISPLAY_H

#include <stdio.h>
#include <math.h>


#include <deque>
#include <list>
#include <FL/Fl.h>
#include <FL/gl.h>
#include <FL/glut.H>
#include <GL/glu.h>
#include <GLUtils/gl_object.h>
#define GL_MAX_GLUT_WINDOWS 256 // the maximum number of glut windows one process can start
class GlutDisplay
{

 private:

  /**
     This contains a list of objects that have been registered with the window
     to be drawn*/
    list<GL_Object *> _reg_objects;

  /**
     This function draws all of the registered objects*/
  void _draw_registered(void);

  // functions that do what the callbacks are supposed to do
  virtual void display();
  void reshape(int, int);
  void menu(int);
  virtual void idle();
  void mousedrag(int, int);
  void mouseidle(int, int);
  void mousebutton(int, int, int, int);
  void keyboard(unsigned char, int, int);

  // initializers
  void initStateVars();
  void assignCallbacks();
  void setKeyCallback();
  void menuInit();
  void glInit();

  // de-initializers
  void deleteStateVars();
  
  // movie functions
  void dumpFrameBuffer(char*);     // writes the frame buffer to a file "fname"

  // drawing functions
  void drawHeightPoints();
  void drawTriangle(float, float, float);


  // static vars for callbacks
  static GlutDisplay* PBD;

  // FLTK frame to draw into
  Fl_Window *frame;
  /**
     which button is pressed*/
  int _button_down;
  /**
     used to deal with mouse movement*/
  int _mouse_start_x;
  /**
     used to deal with mouse movement*/
  int _mouse_start_y;
  /**
     used to deal with mouse movement*/
  int _mouse_start_view_D, _mouse_start_view_Az, _mouse_start_view_El;

  /**
     used to determine wether Z should be rendered up or down
     default is false (i.e. z down)
  */
  bool _z_up;
  protected:

  bool should_draw_in_idle;


  /** This is the unique glut window ID */
  int window_ID; 
  /** this is the width and height of the window*/
  int width, height;

  /**
     the elevation of the camera*/
  double view_elevation;
  /**
     the azimuth of the camera */
  double view_azimuth;
  /** 
      the distance the camera is from the center of rotation*/
  double view_distance;
  /**
     the location of the center of rotation of the camera*/
  double view_x;
  /**
     the location of the center of rotation of the camera*/
  double view_y;
  /**
     the location of the center of rotation of the camera*/
  double view_z; 
  /**
     this is the distance the near clipping plane is from the camera, 
     it defaults to 0.5 units*/
  double clip_near;
  /**
     this is the distance the far clipping plane is from the camera, 
     it defaults to 500.0 units*/
  double clip_far;
  /**
     this is the angular field of view of the camera, it defaults to 30 deg
     it is specified in DEGREES
  */
  double fov_y;
  
  /**
     This is the distance the center of the view moves when an arrow key is 
     pressed.  It defaults to 0.05 units
   */
  double step_size;

  /**
     This is a multiplier used to set the rate at which mouse motion zooms the 
     view in and out*/
  double zoom_scale;

 public:
   static void _display(void);
  static void _reshape(int w,int h);
  static void _mouse(int button, int state, int x, int y);
  static void _key(unsigned char key,int x, int y);
  static void _specialKey(int key,int x, int y);
  static void _mouseMotion(int x, int y);
  static void _timer(int value);
  virtual void init(void) {};
static void  _idle() ;
  /**
     This draws text in the scene, in the current pixel frame*/ 
  void draw_text_in_scene(char *txt, float r=0.5, float g=0.5, float b=0.5,
                          void *font = GLUT_BITMAP_HELVETICA_10);

  /**
     This will draw the given string to the screen at the location given by
     x,y- (0,0 is the bottom left of the window, 1,1 is the top right).
     the color defaults to a gray
     font can be one of the GLUT_BITMAP_*
  */
  void draw_text(char *txt,float x,float y, float r=0.5,float g=0.5, float b=0.5, void *font=GLUT_BITMAP_HELVETICA_10);


 /**
     This returns the width of the glut window*/
  int get_width(void) {return width;};

  /**
     This returns the height of the glut window*/
  int get_height(void) {return height;};

  int get_button(void) {return _button_down;};

 /**
     This adds an object to the list of objects that will be drawn in each
     display pass.*/
  void register_object(GL_Object * obj);
  /**
     This removes every instance of the given object from the display list */
  void unregister_object(GL_Object *obj);

  //    virtual void display(void);

  /**
     This tells the system that we want this window to be redrawn*/
  void post_redisplay(void) {
    int temp_ID = glutGetWindow();
    glutSetWindow(window_ID); 
    glutPostRedisplay();
    glutSetWindow(temp_ID);
  };

  /**
     override this function to draw objects */
  virtual void draw(void) {};
  /**
     This performs generic matrix setup operations, you shouldn't need
     to override this unless you want to render something without perspective
     @see fov_y, near_clip, far_clip
  */
  //  virtual void reshape(int w,int h);
  /**
     This is used in conjunction with mouse_motion to provide the camera 
     movement.  Moving the mouse with the left button down rotates the camera 
     around a fixed point (specified by view_x, view_y, view_z).  Moving the 
     mouse with the right button down zooms the camera in and out

     If you would like to perform other operations with the mouse, override
     this function, you may want to consider calling Glut_Window::mouse() 
     in that routine to maintain the mouse interface.
     @see special_key, position_camera
  */
  virtual void mouse(int button, int state, int x, int y);

  /**
     This routine is called whenever a mouse movement event has occured.
     If you are overriding it, consider calling Glut_Window::mouse_motion()
     in your descendant classes, to maintain the mouse interface
     @see mouse
  */
  virtual void mouse_motion(int x, int y);

  /**
     Override this function to cause action something to occur when a key
     is pressed
  */
  virtual void key(unsigned char key,int x, int y);

  /**
     This function is used to perform actions when one of the non-ascii keys
     are presed.  In the base class, the arrow keys are used to move the 
     center of the field of view around in the plane, while PAGE_UP & 
     PAGE_DOWN are used to change the Z value of the center of view
     @see mouse, position_camera
  */
  virtual void special_key(int key,int x, int y);
  
  /**
     Overide this function to change the location from which the scene is 
     rendered.  It currently uses information from the mouse and the special 
     keyboard keys to set its direction
     @see mouse, position_camera
  */
  virtual void position_camera(void);
  
  void set_draw_in_idle(void) {should_draw_in_idle=true;};
  void unset_draw_in_idle(void) {should_draw_in_idle=false;};

  bool _initialized;

  /**
     These functions toggle wether +ve z points up or down.
  */
  void set_z_up(void) {_z_up=true;};
  void set_z_down(void) {_z_up=false;};
  bool zIsUp(void) {return _z_up;};

 /** 
      This adjusts the step made by the arrow keys*/
  void set_step_size(double ss) {
    step_size = ss;
  };
  
  /**
     This sets the distance to the far clipping plane*/
  void set_far_clip(double farClip) {
    clip_far = farClip;
  };
  
  /**
     This sets the distance to the near clipping plane*/
  void set_near_clip(double nearClip) {
    clip_near = nearClip;
  }

  /**
     This sets the rate of zoom when the mouse is used to adjust the
  viewing distance*/
  void set_zoom_scale(double zs) {
    zoom_scale = zs;
  }

  // change the location where we're viewing from
  void set_view(double x, double y, double z) {
    view_x = x; view_y = y; view_z = z;
  }

  void get_view(double *x, double *y, double *z) {
    *x = view_x; *y = view_y; *z = view_z;
  }

  void set_view_angle(double azimuth, double elevation) {
    view_azimuth = azimuth; view_elevation = elevation;
  }
  void get_view_angle(double* azimuth, double* elevation) {
    *azimuth = view_azimuth; *elevation = view_elevation;
  }


  GlutDisplay();
  virtual ~GlutDisplay();
  GL_Ground_Plane *gp;
  void setCloud(std::deque<SensorPoint3D>* points);
};
class GlutLitDisplay: public GlutDisplay {

protected:
  GLfloat _def_specular[4];
  GLfloat _def_shininess[1];
  GLfloat _def_light_pos[4];
  GLfloat _white_light[4];

public:
  /** 
      This performs the the same function as Glut_Window's version, but it 
      also sets up lighting*/
  virtual void position_camera(void);

 GlutLitDisplay ();


};

#endif
