#include "GlutDisplay.h"
#include <GLUtils/gl_object.h>
GlutDisplay *GlutDisplay::PBD = 0;


void GlutDisplay::_draw_registered(void) {
  list<GL_Object *>::iterator iter;
  for (iter = _reg_objects.begin();iter!=_reg_objects.end();iter++) {
    (*iter)->draw();
  }
}
/**
     This adds an object to the list of objects that will be drawn in each
     display pass.*/
void GlutDisplay::register_object(GL_Object * obj) {
  _reg_objects.push_back(obj);
}


/**
   This removes every instance of the given object from the display list */
void GlutDisplay::unregister_object(GL_Object *obj) {
  _reg_objects.remove(obj);

}
/**
   Overide this function to change the location from which the scene is 
   rendered.  It currently uses information from the mouse and the special 
   keyboard keys to set its direction
   @see mouse, position_camera
*/
void GlutDisplay::position_camera(void) {
  if (!_z_up) {
    /* z down- default*/
    glTranslated(0.0,0.0,-view_distance);
    glRotated(view_elevation-90,1.0,0.0,0.0);
    glRotated(view_azimuth+90,0.0,0.0,1.0);
    glTranslated(-view_x,view_y,view_z);
    glRotated(180,1,0,0);
  } else {
    /* z up */
    glTranslated(0.0,0.0,-view_distance);
    glRotated(-view_elevation,1.0,0.0,0.0);
    glRotated(view_azimuth,0.0,0.0,1.0);
    glTranslated(-view_x,-view_y,-view_z);
  }
}



/**
   This performs generic matrix setup operations, you shouldn't need
   to override this unless you want to render something without perspective
   @see fov_y, near_clip, far_clip
*/
void GlutDisplay::reshape(int w,int h) {
  glViewport(0,0,(GLsizei)w,(GLsizei)h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(fov_y,(GLfloat)w/(GLfloat)h,clip_near,clip_far);
  this->width = w;
  this->height =h;
}

/**
   This is used in conjunction with mouse_motion to provide the camera 
   movement.  Moving the mouse with the left button down rotates the camera 
   around a fixed point (specified by view_x, view_y, view_z).  Moving the 
   mouse with the right button down zooms the camera in and out
   
   If you would like to perform other operations with the mouse, override
   this function, you may want to consider calling GlutDisplay::mouse() 
   in that routine to maintain the mouse interface.
   @see special_key, position_camera
*/
void GlutDisplay::mouse(int button, int state, int x, int y) {
  if (state==GLUT_UP) {
    _button_down =-1;
  } else {
    _button_down = button;
    _mouse_start_x =x;
    _mouse_start_y =y;
    //    printf("button down is :%d\n ", button);
    _mouse_start_view_D = (int)view_distance;
    _mouse_start_view_Az = (int)view_azimuth;
    _mouse_start_view_El = (int)view_elevation;
  }
}


/**
   Override this function to cause action something to occur when a key
   is pressed
*/
void GlutDisplay::key(unsigned char key,int x, int y) {
}


/**
   This routine is called whenever a mouse movement event has occured.
   If you are overriding it, consider calling GlutDisplay::mouse_motion()
   in your descendant classes, to maintain the mouse interface
   @see mouse
*/
void GlutDisplay::mouse_motion(int x, int y) {
  switch (_button_down) {
  case -1:
    return;
  case 2:
    view_distance = _mouse_start_view_D + (y-_mouse_start_y)*zoom_scale/5.0;
    break;
  case 0:
    view_azimuth = _mouse_start_view_Az +(_mouse_start_x-x)/5.0;
    if (!_z_up)
      view_elevation = _mouse_start_view_El + (y-_mouse_start_y)/5.0;
    else 
      view_elevation = _mouse_start_view_El - (y-_mouse_start_y)/5.0;
    if (view_elevation > 90) {
      view_elevation = 90;
    } else if (view_elevation < 0) {
      view_elevation = 0;
    }
    if (view_azimuth>360) {
      view_azimuth-=360;
    } else if (view_azimuth<-360) {
      view_azimuth+=360;
    }
    break;
        
  }
  glutPostRedisplay();
}


void GlutDisplay::draw_text_in_scene(char *txt, float r,float g, 
                                     float b,
                                     void *font) {
  glColor3f(r,g,b);
  glRasterPos3f(0,0,0);
  char *ch;
  for(ch= txt; *ch; ch++) {
    glutBitmapCharacter(font, (int)*ch);
  }
}

/**
   This will draw the given string to the screen at the location given by
   x,y- (0,0 is the bottom left of the window, 1,1 is the top right).
   the color defaults to a gray*/
void GlutDisplay::draw_text(char *txt,float x,float y,
                            float r,float g, float b, 
                            void *font) {
  GLint matrixMode;
  GLboolean lightingOn; 
  char *ch;
  GLenum error;
  lightingOn= glIsEnabled(GL_LIGHTING);        /* lighting on? */
  if (lightingOn) glDisable(GL_LIGHTING);
    
  glGetIntegerv(GL_MATRIX_MODE, &matrixMode);  /* matrix mode? */
    
  glMatrixMode(GL_PROJECTION);

  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0.0, 1.0, 0.0, 1.0);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glPushAttrib(GL_COLOR_BUFFER_BIT);       /* save current colour */
  glColor3f(r, g, b);
  glRasterPos3f(x, y, 0);
  for(ch= txt; *ch; ch++) {
    glutBitmapCharacter(font, (int)*ch);
  }
  glPopAttrib();
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(matrixMode);
  if (lightingOn) glEnable(GL_LIGHTING);
  error = glGetError();
  if (error !=GL_NO_ERROR)
    cerr << gluErrorString(error) << endl;
}


/*
  Callback hooks 
  These don't get overridden, they are static members that call the 
  appropriate virtual member functions
*/

void GlutDisplay::_display(void) {
  // id is the window that the display callback is being requested for

 
  GlutDisplay::PBD->display();
}

void GlutDisplay::_reshape(int w,int h) {

 GlutDisplay::PBD->reshape(w,h);
}

void GlutDisplay::_mouse(int button, int state, int x, int y) {
 
 GlutDisplay::PBD->mouse(button,state,x,y);
}

void GlutDisplay::_key(unsigned char key,int x, int y) {
 
 GlutDisplay::PBD->key(key,x,y);
}

void GlutDisplay::_specialKey(int key,int x, int y) {

 GlutDisplay::PBD->special_key(key,x,y);
}

void GlutDisplay::_mouseMotion(int x, int y) {

 GlutDisplay::PBD->mouse_motion(x,y);
}

void GlutDisplay::_idle() {

 GlutDisplay::PBD->idle();
}


// animates the view... not used
void GlutDisplay::idle()
{
  if (should_draw_in_idle)
    glutPostRedisplay();
}






void GlutDisplay::initStateVars()
{
  _z_up = false;
  view_elevation= 45;
  view_azimuth = 0;
  view_distance = 50;
  view_x=view_y=view_z=0;
  this->width = width;
  this->height= height;
}


void GlutDisplay::display(void) {
  glMatrixMode(GL_MODELVIEW);    
  glLoadIdentity();
  position_camera();
  glClear(GL_COLOR_BUFFER_BIT |GL_DEPTH_BUFFER_BIT);    
  glPushMatrix();
  _draw_registered();
  draw();
  glPopMatrix();
  glFlush();
  glutSwapBuffers();
}
GlutDisplay::GlutDisplay()
{


  frame = new Fl_Window(640, 480);
  PBD = this;

  initStateVars();

  // show the window
  frame->show();

  // attach the glut window to the fltk window
  frame->begin();

  // setup glut window
  glutInitWindowSize(frame->w(), frame->h());
  glutInitWindowPosition(0,0);

  glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH);

  glutCreateWindow("my window");

  // done attaching glut window to the fltk window
  frame->end();
//  frame->size_range(200,200,2000,2000,500,500,500);
  frame->resizable(frame);

  assignCallbacks();

  step_size=0.05;
  zoom_scale=1.0;
  _initialized=true;

  clip_near =0.5;
  clip_far = 500.0;
  fov_y = 30.0;
  should_draw_in_idle=false;
  // setup gl view
   glInit();
  
}

GlutDisplay::~GlutDisplay()
{
  delete frame;
 
}

void GlutDisplay::assignCallbacks()
{
 glutDisplayFunc(_display);
  glutReshapeFunc(_reshape);
  glutMouseFunc(_mouse);
  glutKeyboardFunc(_key);
  glutSpecialFunc(_specialKey);
  glutMotionFunc(_mouseMotion);
  glutIdleFunc(_idle);
}



void GlutDisplay::glInit()
{
  /////// setup gl view here ////////
  glShadeModel(GL_SMOOTH);                // mix colors smoothly between points

  /////// setup viewport size ///////
  glClearColor(0.0, 0.0, 0.0, 0.0);       // background is black
  reshape(frame->w(), frame->h());

  // enable the depth buffer
  glEnable(GL_DEPTH_TEST);
}
/**
   This function is used to perform actions when one of the non-ascii keys
   are presed.  In the base class, the arrow keys are used to move the 
   center of the field of view around in the plane, while PAGE_UP & 
   PAGE_DOWN are used to change the Z value of the center of view
   @see mouse, position_camera
*/
void GlutDisplay::special_key(int key,int x, int y) {
  if (_z_up) {
    switch (key) {
    case FL_Up:
      view_x+=sin(DEG_TO_RAD(view_azimuth))*step_size;
      view_y+=cos(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case FL_Down:
      view_x-=sin(DEG_TO_RAD(view_azimuth))*step_size;
      view_y-=cos(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case FL_Left:
      view_x-=cos(DEG_TO_RAD(view_azimuth))*step_size;
      view_y+=sin(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case FL_Right:
      view_x+=cos(DEG_TO_RAD(view_azimuth))*step_size;
      view_y-=sin(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case FL_Page_Up:
      view_z+=step_size;
    break;
    case FL_Page_Down:
      view_z-=step_size;
      break;
    default:
      cout << "key is: " << (int) key << endl;
    }
  } else { // !_z_up
    switch(key) {
    case FL_Up:
      view_y+=sin(DEG_TO_RAD(view_azimuth))*step_size;
      view_x+=cos(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case FL_Down:
        view_y-=sin(DEG_TO_RAD(view_azimuth))*step_size;
      view_x-=cos(DEG_TO_RAD(view_azimuth))*step_size;
      break;
   
    case FL_Left:

      view_y-=cos(DEG_TO_RAD(view_azimuth))*step_size;
      view_x+=sin(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case FL_Right:
      view_y+=cos(DEG_TO_RAD(view_azimuth))*step_size;
      view_x-=sin(DEG_TO_RAD(view_azimuth))*step_size;
      break;
    case FL_Page_Up:
      view_z-=step_size;
      break;
    case FL_Page_Down:
      view_z+=step_size;
      break;
    default:
      cout << "key is: " << (int) key << endl;
    }
  }
  glutPostRedisplay();
}
GlutLitDisplay::GlutLitDisplay(): GlutDisplay() {
  /* _def_specular[0]=1.0;
  _def_specular[1]=1.0;
  _def_specular[2]=1.0;
  _def_specular[3]=1.0;

  _def_shininess[0] = 50.0;

  _def_light_pos[0] = 0;
  _def_light_pos[1] = 0;
  _def_light_pos[2] = 1;
  _def_light_pos[3] = 0;
  
  _white_light[0] = 1.0;
  _white_light[1] = 1.0;
  _white_light[2] = 1.0;
  _white_light[3] = 1.0;
  
  GLfloat lmodel_ambient[] = {0.0,0.0,0.0,0.0};
  glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,_def_specular);
  glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,_def_shininess);
  glLightfv(GL_LIGHT0,GL_POSITION,_def_light_pos);
  glLightfv(GL_LIGHT0,GL_DIFFUSE,_white_light);
  glLightfv(GL_LIGHT0,GL_SPECULAR,_white_light);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glEnable(GL_COLOR_MATERIAL);
  glColorMaterial(GL_FRONT_AND_BACK,GL_DIFFUSE);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);*/
};

void GlutLitDisplay::position_camera(void) {
  /* glMaterialfv(GL_FRONT,GL_SPECULAR,_def_specular);
  glMaterialfv(GL_FRONT,GL_SHININESS,_def_shininess);
  glLightfv(GL_LIGHT0,GL_POSITION,_def_light_pos);
  glLightfv(GL_LIGHT0,GL_DIFFUSE,_white_light);
  glLightfv(GL_LIGHT0,GL_SPECULAR,_white_light);*/
  //GlutDisplay::position_camera();
};
