#ifndef REAL_DISPLAY_H
#define REAL_DISPLAY_H

#include <utils/SymbolTable.h>
#include <utils/Interface.h>
#include <list>
#include <FL/Fl.H>
#include <Fl/Fl_Gl_Window.H>
#include <GL/gl.h>

/*
 * Actual display window
 */
class DisplayCallback;

class Display : public Fl_Gl_Window
{
 public:
        /*
         * Create a display from a spec
         */
        Display(const char *spec,
                utils::SymbolTable *globals);

        /*
         * Create a display manually
         */
        Display(int w, int h,
                const char *name,
                bool track_vehicle_pos = false,/* Does the display track the vehicle position? */
                bool track_vehicle_rot = false,/* Does the display track the vehicle rotation? */
                bool enable_mouse_move = true, /* Can the user move the display with the mouse? */
                bool enable_zoom = true);      /* Can the user zoom in/out with the mouse?      */

        /* constructor used with fluid in conjunction with setup */
        Display(int X, int Y, int W, int H, const char *l=0);

        virtual ~Display();

        void initialize(utils::SymbolTable* globals, const char* name,
                        bool track_vehicle_pos = false,
                        bool track_vehicle_rot = false,
                        bool enable_mouse_move = true, 
                        bool enable_zoom = true);      
        
        /*
         * FLTK resize override
         */
        virtual void resize(int, int, int, int);
        
        /*
         * Override of FLTK for mouse handling
         */
        virtual int handle(int event);

        /*
         * Change the scale.  Note that aspect ratio is maintained;
         * calling either of these functions will update both x and
         * y window
         */
        void set_easting_window(double new_window);
        void set_northing_window(double new_window);

        /*
         * Change the center of the display
         */
        void set_center(double new_easting, double new_northing);

        /*
         * Set the vehicle position for the display.  This is not needed
         * unless you want to either use vehicle following mode or
         * want to use vehicle-coordinate callbacks.
         */
        void set_veh_pos(double easting, double northing, double yaw, double smooth_x = 0, double smooth_y = 0, double smooth_yaw = 0);


        
        /*
         * Set the status of the mouse actions.  This is useful
         * for cases like being able to toggle the vehicle state.
         */
        void mouse_moveable(bool state) { _mouse_move_enabled = state; }
        void mouse_zoomable(bool state) { _mouse_zoom_enabled = state; }
        bool mouse_is_moveable()        { return _mouse_move_enabled;  }
        bool mouse_is_zoomable()        { return _mouse_zoom_enabled;  }

        /*
         * Set up or query the state of the vehicle following code
         */
        void track_veh_pos(bool state) { _track_veh_pos = state; }
        void track_veh_rot(bool state) { _track_veh_rot = state; }
        bool is_tracking_veh_pos()     { return _track_veh_pos;  }
        bool is_tracking_veh_rot()     { return _track_veh_rot;  }

        /*
         * Add a rendering callback.  Order is processed from lowest
         * to highest
         */
        void add_callback(void (*fn)(double easting_origin, double northing_origin, 
                                     double low_easting, double low_northing, double high_easting, double high_northing, utils::SymbolTable *st, void *data),
                          int priority,
                          DisplayCoordFrame frame,
                          void *data);

        /*
         * Helper function for interfaces to parse common display options 
         */
        static void setup(void (*fn)(double easting_origin, double northing_origin, 
                                     double low_easting, double low_northing, double high_easting, double high_northing, utils::SymbolTable *st, void *data),
                          int default_priority,
                          DisplayCoordFrame frame,
                          void *ptr, utils::ConfigFile *params, utils::SymbolTable *globals);

 protected:
        virtual void draw();
        double _min_easting, _min_northing, _max_easting, _max_northing;
        double _mid_easting;
        double _mid_northing;
        double _northing_window;
        double _easting_window;
 private:
        /* Stuff for handing dragging of the map */
        int _last_mouse_x, _last_mouse_y, _drag_active;
        double _drag_start_mid_x, _drag_start_mid_y;
        int _drag_delta_x, _drag_delta_y;
        int _dragged;

        /*
         * Vehicle position tracking
         */
        double _veh_easting, _veh_northing, _veh_yaw;

        /*
         * Offset between smooth and global coordinate tracking
         */
        double _easting_offset, _northing_offset, _yaw_offset;
        
        bool _track_veh_pos;
        bool _track_veh_rot;


        bool _mouse_move_enabled;
        bool _mouse_zoom_enabled;

        /*
         * Symbols for inter-render communication (e.g. veh_state)
         */
        utils::SymbolTable _symbols;


        /*
         * Callback list
         */
        std::list<DisplayCallback> _callbacks;

        /*
         * Private helper functions
         */
        void _update_easting_window() {
                _easting_window =  _northing_window * w()/((double)h());
        }
        void _update_northing_window() {
                _northing_window = _easting_window * h()/((double)w());
        }
        void _recalc_bounds() {
                _max_northing = _mid_northing + _northing_window/2;
                _min_northing = _mid_northing - _northing_window/2;
                _min_easting = _mid_easting - _easting_window/2;
                _max_easting = _mid_easting + _easting_window/2;
        }

        void _zoom_in();
        void _zoom_out();

        /*
         * String storage for window title
         */
        char *_name;


	// True if the background is black, otherwise white.
	bool _black_background;

};

#endif
                
