

#ifndef DISABLE_DISPLAY

#include <stdio.h>
#include <utils/ConfigFile.h>
#include <Display/Display.h>
#include <GL/gl.h>
#include <math.h>
#include <iostream>
#include <assert.h>

using namespace std;

/*
 * Helper macros
 */
#ifndef PI
#define PI ((double)3.14159265358979323)
#endif
#define _TO_DEGREES(a) ((a) * 360 / (2 * PI))
#ifndef MAX
#define MAX(a, b) (((a)>(b))?(a):(b))
#endif
#ifndef MIN
#define MIN(a, b) (((a)<(b))?(a):(b))
#endif

/*
 * Class for keeping track of callbacks
 */
class DisplayCallback
{
public:
	DisplayCallback(void (*callback_init)(double, double, double, double, double, double, utils::SymbolTable *, void *),
			      int priority_init,
			      DisplayCoordFrame frame_init,
			      void *data_init) : 
		callback(callback_init),
		priority(priority_init),
		frame(frame_init),
		data(data_init) { /* no body */ }

	/*
	 * For sorted list of callbacks, sort based on priority
	 */
	int operator < (const DisplayCallback &rhs) {
		return priority < rhs.priority;
	}
	
	
	void (*callback)(double easting_offset, double northing_offset, 
                         double low_x, double low_northing, double high_x, double high_northing, utils::SymbolTable *st, void *);
	int priority;
	DisplayCoordFrame frame;
	void *data;
};


Display::Display(const char *spec,
                             utils::SymbolTable *globals)
        : Fl_Gl_Window(1, 1, 1, 1, "Foozle")
{
        utils::ConfigFile params;
        params.parse(spec);
        /*
	 * Simply create the display and add the window to the
	 * display window
	 */
	int x, y, w, h;
	const char *name;
	bool enable_mouse_move;
	bool enable_zoom;
	bool track_veh_pos;
	bool track_veh_rot;

        x = params.getInt("x_pos", 0);
        y = params.getInt("y_pos", 0);
	w = params.getInt("width", 200);
	h = params.getInt("height", 200);
	name = params.getString("name", "Unnamed Window");
	_mouse_move_enabled = params.getBool("mouse_move", true);
	_mouse_zoom_enabled = params.getBool("zoom", true);
	_track_veh_pos = params.getBool("track_position", false);
	_track_veh_rot = params.getBool("track_rotation", false);
	_black_background = params.getBool("black_background", true);

        _name = new char [strlen(name) + 1];

        strcpy(_name, name);

        label(_name);
	resize(x, y, w, h);
        _northing_window = h;
        _update_easting_window();
        _recalc_bounds();
	
	/*
	 * Add ourself to the symbol table
	 */
	if (globals->valid(name)) {
		if (*(bool *)globals->get("verbose")) {
                        fprintf(stderr, "Symbol named %s multiply defined!\n", name);
		}
	} 
	if (!globals->set(name, this)) {
                fprintf(stderr, "Unhandled error!  Ack!\n");
	}
}

Display::Display(int w, int h,
                 const char *name,
                 bool track_vehicle_pos,    /* Does the display track the vehicle position? */
                 bool track_vehicle_rot,    /* Does the display track the vehicle rotation? */
                 bool mouse_move_enable, /* Can the user move the display with the mouse?    */
                 bool mouse_zoom_enable) /* Can the user zoom in/out with the mouse?         */
        : Fl_Gl_Window(w, h, name),
          _track_veh_pos(track_vehicle_pos),
          _track_veh_rot(track_vehicle_rot),
          _mouse_move_enabled(mouse_move_enable),
          _mouse_zoom_enabled(mouse_zoom_enable)
{
        resizable(this);
        _northing_window = h;
        _mid_northing = ((double)h) / 2;
        _mid_easting = ((double)w) / 2;
        _update_easting_window();
        _recalc_bounds();
}

Display::Display(int x, int y, int w, int h, const char* l) :
  Fl_Gl_Window(x, y, w, h, l),
  _track_veh_pos(false),
  _track_veh_rot(false),
  _mouse_move_enabled(true),
  _mouse_zoom_enabled(true)
{
  resizable(this);
  _northing_window = h;
  _mid_northing = ((double)h) / 2;
  _mid_easting = ((double)w) / 2;
  _name = NULL;
  _update_easting_window();
  _recalc_bounds();
}

void Display::initialize(utils::SymbolTable* globals, const char* name,
                         bool track_vehicle_pos,
                         bool track_vehicle_rot,
                         bool enable_mouse_move, 
                         bool enable_zoom)
{
  _track_veh_pos = track_vehicle_pos;
  _track_veh_rot = track_vehicle_rot;
  _mouse_move_enabled = enable_mouse_move;
  _mouse_zoom_enabled = enable_zoom;

  _name = new char[strlen(name)+1];
  strcpy(_name, name);

  label(_name);
  
  /* Add ourself to the symbol table */
  if (globals->valid(name)) {
    if (*(bool *)globals->get("verbose")) {
      fprintf(stderr, "Symbol named %s multiply defined!\n", name);
    }
  } 
  if (!globals->set(name, this)) {
    fprintf(stderr, "Unhandled error!  Ack!\n");
  }
}

Display::~Display()
{
        delete [] _name;
}

			     
/*
 * Override the FLTK resize.  We want to catch this so we can keep the
 * scale of the window the same, otherwise, resizing would stretch the
 * image.
 */

void Display::resize(int new_easting, int new_northing, int new_w, int new_h)
{
	if (h()) {
		_northing_window *= ((double)new_h)/h();
	}
	Fl_Gl_Window::resize(new_easting, new_northing, new_w, new_h);
	_update_easting_window();
	_recalc_bounds();
}

/*
 * Handle mouse events.  We handle 3 things here:
 *
 * - Dragging with mouse button 1 moves the display in that direction
 * - Clicking (but not dragging) mb1 zooms in to the spot clicked (action is on button
 *   release, to allow it to be distinguished from a drag
 * - Clicking mb2 recenters on the spot clicked
 * - Clicking mb3 zooms out and recenters on the spot clicked.
 */
int Display::handle(int event)
{
	if (_mouse_move_enabled || _mouse_zoom_enabled) {
		double scale = _easting_window / w();
		
		switch(event) {
		case FL_PUSH:
			switch(Fl::event_button()) {
			case 1:
				_last_mouse_x = Fl::event_x();
				_last_mouse_y = Fl::event_y();
				_dragged = 0;
				return 1;
			case 2:
				if (_mouse_move_enabled) {
					_mid_easting = _min_easting + (scale * Fl::event_x());
					_mid_northing = _min_northing + (scale * (h() - Fl::event_y()));
					_recalc_bounds();
					redraw();
				}
				return 1;
			case 3:
				if (_mouse_move_enabled) {
					_mid_easting = _min_easting + (scale * Fl::event_x());
					_mid_northing = _min_northing + (scale * (h() - Fl::event_y()));
					_recalc_bounds();
				}
				if (_mouse_zoom_enabled) {
					_zoom_out();
				}
				redraw();
				return 1;
			}
			break;
		case FL_RELEASE:
			if (!_dragged
			    && (Fl::event_button() == 1)) {
				if (_mouse_move_enabled) {
					_mid_easting = _min_easting + (scale * Fl::event_x());
					_mid_northing = _min_northing + (scale * (h() - Fl::event_y()));
					_recalc_bounds();
				} 
				if (_mouse_zoom_enabled) {
					_zoom_in();
				}
				redraw();
			}
			return 1;
		case FL_DRAG:
			if (_mouse_move_enabled
			    && Fl::event_button() == 1) {
				int cur_mouse_x = Fl::event_x();
				int cur_mouse_y = Fl::event_y();
				if ((cur_mouse_x != _last_mouse_x)
				    || (cur_mouse_y != _last_mouse_y)) {
					_dragged = 1;
				}
                                /*
                                 * If we're rendering things at a rotation, have
                                 * to translate the mouse movement into that rotational
                                 * frame for the user to get what's expected.
                                 */
                                if (_track_veh_rot) {
                                        double cost = cos(-_veh_yaw);
                                        double sint = sin(-_veh_yaw);
                                        double dx = cur_mouse_x - _last_mouse_x;
                                        /*
                                         * Invert this because x-window coordinates are
                                         * vertically inverted relative to gl coordinates
                                         */
                                        double dy = -(cur_mouse_y - _last_mouse_y);
                                        
                                        _mid_easting -= scale * (dx * cost - dy * sint);
                                        _mid_northing -= scale * (dx * sint + dy * cost);
                                } else {
                                        _mid_easting -= (scale * (cur_mouse_x - _last_mouse_x));
                                        _mid_northing += (scale * (cur_mouse_y - _last_mouse_y));
                                }
				_last_mouse_x = cur_mouse_x;
				_last_mouse_y = cur_mouse_y;
				_recalc_bounds();
				redraw();
			}
			return 1;
		}
	}
	return Fl_Gl_Window::handle(event);
}

void Display::set_easting_window(double new_window) 
{
	_easting_window = new_window;
	_update_northing_window();
}

void Display::set_northing_window(double new_window)
{
	_northing_window = new_window;
	_update_easting_window();
}

void Display::set_center(double new_easting, double new_northing)
{
	if ((new_easting != _mid_easting) || (new_northing != _mid_northing)) {
		_mid_easting = new_easting;
		_mid_northing = new_northing;
		redraw();
	}
}

void Display::add_callback(void (*fn)(double easting_offset, double northing_offset, 
                                      double min_easting, double min_northing, double max_easting, double max_northing, utils::SymbolTable *st, void *ptr),
				 int priority,
				 DisplayCoordFrame frame,
				 void *ptr)
{
	DisplayCallback tmp(fn, priority, frame, ptr);
	_callbacks.push_front(tmp);
	_callbacks.sort();
}


void Display::draw()
{
	double low_easting, low_northing, high_easting, high_northing; /* Rendering bounding box */

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	// set background color.
	if (_black_background)
	  glClearColor(0,0,0,0);
	else
	  glClearColor(1,1,1,0);
	glClear(GL_COLOR_BUFFER_BIT);	// clear the window

	if (_track_veh_pos) {
		_mid_easting = _veh_easting;
		_mid_northing = _veh_northing;
		_recalc_bounds();
	}
	

  
        /*
         * Center the rendering coordinate system around the last vehicle position.  That
         * means we should never see jerkiness in the rendering due to lack of precision
         * in the numbers.
         */
        double adj_mid_easting = _mid_easting - _veh_easting;
        double adj_min_easting = _min_easting - _veh_easting;
        double adj_max_easting = _max_easting - _veh_easting;
        double adj_mid_northing = _mid_northing - _veh_northing;
        double adj_min_northing = _min_northing - _veh_northing;
        double adj_max_northing = _max_northing - _veh_northing;

	glMatrixMode(GL_PROJECTION);
	/*
	 * If we're in vehicle-following mode, set up the projection
	 * accordingly, and recalculate the bounding box
	 */
	double x_ofs = _mid_easting - _min_easting;
	double y_ofs = _mid_northing - _min_northing;
	glLoadIdentity();
	glOrtho(adj_min_easting , adj_max_easting, adj_min_northing, adj_max_northing, -1, 1);
	if (_track_veh_rot) {
		glTranslatef(adj_mid_easting, adj_mid_northing, 0);
		glRotatef(_TO_DEGREES(_veh_yaw), 0, 0, 1);
		glTranslatef(-adj_mid_easting, -adj_mid_northing, 0);
		/*
		 * Recalculate the bounding box
		 */
		double sin_yaw = sin(_veh_yaw);
		double cos_yaw = cos(_veh_yaw);
		double tmp;
                
                low_easting  = high_easting = _mid_easting + (-x_ofs * cos_yaw - y_ofs * -sin_yaw);
                tmp          = _mid_easting  + (-x_ofs * cos_yaw + y_ofs * -sin_yaw);
		low_easting  = MIN(low_easting, tmp);
		high_easting = MAX(high_easting, tmp);
		tmp            = _mid_easting + (x_ofs * cos_yaw - y_ofs * -sin_yaw);
		low_easting  = MIN(low_easting, tmp);
		high_easting = MAX(high_easting, tmp);
		tmp            = _mid_easting + (x_ofs * cos_yaw + y_ofs * -sin_yaw);
		low_easting  = MIN(low_easting, tmp);
		high_easting = MAX(high_easting, tmp);

		low_northing = high_northing = _mid_northing + (-x_ofs * sin_yaw - y_ofs * cos_yaw);
		tmp            = _mid_northing + (-x_ofs * sin_yaw + y_ofs * cos_yaw);
		low_northing = MIN(low_northing, tmp);
		high_northing = MAX(high_northing, tmp);
		tmp            = _mid_northing + ( x_ofs * sin_yaw - y_ofs * cos_yaw);
		low_northing = MIN(low_northing, tmp);
		high_northing = MAX(high_northing, tmp);
		tmp            = _mid_northing + ( x_ofs * sin_yaw + y_ofs * cos_yaw);
		low_northing = MIN(low_northing, tmp);
		high_northing = MAX(high_northing, tmp);
	} else {
		low_easting = _min_easting;
		low_northing = _min_northing;
		high_easting = _max_easting;
		high_northing = _max_northing;
	}
	glViewport(0, 0, w(), h());

	
	
	/*
	 * Iterate through the callbacks and call them.  Setup (and undo) the
         * proper coordinate frame for each call.
	 */
	list<DisplayCallback>::iterator i;
	for (i = _callbacks.begin(); i != _callbacks.end(); i++) {
		switch(i->frame) {
		case DISPLAY_GLOBAL:
			break;
                case DISPLAY_SMOOTH:
                        glMatrixMode(GL_MODELVIEW);
                        glPushMatrix();
                        glTranslatef(-_easting_offset, -_northing_offset, 0);
                        glRotatef(_TO_DEGREES(-_yaw_offset), 0, 0, -1);
                        break;
		case DISPLAY_VEHICLE:
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glRotatef(_TO_DEGREES(_veh_yaw), 0, 0, -1);
			break;
		case DISPLAY_WINDOW:
			glMatrixMode(GL_PROJECTION);
			glPushMatrix();
			glLoadIdentity();
			glOrtho(0, w(), 0, h(), -1, 1);
			break;
		default:
			/* WTF? */
			assert(0);
		}
		(*(i->callback))(_veh_easting, _veh_northing, 
                                 low_easting, low_northing, high_easting, high_northing, &_symbols, i->data);
		switch(i->frame) {
		case DISPLAY_GLOBAL:
			break;
		case DISPLAY_VEHICLE:
                case DISPLAY_SMOOTH:
			glMatrixMode(GL_MODELVIEW);
			glPopMatrix();
			break;
		case DISPLAY_WINDOW:
			glMatrixMode(GL_PROJECTION);
			glPopMatrix();
			break;
		default:
			/* WTF? */
			assert(0);
		}
	}
}

void Display::_zoom_in()
{
	_northing_window /= 1.5;
	_update_easting_window();
	_recalc_bounds();
	redraw();
}

void Display::_zoom_out()
{
	_northing_window *= 1.5;
	_update_easting_window();
	_recalc_bounds();
	redraw();
}

void Display::set_veh_pos(double easting, double northing, double yaw, double smooth_x, double smooth_y, double smooth_yaw)
{
	_veh_easting = easting;
	_veh_northing = northing;
	_veh_yaw = yaw;
        _yaw_offset = smooth_yaw - yaw;
        _easting_offset  = smooth_y * cos(_yaw_offset) - smooth_x * sin(_yaw_offset);
        _northing_offset = smooth_y * sin(_yaw_offset) + smooth_x * cos(_yaw_offset);
}



void Display::setup(void (*fn)(double easting_origin_offset, double northing_origin_offset,
                               double low_easting, double low_northing, double high_easting, double high_northing, utils::SymbolTable *st, void *data),
                          int default_priority, DisplayCoordFrame frame,
                          void *ptr, utils::ConfigFile *params, utils::SymbolTable *globals)
{
	const char *display_window = params->getString("display");
	if (display_window[0]) {
		Display *dw = (Display *)globals->get(display_window);
		if (dw == NULL) {
			bool *tmp;
			tmp = (bool *)globals->get("verbose") ;
			if ((tmp != NULL)
			    && (*tmp == true)) {
                                fprintf(stderr, "Display %s doesn't exist.\n", display_window);
			}
		} else {
			int priority = params->getInt("display_priority", default_priority);
			dw->add_callback(fn, priority, frame, ptr);
		}
	}
}


#endif
