#ifndef _VID_CANNED_DATA_H
#define _VID_CANNED_DATA_H
#define _IN_VID_CANNED_DATA_H

/*
 * Classes for reading and writing video data.
 * These are based on ffmpeg's excellent libavcodec library,
 * and the interfaces are designed to be similar to the
 * existing CannedData interfaces.
 */

#define _LARGEFILE_SOURCE 1
#define _LARGEFILE64_SOURCE 1
#define _FILE_OFFSET_BITS 64
#include <stdint.h>
#include <string>
#include <vector>
#include <stdio.h>
#include <utils/Time.h>
#include <ffmpeg/avcodec.h>
#include <utils/CannedDataAccess.h>
#include <pthread.h>
__UTILS_BEGIN_NAMESPACE

#include <CannedVideo/CannedVideoPriv.h> /* Private declarations */



enum CannedVideoFrameType
{
        CVFT_UNUSED,   /* Reserved                                 */
        CVFT_I,        /* Frame is an index frame                  */
        CVFT_P,        /* Frame is a forward predicted frame       */
        CVFT_B,        /* Frame is a bidirectional predicted frame */
};

enum CannedVideoType
{
        CVT_UNUSED,
        CVT_MPEG1,
        CVT_MPEG2,
        CVT_MJPEG,
        CVT_MPEG4
};


enum CannedVideoFrameFormat
{
        CVFF_UNUSED,
        /*
         * Planar, 8 bit Y values as a block, then 8 bit 2x2 pixel U values as
         * a block, then 8 bit 2x2 pixel V values as a block
         */
        CVFF_YUV,

        /*
         * Interleaved 8 bit RGB values
         */
        CVFF_RGB24,

        /*
         * Interleaved 8 bit BGR values
         */
        CVFF_BGR24,

        /*
         * 8-bit greyscale values
         */
        CVFF_GREYSCALE
};

class CannedVideoFrameIndex;
class CannedVideoFramesBuffer;

class CannedVideoReader : public CannedReader
{
 public:
        CannedVideoReader();
        CannedVideoReader(const char *filename);
        ~CannedVideoReader();

        /*
         * These are specified by the base class
         */
        void open(const char *filename);
        void close();
	uint64_t getCreationTime();
        void getVersion(uint32_t *major, uint32_t *minor);
        FILE *getHeader();
	size_t getHeaderSize();
	void seek(const Time &when);
	int seekBefore(const Time &when);
        int eof();
        void first();
        void next();
        void prev();
        void last();
        Time getCurTimeStamp();
        Time getFirstTimeStamp();
	Time getLastTimeStamp();
	int  getPrevTimeStamp(Time *time);
	int  getNextTimeStamp(Time *time);
        unsigned int getNumRecords();
        

        /*
         * Extract an excerpt segment of this file to another file.  If raw is set, then export
         * in raw format, whatever that was originally.  Either way, this may give you slightly more
         * video on the front side than you request, as it doesn't reencode, and so
         * needs to start with a keyframe
         */        
        void excerpt(const char *filename, const Time &startTime, const Time &endTime, bool raw);

	/*
	 * Get the description of this file 
	 */
	const std::string &getDescription();

        /*
         * Get the aspect ratio of the video data.
         */
        unsigned int width();
        unsigned int height();


        /*
         * Get a pointer to the current frame data in this format.  Note
         * this is only valid until the next member function
         * call.
         */
	const unsigned char *getFrame(CannedVideoFrameFormat format = CVFF_GREYSCALE);
 private:
        void _alloc();
        void _init();

        AVCodec         *_avcodec;
        AVCodecContext  *_avcontext;
        AVFrame         *_frame;

        unsigned int     _last_decoded;

        CannedVideoFramesBuffer    *_frames_buffer;
        
        CannedVideoState _state;
        FILE            *_file;
        CannedVideoType  _type;

        uint32_t         _major_version;
        uint32_t         _minor_version;
        uint64_t         _creation_time;
        std::string      _desc;

        uint32_t         _header_size;
        uint64_t         _header_offset;

        CannedVideoFrameIndex *_index;

        uint32_t         _w;
        uint32_t         _h;

        uint32_t         _frames_per_keyframe;

        unsigned char   *_buf;

        friend class CannedVideoWriter;
        
};



class CannedVideoWriter {
 public:
        CannedVideoWriter();
        ~CannedVideoWriter();

	/*
	 * Open a file for writing.  Create the
	 * file if it doesn't exist.  If it does exist, 
	 * clobber it iff mode doesn't include O_EXCL.  Set 
	 * file type and version data accordingly Ignore
	 * the other mode bits for now.  
         *
         * Use the specified codec, and expect data of the size given
 	 */
        void open(unsigned int width, unsigned int height, CannedVideoType codec, const char *filename, bool encode, 
                  unsigned int _frames_per_keyframe, mode_t mode = 0,
                  /* The rest of these parameters are only needed if you're having the Writer do the encode */
                  int bit_rate = 0, int frame_rate = 0,
                  CannedVideoFrameFormat format = CVFF_UNUSED);

        /*
         * Specific interface for the PC104 hardware encoder we're using.  If
         * you use this, then a thread is spawned that writes the output
         * to the file until you call stop_sx12()
         */
        void open_sx12(const char *dev_name, const char *outfile_name, 
                       int bit_rate, int frame_rate);
        
        /*
         * Extract a segment from this reader (which should already be open).  This open() is unique in that
         * it also closes() before returning.  
         * 
         * If raw is true, strip out all CannedVideo extra information, leaving the raw data stream (e.g. mpeg, usually).
         * 
         * In any case, the actual video range returned may be slightly longer than the requested segment, because we
         * actually write from the keyframe preceeding the requested time.  This shouldn't be a big deal unless you're
         * trying to do sync'ed raw mpeg or somesuch.
         */
        void open(CannedVideoReader &reader, const char *filename, const Time &startTime, const Time &endTime, bool raw, mode_t mode = 0);


        void start_sx12();


        /*
         * Stop a running encode thread.  Won't return until the encode thread is stopped.
         */
        void stop_sx12();

	/*
         * All of the writeXXX functions must be called after a successful
	 * open, and may not be called after a close.  With the exception
	 * of writeRecord, all of these functions can only be called once
	 * for a given file.  Once you've called writeRecord(), calling any
         * of the other writeXXX functions is forbidden -- these are really
         * header information, and so I don't deal with getting requests
         * to write them midstream.
	 */

        /*
         * Write a description to the file.  
         */
        void writeDescription(const char *desc);

	/* 
	 * Set the version information for this file.  
	 */
	
	void writeVersion(uint32_t major, uint32_t minor);

	/*
	 * Write some custom header information for this file.
	 */
	void writeHeader(void *buf, size_t size);
	
        /*
         * write a pre-encoded frame
         */
        void writeFrame(const Time &timestamp, unsigned char *buf,
                        /*
                         * These fields are only needed for pre-encoded frames 
                         */
                        uint32_t size = 0, CannedVideoFrameType type = CVFT_UNUSED);

        /*
         * Write all metadata to disk, and finish out the file, closing it.
         */
        void close();


 private:
        void _writeFrame(FILE *file, const Time &time, unsigned char *buf, uint32_t size, CannedVideoFrameType type);
        void _init();
        void _writeIndex(FILE *file, Time time, CannedVideoFrameType type, uint32_t size);
        void _writeRecord(FILE *file, CannedVideoFieldID id, const void *buf, uint64_t size);
        void _doFwrite(FILE *file, const void *buf, size_t size);
        void _cleanup();
        void _checkClobber(const char *name, mode_t mode);
        void _createIndexBlock(FILE *file);
        void _pushFilePos(FILE *file);
        void _popFilePos(FILE *file);

        void _updateFrameBlockSize(FILE *file, size_t size);

        


        off64_t            _idx_ptr;
        off64_t            _idx_slots_ptr;
        unsigned int       _idx_slots;
        unsigned int       _idx_slots_used;

        /*
         * Most recently decoded frame
         */
        unsigned int       _last_decoded;


        unsigned int       _frames_per_index_block;
        uint32_t           _headers_written;
        CannedVideoState   _state;
        FILE              *_file;

        uint32_t           _w;
        uint32_t           _h;

        unsigned int       _depth;

        off64_t            _saved_ptr;
        off64_t            _frame_block_size_ptr;

        uint64_t           _frame_block_size;

        bool               _encode;

        /*
         * Stuff for doing encode
         */
        AVCodec               *_codec;
        AVCodecContext        *_context;
        AVFrame               *_picture;

        unsigned char         *_outbuf;
        int                    _outbuf_size;
        unsigned char         *_convert_buf;
        int                    _incoming_format;
        
        /*
         * sx12 specific stuff
         */
        double                 _expected_frame_interval;
        volatile int           _stop_sx12;
        volatile int           _sx12_fd;
        pthread_t              _encode_thread;
 public:
        /* 
         * This is only public to workaround pthread stuff.  Don't call it
         * directly
         */
        void                   _sx12_encode_loop();
        

};


__UTILS_END_NAMESPACE

#undef _IN_VID_CANNED_DATA_H
#endif

