
#include "options.h"
#include <stdio.h>
#include "assert.h"
#include "macros.h"
#include "index.h"
#include "split.q.h"

# ifndef	lint
	static	char	splitqSccsid[] = "%W% %G%";
# endif	lint

#ifdef GRAPHICS
#	define UNIX
#	include "/jb/ingres/toni/cad/lib/mfb.h"
#endif


/*-----------------------------------------------------------------------------
| Split a node.
| Divides the nodes branches and the extra one between two nodes.
| Old node is one of the new ones, and one really new one is created.
| Tries more than one method for choosing a partition, uses best result.
-----------------------------------------------------------------------------*/
SplitNode(n, b, nn)
register struct Node *n;
register struct Branch *b;
register struct Node **nn;
{
	register struct PartitionVars *p;
	register int level;
	struct Node *NewNode();
	int area;

	assert(n);
	assert(b);

#	ifdef PRINT
		printf("Splitting:\n");
		PrintNode(n);
		printf("new branch:\n");
		PrintBranch(b);
#	endif

	if (StatFlag)
	{
		if (Deleting)
			DeSplitCount++;
		else
			InSplitCount++;
	}

	/* load all the branches into a buffer, initialize old node */
	level = n->level;
	GetBranches(n, b);

#	ifdef SHOWSPLIT
	{
		int i;
		/* Indicate that a split is about to take place */
		MFBSetColor(RED);
		for (i=0; i<NODECARD+1; i++)
		{
			PrintRect(&BranchBuf[i].rect);
		}
		MFBSetColor(YELLOW);
		PrintRect(&CoverSplit);
		GraphChar();
	}
#	endif

	/* find partition */
	p = &Partitions[0];
	MethodZero(p);

	area = RectArea(&p->cover[0]) + RectArea(&p->cover[1]);

	/* record how good the split was for statistics */
	if (StatFlag && !Deleting && area)
		SplitMeritSum += (float)CoverSplitArea / area;

	/* put branches from buffer into 2 nodes according to chosen partition */
	*nn = NewNode();
	(*nn)->level = n->level = level;
	LoadNodes(n, *nn, p);
	assert(n->count+(*nn)->count == NODECARD+1);

#	ifdef PRINT
		PrintPVars(p);
		printf("group 0:\n");
		PrintNode(n);
		printf("group 1:\n");
		PrintNode(*nn);
		printf("\n");
#	endif

#	ifdef SHOWSPLIT
		EraseRect(&CoverSplit);
#	endif
}

/*-----------------------------------------------------------------------------
| Load branch buffer with branches from full node plus the extra branch.
-----------------------------------------------------------------------------*/
GetBranches(n, b)
register struct Node *n;
register struct Branch *b;
{
	register int i;
	struct Rect CombineRect();

	assert(n);
	assert(b);

	/* load the branch buffer */
	for (i=0; i<NODECARD; i++)
	{
		assert(n->branch[i].child);  /* node should have every entry full */
		BranchBuf[i] = n->branch[i];
	}
	BranchBuf[NODECARD] = *b;

	/* calculate rect containing all in the set */
	CoverSplit = BranchBuf[0].rect;
	for (i=1; i<NODECARD+1; i++)
	{
		CoverSplit = CombineRect(&CoverSplit, &BranchBuf[i].rect);
	}
	CoverSplitArea = RectArea(&CoverSplit);

	InitNode(n);
}

/*-----------------------------------------------------------------------------
| Method #0 for choosing a partition:
| As the seeds for the two groups, pick the two rects that would waste the
| most area if covered by a single rectangle, i.e. evidently the worst pair
| to have in the same group.
| Of the remaining, one at a time is chosen to be put in one of the two groups.
| The one chosen is the one with the greatest difference in area expansion
| depending on which group - the rect most strongly attracted to one group
| and repelled from the other.
| If one group gets too full (more would force other group to violate min
| fill requirement) then other group gets the rest.
| These last are the ones that can go in either group most easily.
-----------------------------------------------------------------------------*/
MethodZero(p)
register struct PartitionVars *p;
{
	register struct Rect *r;
	register int i, growth0, growth1, diff, biggestDiff;
	register int group, chosen, betterGroup;
	assert(p);

#	ifdef SHOWSPLIT
		EraseData();
#	endif

	InitPVars(p);
	PickSeeds(p);

	while (p->count[0] + p->count[1] < NODECARD + 1
		&& p->count[0] < NODECARD + 1 - MinFill
		&& p->count[1] < NODECARD + 1 - MinFill)
	{
		biggestDiff = -1;
		for (i=0; i<NODECARD+1; i++)
		{
			if (!p->taken[i])
			{
				r = &BranchBuf[i].rect;
				growth0 = RectArea(&CombineRect(r, &p->cover[0])) - p->area[0];
				growth1 = RectArea(&CombineRect(r, &p->cover[1])) - p->area[1];
				diff = growth1 - growth0;
				if (diff >= 0)
					group = 0;
				else
				{
					group = 1;
					diff = -diff;
				}

				if (diff > biggestDiff)
				{
					biggestDiff = diff;
					chosen = i;
					betterGroup = group;
				}
				else if (diff==biggestDiff && p->count[group]<p->count[betterGroup])
				{
					chosen = i;
					betterGroup = group;
				}
			}
		}
		Classify(chosen, betterGroup, p);
	}

	/* if one group too full, put remaining rects in the other */
	if (p->count[0] + p->count[1] < NODECARD + 1)
	{
		if (p->count[0] >= NODECARD + 1 - MinFill)
			group = 1;
		else
			group = 0;
		for (i=0; i<NODECARD+1; i++)
		{
			if (!p->taken[i])
				Classify(i, group, p);
		}
	}

	assert(p->count[0] + p->count[1] == NODECARD+1);
	assert(p->count[0] >= MinFill && p->count[1] >= MinFill);

#	ifdef SHOWSPLIT
		/* erase the final split display */
		EraseRect(&p->cover[0]);
		EraseRect(&p->cover[1]);
		EraseData(p);
#	endif
}

/*-----------------------------------------------------------------------------
| Pick two rects from set to be the first elements of the two groups.
| Pick the two that waste the most area if covered by a single rectangle.
-----------------------------------------------------------------------------*/
PickSeeds(p)
register struct PartitionVars *p;
{
	register int i, j, waste, worst, seed0, seed1;
	int area[NODECARD+1];
	struct Rect CombineRect();

	for (i=0; i<NODECARD+1; i++)
		area[i] = RectArea(&BranchBuf[i].rect);

	worst = -CoverSplitArea - 1;
	for (i=0; i<NODECARD; i++)
	{
		for (j=i+1; j<NODECARD+1; j++)
		{
			waste = RectArea(&CombineRect(&BranchBuf[i].rect, &BranchBuf[j].rect))
				- area[i] - area[j];
			if (waste > worst)
			{
				worst = waste;
				seed0 = i;
				seed1 = j;
			}
		}
	}
	Classify(seed0, 0, p);
	Classify(seed1, 1, p);
}

/*-----------------------------------------------------------------------------
| Put a branch in one of the groups.
-----------------------------------------------------------------------------*/
Classify(i, group, p)
register int i, group;
register struct PartitionVars *p;
{
	struct Rect CombineRect();

	assert(p);
	assert(!p->taken[i]);

#	ifdef SHOWSPLIT
		/* erase old group cover if nec. */
		if (p->count[group] > 0)
			EraseRect(&p->cover[group]);
#	endif

	p->partition[i] = group;
	p->taken[i] = TRUE;

	if (p->count[group] == 0)
		p->cover[group] = BranchBuf[i].rect;
	else
		p->cover[group] = CombineRect(&BranchBuf[i].rect, &p->cover[group]);
	p->area[group] = RectArea(&p->cover[group]);
	p->count[group]++;

#	ifdef SHOWSPLIT
	{
		/* redraw entire group and its cover */
		int j;
		MFBSetColor(WHITE); /* cover is white */
		PrintRect(&p->cover[group]);
		MFBSetColor(group + 3); /* group 0 green, group 1 blue */
		for (j=0; j<NODECARD+1; j++)
		{
			if (p->taken[j] && p->partition[j]==group)
			PrintRect(&BranchBuf[j].rect);
		}
		GraphChar();
	}
#	endif
}

/*-----------------------------------------------------------------------------
| Copy branches from the buffer into two nodes according to the partition.
-----------------------------------------------------------------------------*/
LoadNodes(n, q, p)
register struct Node *n, *q;
register struct PartitionVars *p;
{
	register int i;
	assert(n);
	assert(q);
	assert(p);

	for (i=0; i<NODECARD+1; i++)
	{
		assert(p->partition[i] == 0 || p->partition[i] == 1);
		if (p->partition[i] == 0)
			AddBranch(&BranchBuf[i], n, NULL);
		else if (p->partition[i] == 1)
			AddBranch(&BranchBuf[i], q, NULL);
	}
}

/*-----------------------------------------------------------------------------
| Initialize a PartitionVars structure.
-----------------------------------------------------------------------------*/
InitPVars(p)
register struct PartitionVars *p;
{
	register int i;
	struct Rect NullRect();
	assert(p);

	p->count[0] = p->count[1] = 0;
	p->cover[0] = p->cover[1] = NullRect();
	p->area[0] = p->area[1] = 0;
	for (i=0; i<NODECARD+1; i++)
	{
		p->taken[i] = FALSE;
		p->partition[i] = -1;
	}
}

/*-----------------------------------------------------------------------------
| Print out data for a partition from PartitionVars struct.
-----------------------------------------------------------------------------*/
PrintPVars(p)
register struct PartitionVars *p;
{
	register int i;
	assert(p);

	printf("\npartition:\n");
	for (i=0; i<NODECARD+1; i++)
	{
		printf("%3d\t", i);
	}
	printf("\n");
	for (i=0; i<NODECARD+1; i++)
	{
		if (p->taken[i])
			printf("  t\t");
		else
			printf("\t");
	}
	printf("\n");
	for (i=0; i<NODECARD+1; i++)
	{
		printf("%3d\t", p->partition[i]);
	}
	printf("\n");

	printf("count[0] = %d  area = %d\n", p->count[0], p->area[0]);
	printf("count[1] = %d  area = %d\n", p->count[1], p->area[1]);
	if (p->area[0] + p->area[1] > 0)
	{
		printf("total area = %d  effectiveness = %3.2f\n",
			p->area[0] + p->area[1],
			(float)CoverSplitArea / (p->area[0] + p->area[1]));
	}
	printf("cover[0]:\n");
	PrintRect(&p->cover[0]);

	printf("cover[1]:\n");
	PrintRect(&p->cover[1]);
}

/*-----------------------------------------------------------------------------
| Erase all rects in the set to be split by drawing them black.  Graphics only.
-----------------------------------------------------------------------------*/
EraseData()
{
	int i;

	for (i=0; i<NODECARD+1; i++)
		EraseRect(&BranchBuf[i].rect);
}
