/*-----------------------------------------------------------*/
//
// File:	helloblk.cxx								
//															
// Synopsis:	This is the "Hello, Block" example program.
//	
// Description:	
//
//	This program will test ACIS 3.0 capability for creating a
//	simple model, writing it to a (*.sat) file and retrieving 
//	its debug output.
//
// Input:	none.
//		
// Output:	Screen:	model debug output
//		File(s):	helloblk.dbg -- same debug output
//
//				apierror.txt -- if any api function
//				fails, the error message is written
//				to this file.
//
//				my_block.sat -- the ACIS (*.sat)
//				file format for the block just
//				created.
//
/*-----------------------------------------------------------*/

/*-----------------------------------------------------------*/
// HEADER FILE INCLUSIONS
/*-----------------------------------------------------------*/

//	Standard C++ Head Files
#include <stdio.h>				// standard i/o
#include <stdlib.h>				// standard lib
#include <fstream.h>				// c++ file i/o

//	ACIS-specific Header Files

//	Required for all ACIS functions
#include "kernel/acis.hxx"						

//	Required for function(s):	api_start_modeller
//					api_stop_modeller
//					api_initialize_kernel
//					api_terminate_kernel
//					api_del_entity
//					api_save_entity_list
#include "kernel/kernapi/api/kernapi.hxx"

//	Required for function(s):	api_start_modeller
//					api_stop_modeller
//					api_initialize_constructors
//					api_terminate_constructors
//					api_initialize_kernel
//					api_terminate_kernel
//					api_initialize_spline
//					api_terminate_spline
//					api_initialize_intersectors
//					api_terminate_intersectors
//					api_del_entity
//					api_save_entity_list
#include "kernel/kernapi/api/api.hxx"

//	Required for function(s):	api_make_cuboid
#include "operator/kernapi/api/apimsc.hxx"

//	Required for function(s):	api_initialize_constructors
//					api_terminate_constructors
//					api_make_cuboid
#include "constrct/kernapi/api/cstrapi.hxx"

//	Required for function(s):	api_initialize_spline
//					api_terminate_spline
#include "kernel/spline/api/spl_api.hxx"

//	Required for function(s):	api_initialize_intersectors
//					api_terminate_intersectors
#include "intersct/kernapi/api/intrapi.hxx"

//	Required for function(s):	debug_entity
#include "kernel/kerndata/lists/lists.hxx"
#include "kernel/kerndata/data/debug.hxx"

//	Required for function(s):	api_del_entity
#include "kernel/kerndata/data/entity.hxx"

//	Required for class:		ENTITY_LIST
#include "kernel/kerndata/lists/lists.hxx"

//	Required for class:		ENTITY
#include "kernel/kerndata/data/entity.hxx"

//	Required for class:		BODY
#include "kernel/kerndata/top/body.hxx"

/*-----------------------------------------------------------*/
// MACRO DEFINITIONS
/*-----------------------------------------------------------*/

/*-----------------------------------------------------------*/
//  PROCESS_RESULT -- writes out error message if an outcome
//	variable returns a FALSE or (0) value, which indicates
//	that the function did not execute properly.
//	Since each api function returns an outcome variable type,
//	this macro will be called after each api function call.
//	The error message will be stored in file:  apierror.txt.
/*-----------------------------------------------------------*/
#define PROCESS_RESULT						\
	if (!result.ok()) {					\
	  FILE* efp = fopen("apierror.txt","w");		\
	  print_warnerr_mess("API",result.error_number(), efp);	\
	  fclose(efp);						\
	  exit (1);						\
	} // end if(!result.ok())


/*-----------------------------------------------------------*/
//	PROCESS_FILE -- prints out error message if a file pointer
//	returns FALSE or (0) value, which indicates the file was
//	not opened.
/*-----------------------------------------------------------*/
#define PROCESS_FILE						\
	if (!fp) {						\
		printf("unable to open output file");		\
		exit(1);					\
	} // end if (!fp)


/*-----------------------------------------------------------*/
// FUNCTION PROTOTYPES
/*-----------------------------------------------------------*/

/*-----------------------------------------------------------*/
//	print_file_data -- opens a text file and prints contents
//	to the screen with the same format as in the file.
/*-----------------------------------------------------------*/
void print_file_data(
	char* filename		// name of file to be opened
	);

/*-----------------------------------------------------------*/
// START MAIN PROGRAM
/*-----------------------------------------------------------*/
main()
{
	/*-------------------------------------------------------*/
	// INITIALIZE THE MODELER.
	//	Start the ACIS modeler.  This function must be called
	//	prior to using any other ACIS function.
	/*-------------------------------------------------------*/
	int	memory_size	= 0;	//	size of model working space
					//	(in bytes)
	outcome result;			//	stores result of api calls
	
	result	= api_start_modeller(memory_size);
	PROCESS_RESULT

	/*-------------------------------------------------------*/
	// INITIALIZE OBJECT LIBRARIES
	/*-------------------------------------------------------*/
	
	//	Library:	constrct.lib
	result = api_initialize_constructors();
	PROCESS_RESULT

	//	Library:	intersct.lib
	result = api_initialize_intersectors();
	PROCESS_RESULT

	//	Library:	kernel.lib
	result = api_initialize_kernel();
	PROCESS_RESULT

	//	Library:	spline.lib
	result = api_initialize_spline();
	PROCESS_RESULT

	/*-------------------------------------------------------*/
	// CREATE A SIMPLE MODEL
	//	Create a cubical solid block of dimensions:
	//	width (x) = 50, depth (y) = 50 & height (z) = 50,
	//	centered at the origin.
	/*-------------------------------------------------------*/
	BODY* block_body;	//	block body declaration
	int	width = 50;	//	block width = 50 units
	int	depth = 50;	//	block depth = 50 units
	int	height= 50;	//	block height = 50 units

	//	This function uses a "new" operator and allocates
	//	space on the heap for the entity.
	result = api_make_cuboid(width,depth,height,block_body);
	PROCESS_RESULT

	/*-------------------------------------------------------*/
	// CREATE A (*.SAT) FILE & SAVE MODEL TO IT
	//	In order to save the model to a *.sat file, ACIS
	//	standard file format, an ENTITY_LIST must be created
	//	which contains all the ENTITY* or in this case
	//	BODY* classes.  To do this, first, the BODY* class
	//	must be added to the ENTITY_LIST, then using an api
	//	function, the ENTITY_LIST can be saved to a *.sat file.
	/*-------------------------------------------------------*/
	ENTITY_LIST block_elist;

	block_elist.add((ENTITY*)block_body);

	//	Open file to which model will be saved as (*.sat).
	FILE* fp = fopen("my_block.sat","w");
	PROCESS_FILE

	//	Save ENTITY_LIST to file.
	logical	textmode = TRUE;	//	file format is text
	result = api_save_entity_list(fp, textmode, block_elist);
	PROCESS_RESULT

	//	Close file *.sat file
	fclose(fp);

	/*-------------------------------------------------------*/
	// DEBUG THE MODEL & SAVE DATA TO A FILE
	//	Open up a file to contain model (body) debug output.
	/*-------------------------------------------------------*/
	fp = fopen("helloblk.dbg","w");
	PROCESS_FILE

	//	ACIS function "debug_entity" prints to the file given
	//	several lists of data regarding the representation of
	//	the body being debugged.  The lists include:  BODY,
	//	LUMP, SHELL, FACE, LOOP, SURFACE, COEDGE, EDGE, VERTEX,
	//	CURVE, and POINT lists for the specified body.  The
	//	output also contains a summary of the number of each
	//	of these topological entities in the body.

	debug_entity(block_body,fp);

	//	Close file containing model debug output.
	fclose(fp);

	/*-------------------------------------------------------*/
	// RETRIEVE DATA FROM DEBUG FILE AND ECHO TO SCREEN
	/*-------------------------------------------------------*/
	print_file_data("helloblk.dbg");

	/*-------------------------------------------------------*/
	// CLEAN UP MEMORY ALLOCATION
	/*-------------------------------------------------------*/

	//	Clear all entities on the ENTITY_LIST created.
	if (block_elist[0])
		block_elist.clear();

	//	Delete all entities created on the heap.
	if (block_body) {
		result = api_del_entity(block_body);
		PROCESS_RESULT
	}

	/*-------------------------------------------------------*/
	// TERMINATE OBJECT LIBRARIES
	/*-------------------------------------------------------*/
	result = api_terminate_constructors();
	PROCESS_RESULT

	result = api_terminate_intersectors();
	PROCESS_RESULT

	result = api_terminate_kernel();
	PROCESS_RESULT

	result = api_terminate_spline();
	PROCESS_RESULT

	/*-------------------------------------------------------*/
	// STOP THE MODELER.
	//	Stop the ACIS modeler.  This function may be called
	//	after finished using all ACIS functions.
	/*-------------------------------------------------------*/
	result = api_stop_modeller();
	PROCESS_RESULT

/*-----------------------------------------------------------*/
}// end main program
/*-----------------------------------------------------------*/

/*-----------------------------------------------------------*/
// FUNCTION DEFINITIONS
/*-----------------------------------------------------------*/

/*-----------------------------------------------------------*/
//	print_file_data -- opens a text file and prints contents
//	to the screen with the same format as in the file.
/*-----------------------------------------------------------*/
void
print_file_data(char* filename) {

	//	Read debug file and write the contents to the screen.
	char input_char;		// file char declaration
	cout << "The following debug output is written" << endl;
	cout << "to file:  " << filename << "." << endl << endl;

	//	Open file.
	ifstream inf(filename);

	while (inf) { // Get characters from file & print to screen
		inf.get(input_char);	// get the char from file	
		cout << input_char;	// print char to screen
	} // end while (inf)

	//	Close file.
	inf.close();

/*-----------------------------------------------------------*/
} // end function print_file_data
/*-----------------------------------------------------------*/

/*------------ End of File helloblk.cxx ---------------------*/
