// ======================================================================
// value.hpp - Values for Feature Structures
// 
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 120202: Benjamin Han <benhdj@cs.cmu.edu> Value::_StrictOp::operator() takes
//         an argument in Tree<Symbol>::ConstIterator instead of 
//         Value::ConstIterator - M$ VC6 barfs over this.
// 101302: Benjamin Han <benhdj@cs.cmu.edu> Introduce Box (undefined symbol):
//         this only occurs in RHS penetrating paths with undefined
//         branches (see fStruc.?pp, especially FStruc::collect() )
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Added Value::operator==().
// 091701: Benjamin Han <benhdj@cs.cmu.edu> Revised Value::compact() to use
//         TIsomorphismMixed<> to correctly compact various sub-values 
//         recursively.
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Renamed equal() to test().
// 090301: Benjamin Han <benhdj@cs.cmu.edu> Minor revisions to supress warnings
//         under -pedantic -Wall; moved IndexList typedef from value.hpp to
//         value.cpp; renamed Equivalence*<> to TIsomorphism*<>.
// 080701: Benjamin Han <benhdj@cs.cmu.edu> Renamed dSize() to size() in 
//         _TreeIteratorBase<>; renamed Homomorphic* function objects to 
//         Equivalence*.
// 080201: Benjamin Han <benhdj@cs.cmu.edu> Renamed dBegin() and dEnd() to 
//         begin() and end() in _TreeIteratorBase<>.
// 072001: Benjamin Han <benhdj@cs.cmu.edu> Revised using the new ways to
//         declare tree iterators.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Minor header dependency change.
// 070901: Benjamin Han <benhdj@cs.cmu.edu> Added Value::multiplePU().
// 070601: Benjamin Han <benhdj@cs.cmu.edu> Changed Value::append() to 
//         Value::push().
// 070401: Benjamin Han <benhdj@cs.cmu.edu> 
//         ostream& operator<<(ostream&,const Value&) now calls the function
//         ostream& operator<<(ostream&,const Tree<>&) in treeAlgorithm.hpp.
// 070301: Benjamin Han <benhdj@cs.cmu.edu> Moved EraseList from value.cpp to 
//         IndexList in value.hpp.
// 070201: Benjamin Han <benhdj@cs.cmu.edu> Added Value::mergeOr().
// 070101: Benjamin Han <benhdj@cs.cmu.edu> Revised the comments; removed
//         Value::puOrMultiple() - identical to the new Value::puDifference();
//         in two reduceToAtom() call _TreeIteratorBase<>::update() to update
//         the data instead of direct manipulation; restrict the use of
//         _VIterator and _VConstIterator: now only Value::compact() and 
//         Value::equal() use them.
// 062901: Benjamin Han <benhdj@cs.cmu.edu> Revised the comments of
//         Value::compact() and Value::pseudoUnify(); removed Value::puUnion().
// 062501: Benjamin Han <benhdj@cs.cmu.edu> Added _VIterator(_TreeNode<Symbol>*).
// 061301: Benjamin Han <benhdj@cs.cmu.edu> Added isAtom().
// 051501: Benjamin Han <benhdj@cs.cmu.edu> Added isNumber(), isInteger()
//         and isPositive(); code cleanup.
// 050801: Benjamin Han <benhdj@cs.cmu.edu> Now the tree nodes carry Symbol
//         instead of int.
// 042101: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef VALUE_HPP
#define VALUE_HPP

#include "symbol.hpp"

#include "tree.hpp"
#include "treeAlgorithm.hpp"

#include <iostream>

namespace UKernel {

using namespace Toolbox;

class Value: public Tree<Symbol> {
  struct _StrictOp {
    bool operator () (Tree<Symbol>::ConstIterator &ti) const {
      return (ti->isSpecial(Symbol::MULTIPLE));
    }
  };
  static TIsomorphismMixed<Symbol,_StrictOp> isoMFunc;

  // remove all duplicate atomic or complex values
  void compact (Iterator &iter);

  // ASSUMPTION: in all of the following methods the first iterator arguement should
  // point to some nodes in *this tree

  // reduce to an atom with the 1st daughter's symbol or 'sym'
  // the 1st form promotes the leftmost daughter tree
  void reduceToAtom (Iterator &iter);
  void reduceToAtom (Iterator &iter, const Symbol &sym);

  // filter daughters of iter1 (*OR* value) using iter2; return true if at least 
  // one is left; note iter1 may have boxes.
  // ASSUMPTION: iter1 points to an *OR* and iter2 points to a *NOT*
  bool puDifference (Iterator &iter1, const ConstIterator &iter2, bool update);

  // ASSUMPTION: iter1 points to a non-*MULTIPLE* and iter2 points to a *MULTIPLE*
  bool multiplePU (Iterator &iter1, const ConstIterator &iter2, bool update);

  // set update to false if you don't want to update iter1 (used for equivalence tests)
  // even when update=true iter1 is updated iff the unification succeeds
  bool pseudoUnify (Iterator &iter1, const ConstIterator &iter2, bool update);

  // remove any possible box under iter (inclusive); return true if such removal
  // takes place
  // NOTE: iter is *NOT* compacted
  bool removeBox (Iterator iter);

  friend std::ostream &operator << (std::ostream &os, const Value &v);
  
public:

  // ASSUMPTION: Only for an RHS *OR* value there can be boxes embedded inside 
  // it, in which case the box is always the *first* daughter of the mother 
  // *OR* node.

  // remove duplicate daughters for *OR* and *NOT* - ordering is not reserved
  // if only one element is left for an *OR* make it an atom
  // ASSUMPTION: *this is not empty
  // IMPORTANT: this is only a shallow compaction - it doesn't pull the deeper
  //            complex values up; e.g. it doesn't compact (*or* 0 (*or* 1 2))
  //            to (*or* 0 1 2), or (*not* 0 (*or* 1 2)) to (*not* 0 1 2)
  //            UNLESS there's only one component inside a complex value, e.g.,
  //            (*OR* 1) == 1, (*MULTIPLE* 1) == 1
  void compact () { Iterator rootIter; compact(rootIter=begin()); }

  // merge 'v' in to become an *OR* value
  // CAUTION: may let RHS box values creep into LHS! BE CAREFUL!
  // ASSUMPTION: both *this and v are not empty
  void mergeOr (const Value &v);

  // ASSUMPTION: both *this and v are not empty
  bool pseudoUnify (const Value &v);
  bool test (const Value &v);

  // ASSUMPTION: no box can be in v for push()!
  void push (const Value &v);
  bool top (Value &v);   // return true iff this value is not empty
  bool pop ();           // return false iff after popping this value is empty

  // ASSUMPTION: *this can't be a box
  bool isNumber () const;
  bool isInteger () const;
  bool isPositive () const;

  bool isAtom () const;  // return true if it's an atomic value

  // remove any possible box; return true if such removal takes place
  // NOTE: no value compaction happens automatically here
  bool removeBox () { return removeBox(begin()); }

  // isomorphism tests
  bool operator == (const Value &v) const { return isoMFunc(begin(),v.begin()); }
};

// ============================ inline functions ============================

inline void Value::reduceToAtom (Iterator &iter) {
  assert(iter.size());

  DIterator daughterIter=iter.begin();
  promote(daughterIter);
  iter=daughterIter;
}

inline void Value::reduceToAtom (Iterator &iter, const Symbol &sym) {
  DIterator di;
  
  iter.update(sym);
  for (di=iter.begin();di!=iter.end();di=erase(di));
}

inline bool Value::pseudoUnify (const Value &v) {
  assert(size() && v.size());
  Iterator iter1=begin();
  ConstIterator iter2=v.begin();

  return pseudoUnify(iter1,iter2,true);
}

inline bool Value::test (const Value &v) {
  assert(size() && v.size());
  Iterator iter1=begin();
  ConstIterator iter2=v.begin();

  return pseudoUnify(iter1,iter2,false);
}

inline bool Value::isNumber () const {
  if (size()==1) {
    assert(!begin()->isSpecial(Symbol::BOX));
    return begin()->isNumber();
  }
  else return false;
}

inline bool Value::isInteger () const {
  if (size()==1) {
    assert(!begin()->isSpecial(Symbol::BOX));
    return begin()->isInteger();
  }
  else return false;
}

inline bool Value::isPositive () const {
  if (size()==1) {
    assert(!begin()->isSpecial(Symbol::BOX));
    return begin()->isPositive();
  }
  else return false;
}

inline bool Value::isAtom () const {
  return (size()==1 && begin()->readCode()>Symbol::UNKNOWN);
}

inline std::ostream &operator << (std::ostream &os, const Value &v) {
  return os<<(Tree<Symbol>)v;
}

};

#endif
