// ======================================================================
// value.cpp - Values for Feature Structures
// 
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 112002: Benjamin Han <benhdj@cs.cmu.edu> nullTI -> nullTI().
// 101802: Benjamin Han <benhdj@cs.cmu.edu> Added Value::removeBox() to remove
//         possible boxes (e.g. when assigning RHS to LHS).
// 101302: Benjamin Han <benhdj@cs.cmu.edu> Value::compact() now compacts
//         *MULTIPLE* with only one daughter (by simply removing *MULTIPLE*).
// 092802: Benjamin Han <benhdj@cs.cmu.edu> For Value::pseudoUnify() with
//         update=false (for testing, not really doing unification), changed
//         the behavior back to the shallow union set operation.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Changed several int declarations
//         into the rightful size_type; namespace added.
// 091701: Benjamin Han <benhdj@cs.cmu.edu> Fixed a bug that preventing
//         compaction of values with single daughters; fixed another bugs in 
//         value compaction that prevent *MULTIPLE* inside *OR*/*NOT*, and
//         *OR*/*NOT* inside *MULTIPLE from being correctly compared; revised
//         isomorphism detection in pseudoUnify() to use TIsomorphismMixed<>.
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Renamed equal() to test().
// 090301: Benjamin Han <benhdj@cs.cmu.edu> Minor revisions to supress warnings
//         under -pedantic -Wall; moved IndexList typedef from value.hpp to
//         value.cpp; renamed Equivalence*<> to TIsomorphism*<>.
// 080901: Benjamin Han <benhdj@cs.cmu.edu> Fixed a potential bug in 
//         Value::push() - dConstIter points before iter2.begin().
// 080701: Benjamin Han <benhdj@cs.cmu.edu> Renamed dSize() to size() in 
//         _TreeIteratorBase<>; renamed Homomorphic* function objects to
//         Equivalence*.
// 080201: Benjamin Han <benhdj@cs.cmu.edu> Renamed dBegin() and dEnd() to 
//         begin() and end() in _TreeIteratorBase<>.
// 072001: Benjamin Han <benhdj@cs.cmu.edu> Revised using the new ways to
//         declare tree iterators.
// 070901: Benjamin Han <benhdj@cs.cmu.edu> Revised the unification/equivalence
//         tests involving *MULTIPLE* values to accurately model the behavior
//         of the 1988 GenKit; Added Value::multiplePU().
// 070601: Benjamin Han <benhdj@cs.cmu.edu> Changed Value::append() to 
//         Value::push(), and it's now pushing the RHS on top of the LHS.
// 070401: Benjamin Han <benhdj@cs.cmu.edu> 
//         ostream& operator<<(ostream&,const Value&) now calls the function
//         ostream& operator<<(ostream&,const Tree<>&) in treeAlgorithm.hpp;
//         removed PrintValueAgenda.
// 070301: Benjamin Han <benhdj@cs.cmu.edu> Moved EraseList from value.cpp
//         to IndexList in value.hpp.
// 070201: Benjamin Han <benhdj@cs.cmu.edu> Added Value::mergeOr().
// 070101: Benjamin Han <benhdj@cs.cmu.edu> Code cleaning; removed
//         Value::puOrMultiple() - identical to the new Value::puDifference();
//         changed all calls to Tree<>::promote() to Value::reduceToAtom(); 
//         restrict the use of _VIterator and _VConstIterator: now only 
//         Value::compact() and Value::equal() use them.
// 062901: Benjamin Han <benhdj@cs.cmu.edu> Changed the behaviors of
//         unification/equivalence test of two *MULTIPLE* values and two *NOT*
//         values (see Operators.txt for detailed descriptions).
// 062501: Benjamin Han <benhdj@cs.cmu.edu> Fixed Value::compact() - recursive
//         value compaction now added.
// 062401: Benjamin Han <benhdj@cs.cmu.edu> Removed a unnecessary diList.clear()
//         in the initialization of the for-loop of the default case in 
//         Value::pseudoUnify().
// 061401: Benjamin Han <benhdj@cs.cmu.edu> Minor revision of comments.
// 051701: Benjamin Han <benhdj@cs.cmu.edu> Removed the dependency on sstream
//         library.
// 050801: Benjamin Han <benhdj@cs.cmu.edu> Now the tree nodes carry Symbol
//         instead of int.
// 042101: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "value.hpp"

#include <list>

namespace UKernel {

typedef std::list<Value::Iterator::size_type> IndexList;

TIsomorphismMixed<Symbol,Value::_StrictOp> Value::isoMFunc;

void Value::compact (Value::Iterator &iter) {
  // no compaction for atoms
  if (!iter.size()) return;
  
  DIterator iter1,iter2;

  for (iter1=iter.begin();iter1!=iter.end();++iter1) compact(iter1);

  // *MULTIPLE* allows multiple same daughters
  if (iter->isSpecial(Symbol::MULTIPLE)) {
    if (iter.size()==1) reduceToAtom(iter);
    return;
  }
  
  for (iter1=iter.begin();iter1!=iter.end();++iter1)
    for (++(iter2=iter1);iter2!=iter.end();)
      if (isoMFunc(iter1,iter2)) iter2=erase(iter2);
      else ++iter2;

  if (iter.size()==1 && !iter->isSpecial(Symbol::NOT)) reduceToAtom(iter);
}

// ASSUMPTION: iter1 points to an *OR* and iter2 points to a *NOT*
bool Value::puDifference (Value::Iterator &iter1, const Value::ConstIterator &iter2, bool update) {
  DIterator dIter;
  Iterator::size_type i,s=iter1.size();
 
  if (update) {
    bool haveBox=false;
    IndexList eraseList;
    IndexList::const_iterator ei;
 
    // check if we have a leading box
    if ((dIter=iter1.begin())->isSpecial(Symbol::BOX)) {
      i=1;
      haveBox=true;
    }
    else i=0;

    for (;i<s;++i,++dIter)
      if (!pseudoUnify(dIter,iter2,true)) eraseList.push_back(i);

    if (haveBox)
      if (eraseList.size()!=s-1)
	// there is at least one successful unification before: nuke the box
	eraseList.push_back(0);
      else {
	// all of the previous unifications fail: the box becomes iter2
	dIter=iter1.begin();
	insert(dIter,iter2);
	prune(dIter);
      }
    
    if (eraseList.size()!=s) {
      for (ei=eraseList.begin(),i=0,dIter=iter1.begin();ei!=eraseList.end();++ei,++i) {
	for (;i<*ei;++i,++dIter);
	dIter=erase(dIter);
      }
 
      // reduce to an atom if only one daughter is left
      if (iter1.size()==1) reduceToAtom(iter1);
 
      return true;
    }
    else return false;
  }
  else {
    // in case we have a box as the first daughter, return true immediately
    if ((dIter=iter1.begin())->isSpecial(Symbol::BOX)) return true;
    for (i=0;i<s && !pseudoUnify(dIter,iter2,false);++i,++dIter);
    return (i!=s);
  }
}

// ASSUMPTION: iter1 points to a non-*MULTIPLE* and iter2 points to a *MULTIPLE*
bool Value::multiplePU (Value::Iterator &iter1, const Value::ConstIterator &iter2, bool update) {
  assert(!iter1->isSpecial(Symbol::MULTIPLE));
  assert(iter2->isSpecial(Symbol::MULTIPLE));

  ConstDIterator dConstIter;

  if (update) {
    DIterator newDIter,dIter,dBackup=iter1.begin();

    for (dConstIter=iter2.begin();dConstIter!=iter2.end();++dConstIter) {
      // copy the entire subtree rooted at iter1 to the first daughter of
      // iter1 - DO NOT copy the new daughter, i.e., do not dive into 
      // infinite recursion
      newDIter=insert(iter1.begin(),*iter1);
      for (dIter=dBackup;dIter!=iter1.end();++dIter)
	insert(newDIter.end(),dIter);
      
      if (!pseudoUnify(newDIter,dConstIter,true)) {
	for (dIter=iter1.begin();dIter!=dBackup;dIter=erase(dIter));
	return false;
      }
    }

    for (dIter=dBackup;dIter!=iter1.end();dIter=erase(dIter));
    iter1.update(symMultiple);
    return true;
  }
  else {
    for (dConstIter=iter2.begin();dConstIter!=iter2.end();++dConstIter)
      if (!pseudoUnify(iter1,dConstIter,false)) return false;
    return true;
  }
}

bool Value::pseudoUnify (Value::Iterator &iter1, const Value::ConstIterator &iter2, bool update) {
  DIterator dIter;
  ConstDIterator dConstIter;

  int code1=iter1->readCode();
  int code2=iter2->readCode();

  // ASSUMPTION: Only for an RHS *OR* value there can be boxes embedded inside 
  // it, in which case the box is always the *first* daughter of the mother 
  // *OR* node.

  switch (code1) {
    // -------------------- LHS is *NOT* --------------------
  case Symbol::NOT:

    switch (code2) {
    case Symbol::NOT:                         // * "shallow" set union *

      if (update) {
	for (dConstIter=iter2.begin();dConstIter!=iter2.end();++dConstIter)
	  insert(iter1.end(),dConstIter);
	compact(iter1);
      }
      return true;

      // 20020928: disable the "superset" testing version for the 
      //           non-destructive version
//       this is the disabled part if update==false:
// 	// only return true iff for every RHS daughter value at least one of 
// 	// the LHS daughter value is equal to it
// 	for (dConstIter=iter2.begin();dConstIter!=iter2.end();++dConstIter) {
// 	  for (dIter=iter1.begin();
// 	       dIter!=iter1.end() && !pseudoUnify(dIter,dConstIter,false);
// 	       ++dIter);
// 	  if (dIter==iter1.end()) return false;
// 	}
// 	return true;
      
    case Symbol::OR: {
      bool result;
      Value newV;
      Iterator vIter;
      
      // newV is an *OR* value
      newV.insert(newV.begin(),iter2);
      vIter=newV.begin();

      // box is dealt within puDifference()
      if ((result=newV.puDifference(vIter,iter1,update))==true && update)
	if (iter1.isRoot()) insert(iter1,vIter);
	else {
	  ++(dIter=iter1);
	  prune(iter1);
	  insert(dIter,vIter);
	}
      return result;
    }
      
    case Symbol::MULTIPLE:
      return multiplePU(iter1,iter2,update);
      
    default:
      for (dIter=iter1.begin();dIter!=iter1.end();++dIter)
	if (pseudoUnify(dIter,iter2,false)) return false;
      
      if (update) reduceToAtom(iter1,*iter2);
      return true;
    }
    
    // -------------------- LHS is *OR* --------------------
  case Symbol::OR:
    
    switch (code2) {
    case Symbol::NOT:                         // * set difference *
      return puDifference(iter1,iter2,update);
      
    case Symbol::OR: {                        // * set intersection (almost) *
      bool haveBox=iter2.begin()->isSpecial(Symbol::BOX);

      if (update) {
	DIterator di,newDI,dBackup=iter1.begin();
	ConstDIterator rhsStart;

	// this excludes the leading RHS daughter from unifications if it's a box	
	rhsStart=iter2.begin();
	if (haveBox) ++rhsStart;
	
	// for every LHS daughter, unify each and every RHS daughter with it
	for (dIter=dBackup;dIter!=iter1.end();++dIter) {
	  newDI=insert(dBackup,dIter);
	  dConstIter=rhsStart;
	  do {
	    if (pseudoUnify(newDI,dConstIter,true)) {
	      // check to see if newDI is isomorphic to any of the
	      // new daughter values (by default isoMFunc() does strict
	      // tests first)
	      for (di=iter1.begin();di!=newDI && !isoMFunc(newDI,di);++di);
	      if (di!=newDI) prune(newDI);
	      
	      if (++dConstIter!=iter2.end())
		// prepare the next fresh copy of the same LHS daughter
		newDI=insert(dBackup,dIter);
	      else break;
	    }
	    // unification fails
	    else if (++dConstIter==iter2.end()) {
	      prune(newDI);
	      break;
	    }
	  } while (true);
	}
	
	if (dBackup!=iter1.begin()) {
	  // unification succeeded - erase the old value
	  for (dIter=dBackup;dIter!=iter1.end();dIter=erase(dIter));
	  if (iter1.size()==1) reduceToAtom(iter1);
	  return true;
	}
	// if all unifications fail, return true iff we have a box
	else return haveBox;
      }
      else {	
	if (haveBox) return true;  // a leading RHS box shortcuts the operation

	for (dIter=iter1.begin();dIter!=iter1.end();++dIter)
	  for (dConstIter=iter2.begin();dConstIter!=iter2.end();++dConstIter)
	    if (pseudoUnify(dIter,dConstIter,false)) return true;
      }
      
      return false;
    }
      
    case Symbol::MULTIPLE:
      return multiplePU(iter1,iter2,update);      

    default:                                  // * atom *
      for (dIter=iter1.begin();dIter!=iter1.end();++dIter)
	if (pseudoUnify(dIter,iter2,false)) {
	  if (update) reduceToAtom(iter1,*iter2);
	  return true;
	}
      return false;
    }
    
    // -------------------- LHS is *MULTIPLE* --------------------
  case Symbol::MULTIPLE: {
    DIterator dBackup=iter1.begin(),newDI;

    // we don't need to worry about the possible RHS boxes here - they are delegated
    // to individual pseudoUnify() calls

    if (code2!=Symbol::MULTIPLE)
      // the RHS is NOT a *MULTIPLE*
      if (update) {
	for (dIter=dBackup;dIter!=iter1.end();++dIter) {
	  newDI=insert(iter1.begin(),dIter);
	  if (!pseudoUnify(newDI,iter2,true)) {
	    for (dIter=iter1.begin();dIter!=dBackup;dIter=erase(dIter));
	    return false;
	  }
	}
	
	for (dIter=dBackup;dIter!=iter1.end();dIter=erase(dIter));
	return true;
      }
      else {
	for (dIter=iter1.begin();dIter!=iter1.end();++dIter)
	  if (!pseudoUnify(dIter,iter2,false)) return false;
	return true;
      }
    else
      // the RHS is a *MULTIPLE*
      if (update) {
	for (dIter=dBackup;dIter!=iter1.end();++dIter)
	  for (dConstIter=iter2.begin();dConstIter!=iter2.end();++dConstIter) {
	    newDI=insert(iter1.begin(),dIter);
	    if (!pseudoUnify(newDI,dConstIter,true)) {
	      for (dIter=iter1.begin();dIter!=dBackup;dIter=erase(dIter));
	      return false;
	    }
	  }
	      
	for (dIter=dBackup;dIter!=iter1.end();dIter=erase(dIter));
	return true;
      }
      else {
	for (dIter=iter1.begin();dIter!=iter1.end();++dIter)
	  for (dConstIter=iter2.begin();dConstIter!=iter2.end();++dConstIter)
	    if (!pseudoUnify(dIter,dConstIter,false)) return false;
	return true;
      }
  }

    // -------------------- LHS is an atom --------------------
  default:

    switch (code2) {
    case Symbol::NOT:
      for (dConstIter=iter2.begin();dConstIter!=iter2.end();++dConstIter)
	if (pseudoUnify(iter1,dConstIter,false)) return false;
      return true;

    case Symbol::OR: {                        // * atom *
      // a leading RHS box shortcuts the operation
      if ((dConstIter=iter2.begin())->isSpecial(Symbol::BOX)) return true;

      for (;dConstIter!=iter2.end();++dConstIter)
	if (pseudoUnify(iter1,dConstIter,false)) return true;
      return false;
    }
      
    case Symbol::MULTIPLE:
      return multiplePU(iter1,iter2,update);
      
    default:                                  // * atom *
      return (*iter1==*iter2);
    }
  }
}

// INTERNAL USE: for traversing each and every box
template <class T, class Ref, class Ptr, class TNode, class Cmp>
struct _BoxAgenda {
  _TreeIteratorBase<T,Ref,Ptr,TNode,Cmp> start;
  
  void init (_TreeIteratorBase<T,Ref,Ptr,TNode,Cmp> &i) {
    if (!(start=i)->isSpecial() || i->isSpecial(Symbol::NOT)) {
      // impossible to have boxes
      i=i.nullTI();
      return;
    }

    _TreeDIterator<T,Ref,Ptr,TNode,Cmp> di=i.begin();
    do {
      switch (di->readCode()) {
      case Symbol::OR:
      case Symbol::MULTIPLE: di=di.begin(); break;

      case Symbol::BOX: i=di; return;

      default:  
	// for *NOT* and atoms: move to a node to the right
	while (++di==di.nullTI())
	  if ((di=di.up())==start) {
	    i=i.nullTI();
	    return;
	  }
      }
    } while (true);
  }
  
  void visitNext (_TreeIteratorBase<T,Ref,Ptr,TNode,Cmp> &i) {
    _TreeDIterator<T,Ref,Ptr,TNode,Cmp> di=i;

    // move to a node to the right
    while (++di==di.nullTI())
      if ((di=di.up())==start) {
	i=i.nullTI();
	return;
      }

    do {
      switch (di->readCode()) {
      case Symbol::OR:
      case Symbol::MULTIPLE: di=di.begin(); break;

      case Symbol::BOX: i=di; return;

      default:  
	// for *NOT* and atoms: move to a node to the right
	while (++di==di.nullTI())
	  if ((di=di.up())==start) {
	    i=i.nullTI();
	    return;
	  }
      }
    } while (true);
  }

  void remove (const _TreeIteratorBase<T,Ref,Ptr,TNode,Cmp> &ti) {}
};

bool Value::removeBox (Iterator iter) {
  // to workaround "macro uses ',' as delimiter" problem
  typedef _TreeNode<Symbol,std::less<Symbol> > VNode;
  TreeIterator(Symbol,_BoxAgenda,VNode,std::less<Symbol>) bi;
  bool result=false;

  for (bi=iter;bi!=bi.nullTI();result=true,bi=erase(bi));
  return result;
}

// almost identical to Value::push()
void Value::mergeOr (const Value &v) {
  assert(size() && v.size());

  DIterator iter1=begin();
  ConstIterator iter2=v.begin();

  // if the LHS is not an *OR* value make it so
  if (!iter1->isSpecial(Symbol::OR)) {
    demote(iter1,symOr);
    iter1.up();
  }

  // insert the RHS after the last daughter value
  if (iter2->isSpecial(Symbol::OR))
    for (ConstDIterator dConstIter=iter2.begin();
	 dConstIter!=iter2.end();++dConstIter) insert(iter1.end(),dConstIter);
  else insert(iter1.end(),iter2);  

  compact(iter1);
}

// ASSUMPTION: no box can be in v here!
void Value::push (const Value &v) {
  DIterator iter1=begin();
  ConstIterator iter2=v.begin();
  Iterator iter;

  if (!iter1->isSpecial(Symbol::MULTIPLE)) {
    demote(iter1,symMultiple);
    iter1.up();
  }
  
  if (iter2->isSpecial(Symbol::MULTIPLE)) {
    size_type i;
    ConstDIterator dConstIter;

    for (i=iter2.size(),dConstIter=iter2.end();i;--i)
      iter=insert(iter1.begin(),--dConstIter);
  }
  else iter=insert(iter1.begin(),iter2);
}

bool Value::top (Value &v) {
  Iterator iter=begin();

  if (iter->isSpecial(Symbol::MULTIPLE)) {
    v.clear();
    v.insert(v.begin(),iter.begin());
  }
  else v=*this;

  return (size()!=0);
}

bool Value::pop () {
  Iterator iter=begin();

  if (iter->isSpecial(Symbol::MULTIPLE)) {
    if (iter.size()==1) clear();
    else {
      prune(iter.begin());	  
      if (iter.size()==1) reduceToAtom(iter);
    }
  }
  else clear();

  return (size()!=0);
}

};
