// ======================================================================
// ushHelp.cpp - Printing version and help messages for UShell.
// 
// 061504: Benjamin Han <benhdj@cs.cmu.edu> Copyright notice changed to 2004.
// 092802: Benjamin Han <benhdj@cs.cmu.edu> Added help info for new operators
//         =c', =t', and =i', and modified some help info for the existing
//         operators.
// 091802: Benjamin Han <benhdj@cs.cmu.edu> Silly copyright year update in
//         UShell::printVersion().
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Added help info for the new 
//         isomorphism operator "=i" (Symbol::OP_ISO).
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Changed "=c" to "=t".
// 081601: Benjamin Han <benhdj@cs.cmu.edu> Made the kernel commands case-
//         sensitive.
// 070601: Benjamin Han <benhdj@cs.cmu.edu> Changed all references to 'append'
//         to 'push'.
// 062201: Benjamin Han <benhdj@cs.cmu.edu> Added a new shell command
//         "SIZE [n]"; revised the "Implementation note" section of the help
//         message to indicate the lack of the support of the lexical lookup
//         equations; fixed the case porblem of printHelp().
// 062201: Benjamin Han <benhdj@cs.cmu.edu> Added a new shell command
//         "INDENT [ON|OFF]".
// 061201: Benjamin Han <benhdj@cs.cmu.edu> Now (FS path) =c (FS path) is
//         allowed.
// 052001: Benjamin Han <benhdj@cs.cmu.edu> Minor revisions.
// 051701: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "ush.hpp"

using namespace UKernel;

void UShell::printVersion () const {
  out<<"UShell - an interactive shell for the Unification Kernel (UKernel) v"<<VERSION<<endl;
  out<<"(c) 2004 Benjamin Han <benhdj@cs.cmu.edu>"<<endl;
  out<<"Type \"!help\" to list the commands available."<<endl<<endl;
}

void UShell::printHelp (const TokenString &tokStr) const {
  TokenString::size_type s=tokStr.size();

  if (s==1) {
    // print a list of all commands
    out<<PROMPT_SYS<<"Complete list of UShell commands:"<<endl;
    out<<"  (FS=Feature Structure; v=Value; LHS=Left-hand Side; RHS=Right-hand Side)"<<endl<<endl;
    
    out<<"----------- Shell commands (case-insensitive; with a prefix \'"<<SHELL_CMD_PREFIX<<"\') -----------"<<endl;
    out<<PROMPT_SYS<<"help [cmd/operator]: description of commands/operators."<<endl;
    out<<PROMPT_SYS<<"quit | exit | bye | ctrl-c | ctrl-d: Exit UShell."<<endl;
    out<<PROMPT_SYS<<"show [fs1] [fs2]...: show the content of feature structure(s)."<<endl;
    out<<PROMPT_SYS<<"indent [on|off]: turn on/off or show indentation printing status."<<endl;
    out<<PROMPT_SYS<<"size [n]: set the size of the FS registers to \'n\'."<<endl<<endl;
    
    out<<"-------------- Kernel commands (case-sensitive) --------------"<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) = v | FS | (FS path): pseudo-unification."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) =c v | FS | (FS path): constraint test."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) =c' v | FS | (FS path): constraint test (no filtering)."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) =t v | FS | (FS path): equivalence test."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) =t' v | FS | (FS path): equivalence test (no filtering)."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) =i v | FS | (FS path): isomorphism test."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) =i' v | FS | (FS path): isomorphism test (no filtering)."<<endl;

    out<<PROMPT_SYS<<"FS | (FS path) <= v | FS | (FS path): assign RHS to LHS."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) == FS | (FS path): removal assign RHS to LHS."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) > FS | (FS path): push RHS to LHS."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) < FS | (FS path): pop the 1st element of RHS to LHS."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) = *DEFINED*: test if LHS is defined."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) = *UNDEFINED*: test if LHS is undefined."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) = *REMOVE*: remove LHS."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) = *NUMBER*: test if LHS is a real number."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) = *INTEGER*: test if LHS is an integer."<<endl;
    out<<PROMPT_SYS<<"FS | (FS path) = *POSITIVE*: test if LHS is a positive real number."<<endl<<endl;
    
    out<<PROMPT_SYS<<"A value can be either atomic or complex:"<<endl;
    out<<"  "<<"Atomic: e.g. apple, banana, 12, etc."<<endl;
    out<<"  "<<"Complex: e.g. (*NOT* apple banana), (*OR* apple banana),"<<endl;
    out<<"  "<<"         (*MULTIPLE* apple, banana), or any possible embeddings."<<endl<<endl;
    
    out<<PROMPT_SYS<<"An FS must be named as X0, X1, ... etc - the name is used to refer"<<endl;
    out<<"  "<<"to the corresponding FS register. You can increase the size of the"<<endl;
    out<<"  "<<"FS registers by using shell command \'size\'"<<endl<<endl;
    
    out<<PROMPT_SYS<<"A path is a list of feature names delimited by space."<<endl<<endl;
    
    out<<PROMPT_SYS<<"Implementation note: so far UShell doesn't support equation blocks"<<endl;
    out<<"  "<<"(multiple equations grouped by *OR*, *EOR*, *CASE*, or a pair of"<<endl;
    out<<"  "<<"parentheses - in the case it's an implicit *AND*), and all lexical"<<endl;
    out<<"  "<<"lookup equations."<<endl;
  }
  else {
    string cmd=tokStr[1];
    
    if (cmd[0]==SHELL_CMD_PREFIX) cmd.erase(0,1);
    
    if (cmd=="HELP") {
      out<<PROMPT_SYS<<"help [cmd/operator]: description of commands/operators."<<endl<<endl;
      out<<"Without supplying the argument 'cmd/operator' the entire list of "<<endl;
      out<<"commands/operators will be shown; supply 'cmd/operator' to show the "<<endl;
      out<<"specific info about that command/operator. Note for shell commands you"<<endl;
      out<<"don't need to add the shell command prefix to get the help."<<endl;
    }
    else if (cmd=="QUIT" || cmd=="EXIT" || cmd=="BYE") {
      out<<PROMPT_SYS<<"quit | exit | bye | ctrl-c | ctrl-d: Exit UShell."<<endl<<endl;
      out<<"No way you need more info on this one!"<<endl;
    }
    else if (cmd=="SHOW") {
      out<<PROMPT_SYS<<"show [fs1] [fs2]...: show the content of feature structure(s)."<<endl<<endl;
      out<<"Without supplying the arguments all FS defined will be listed; supply a list"<<endl;
      out<<"of FS names to show only those FS."<<endl;
    }
    else if (cmd=="INDENT") {
      out<<PROMPT_SYS<<"indent [on|off]: turn on/off or show indentation printing status."<<endl<<endl;
      out<<"Without supplying the argument the current indentation printing status"<<endl;
      out<<"is shown."<<endl;
    }
    else if (cmd=="SIZE") {
      out<<PROMPT_SYS<<"size [n]: set the size of the FS registers to \'n\'."<<endl<<endl;
      out<<"Without supplying the arguments the current size is displayed."<<endl;
    }
    else if (cmd=="=") {
      out<<PROMPT_SYS<<"FS | (FS path) = v | FS | (FS path): pseudo-unification."<<endl;
      out<<PROMPT_SYS<<"FS | (FS path) = *DEFINED*: test if LHS is defined."<<endl;
      out<<PROMPT_SYS<<"FS | (FS path) = *UNDEFINED*: test if LHS is undefined."<<endl;
      out<<PROMPT_SYS<<"FS | (FS path) = *REMOVE*: remove LHS."<<endl;
      out<<PROMPT_SYS<<"FS | (FS path) = *NUMBER*: test if LHS is a real number."<<endl;
      out<<PROMPT_SYS<<"FS | (FS path) = *INTEGER*: test if LHS is an integer."<<endl;
      out<<PROMPT_SYS<<"FS | (FS path) = *POSITIVE*: test if LHS is a positive real number."<<endl<<endl;
      
      out<<"If RHS is a constraint value (the one surrounded by a pair of \'*\'s)"<<endl;
      out<<"the corresponding tests/removal will be executed."<<endl;
      out<<"If RHS is not a constraint value this command peudo-unifies both sides - only"<<endl;
      out<<"LHS is updated iff the unification succeeds."<<endl;
      out<<"If the LHS FS has not been defined it will be created if the equation succeeds."<<endl;
      out<<"This command returns true iff unification/tests/removal succeeds."<<endl;
    }
    else if (cmd=="=c") {
      out<<PROMPT_SYS<<"FS | (FS path) =c v | FS | (FS path): constraint test."<<endl<<endl;
      out<<"This command returns true iff the LHS is defined and is isomorphic to"<<endl;
      out<<"the RHS up to the unifiability of values; for an *OR* FS the branches that"<<endl;
      out<<"do not pass the tests will be filtered."<<endl;
    }
    else if (cmd=="=c'") {
      out<<PROMPT_SYS<<"FS | (FS path) =c' v | FS | (FS path): constraint test (no filtering)."<<endl<<endl;
      out<<"This command returns true iff the LHS is defined and is isomorphic to"<<endl;
      out<<"the RHS; NOTE the LHS will NOT be modified after the operation."<<endl;
    }
    else if (cmd=="=t") {
      out<<PROMPT_SYS<<"FS | (FS path) =t v | FS | (FS path): equivalence test."<<endl<<endl;
      out<<"This command returns true iff the LHS is defined and is unifiable with"<<endl;
      out<<"the RHS (but NO unification takes place); for an *OR* FS the branches that"<<endl;
      out<<"do not pass the tests will be filtered."<<endl;
    }
    else if (cmd=="=t'") {
      out<<PROMPT_SYS<<"FS | (FS path) =t' v | FS | (FS path): equivalence test (no filtering)."<<endl<<endl;
      out<<"This command returns true iff the LHS is defined and is unifiable with"<<endl;
      out<<"the RHS (but NO unification takes place); NOTE the LHS is NOT modified."<<endl;
    }
    else if (cmd=="=i") {
      out<<PROMPT_SYS<<"FS | (FS path) =i v | FS | (FS path): isomorphism test."<<endl<<endl;
      out<<"This command returns true iff the LHS is isomorphic to the RHS (the siblings"<<endl;
      out<<"ordering is significant iff they're under a *MULTIPLE* FS; for an *OR* FS "<<endl;
      out<<"the branches that do not pass the tests will be filtered."<<endl;
    }
    else if (cmd=="=i'") {
      out<<PROMPT_SYS<<"FS | (FS path) =i' v | FS | (FS path): isomorphism test (no filtering)."<<endl<<endl;
      out<<"This command returns true iff the LHS is isomorphic to the RHS (the siblings"<<endl;
      out<<"ordering is significant iff they're under a *MULTIPLE* FS; NOTE the LHS is NOT"<<endl;
      out<<"modified."<<endl;
    }
    else if (cmd=="<=") {
      out<<PROMPT_SYS<<"FS | (FS path) <= v | FS | (FS path): assign RHS to LHS."<<endl<<endl;
      out<<"If the LHS FS has not been defined it will be created if the equation succeeds."<<endl;
      out<<"If RHS is an empty value/FS this is equivalent to FS | (FS Path) = *REMOVE*."<<endl;
      out<<"This command always returns true."<<endl;
    }
    else if (cmd=="==") {
      out<<PROMPT_SYS<<"FS | (FS path) == FS | (FS path): removal assign RHS to LHS."<<endl<<endl;
      out<<"Get a value from RHS, unify it with LHS, and remove it from RHS."<<endl;
      out<<"If the LHS FS has not been defined it will be created if the equation succeeds."<<endl;
      out<<"This command returns true iff the entire process is completed without failure."<<endl;
    }
    else if (cmd==">") {
      out<<PROMPT_SYS<<"FS | (FS path) > FS | (FS path): push RHS to LHS."<<endl<<endl;
      out<<"Get a value from RHS and push it to LHS (i.e., the new element is at the"<<endl;
      out<<"end); LHS will become a *MULTIPLE* value if it is not before this operation."<<endl;
      out<<"If the LHS FS has not been defined it will be created if the equation succeeds."<<endl;
      out<<"This command returns true iff RHS exists."<<endl;
    }
    else if (cmd=="<") {
      out<<PROMPT_SYS<<"FS | (FS path) < FS | (FS path): pop the 1st element of RHS to LHS."<<endl<<endl;
      out<<"Pop the 1st value from RHS and unify it with LHS; if RHS has no value left"<<endl;
      out<<"after the operation the RHS will be removed."<<endl;
      out<<"This command returns true iff RHS exists and the unification succeeds."<<endl;
    }
    else
      out<<PROMPT_ERR<<"Command \'"<<cmd<<"\' is not currently supported."<<endl;
  }
}
