// ======================================================================
// ush.hpp - UShell: an interactive shell for UKernel
// 
// 061404: Benjamin Han <benhdj@cs.cmu.edu> Added "using namespace std";
//         added operator =t', =c' and =i' into opVec.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Changed several int declarations
//         into the rightful size_type; namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Added new isomorphism operator
//         "=i" (Symbol::OP_ISO).
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Changed Symbol::OP_EQUAL to
//         Symbol::OP_TEST; changed calls to FStruc::equal() to FStruc::test();
//         added constraint operator "=c" (the old "=c" is changed to "=t").
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Minor revisions after implementing
//         the extension functions.
// 070901: Benjamin Han <benhdj@cs.cmu.edu> Chagned Symbol::OP_APPEND to
//         Symbol::OP_PUSH.
// 063001: Benjamin Han <benhdj@cs.cmu.edu> Revised after changing the way
//         an FS is addressed in initializing equations: removed FSDict - now
//         the FS name must be X0, X1, ... etc.
// 052801: Benjamin Han <benhdj@cs.cmu.edu> Changed to all uppercase
//         printing/command matching.
// 052001: Benjamin Han <benhdj@cs.cmu.edu> Revised after adding eBlock.*.
// 051701: Benjamin Han <benhdj@cs.cmu.edu> Removed operator << for Path.
// 051501: Benjamin Han <benhdj@cs.cmu.edu> Major rewrite.
// 042101: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef USH_HPP
#define USH_HPP

#include "eBlock.hpp"

#include "tokStr.hpp"

#include <iostream>

using namespace std;

namespace UKernel {

enum ConvErrCode {CONV_SUCC,CONV_LPAREN,CONV_RPAREN,CONV_SPECIAL,
		  CONV_NAKED,CONV_PAREN,CONV_FS_RANGE,CONV_FS_NAME};

#define SHELL_CMD_PREFIX '!'

#define PROMPT_CMD    "> "
#define PROMPT_SYS    "* "
#define PROMPT_ERR    ">> "
#define PROMPT_DBG    "- "

// INTERNAL USE: An entry for UShell::FSDict
struct _FSEntry {
  string label;
  FStruc fs;  
  
  _FSEntry (const string &label):label(label),fs(symNull) {}

  operator const string &() const { return label; }
};

// INTERNAL USE: An entry for UShell::OpVec
struct _OpEntry {
  string str;
  Symbol::SymCode code;

  _OpEntry (const string &str, Symbol::SymCode code):str(str),code(code) {}
};

class UShell {
  typedef map<string,Symbol::SymCode> SpecialVMap;
  typedef vector<_OpEntry> OpVec;
  typedef TokenDict<_FSEntry> FSDict;

  istream &in;
  ostream &out;
  
  SpecialVMap svMap;
  OpVec opVec;
  
  FSRegisters fsRegs;
  EBlockMain eb;

  // return true and update fsIdx iff tok is a valid FS name
  bool isValidFSName (const string &tok, FSRegisters::size_type &fsIdx) const;

  // return true iff rhs is a value; if the value is a special value,
  // sym is set to that special value, otherwise sym is UNKNOWN
  bool isValue (const TokenString &rhs, Symbol &sym);

  // return CONV_SUCC iff no syntax error; if the FS is in fsRegs, fsIdx is 
  // updated; otherwise if the FS is out of the range, CONV_FS_RANGE is
  // returned, or if the FS name is ill-formed (must be in the form of X0,
  // X1, ...) CONV_FS_NAME is returned.
  ConvErrCode convertPath (const TokenString &pathStr, FSRegisters::size_type &fsIdx, Path &path);
  ConvErrCode convertValue (const TokenString &rhs, Value &v);

  void printVersion () const;
  void printHelp (const TokenString &tokStr) const;

  // execute shell- & kerenel-level commands
  bool shellCmd (const string &cmd);         // return true if exiting
  void kernelCmd (string &cmd);

public:
  
  UShell (istream &in, ostream &out);

  void run ();
};

// ============================ inline functions ============================

inline UShell::UShell (istream &in, ostream &out):in(in),out(out) {
  string str;

  // set up a map for special value lookup
  svMap.insert(make_pair(str="*NOT*",Symbol::NOT));
  svMap.insert(make_pair(str="*OR*",Symbol::OR));
  svMap.insert(make_pair(str="*MULTIPLE*",Symbol::MULTIPLE));
  svMap.insert(make_pair(str="*REMOVE*",Symbol::REMOVE));
  svMap.insert(make_pair(str="*UNDEFINED*",Symbol::UNDEFINED));
  svMap.insert(make_pair(str="*DEFINED*",Symbol::DEFINED));
  svMap.insert(make_pair(str="*POSITIVE*",Symbol::POSITIVE));
  svMap.insert(make_pair(str="*INTEGER*",Symbol::INTEGER));
  svMap.insert(make_pair(str="*NUMBER*",Symbol::NUMBER));
  
  // set up a vector for operator lookup - from the longest to the shortest
  opVec.push_back(_OpEntry("=t'",Symbol::OP_TEST_NOFILTER));
  opVec.push_back(_OpEntry("=c'",Symbol::OP_CONSTRAIN_NOFILTER));
  opVec.push_back(_OpEntry("=i'",Symbol::OP_ISO_NOFILTER));
  opVec.push_back(_OpEntry("<=",Symbol::OP_ASSIGN));
  opVec.push_back(_OpEntry("==",Symbol::OP_REMOVE_ASSIGN));
  opVec.push_back(_OpEntry("=t",Symbol::OP_TEST));
  opVec.push_back(_OpEntry("=c",Symbol::OP_CONSTRAIN));
  opVec.push_back(_OpEntry("=i",Symbol::OP_ISO));
  opVec.push_back(_OpEntry("=",Symbol::OP_PSEUDO_UNIFY));
  opVec.push_back(_OpEntry(">",Symbol::OP_PUSH));
  opVec.push_back(_OpEntry("<",Symbol::OP_POP));
}

};

#endif
