// ======================================================================
// ush.cpp - UShell: an interactive shell for UKernel
// 
// 061404: Benjamin Han <benhdj@cs.cmu.edu> UShell::isValidFSName() - cast fsIdx
//         to int&.
// 072102: Benjamin Han <benhdj@cs.cmu.edu> Added "default:" statement to
//         several switch statements to suppress gcc 3.1 warnings (about
//         some enum values not handled).
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Changed several int declarations
//         into the rightful size_type; namespace added.
// 120501: Benjamin Han <benhdj@cs.cmu.edu> Bugfix: detecting when LHS or
//         RHS is missing in a kernel command.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Added new isomorphism operator
//         "=i" (Symbol::OP_ISO); fixed the bug which prevented UShell from
//         showing the help messages of operators =c, =t and =i.
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Changed Symbol::OP_EQUAL to
//         Symbol::OP_TEST; changed "=c" to "=t"; added constraint operator "=c".
// 090301: Benjamin Han <benhdj@cs.cmu.edu> Minor revisions to supress warnings
//         under -pedantic -Wall.
// 081601: Benjamin Han <benhdj@cs.cmu.edu> Made the kernel commands case-
//         sensitive.
// 080201: Benjamin Han <benhdj@cs.cmu.edu> Renamed dBegin() and dEnd() to 
//         begin() and end() in _TreeIteratorBase<>.
// 072001: Benjamin Han <benhdj@cs.cmu.edu> Revised using the new ways to
//         declare tree iterators.
// 071701: Benjamin Han <benhdj@cs.cmu.edu> Fixed the bug when assigning an
//         FS with 'c' without space character - it was interpreted as an
//         operator '=c'.
// 070901: Benjamin Han <benhdj@cs.cmu.edu> Chagned Symbol::OP_APPEND to
//         Symbol::OP_PUSH.
// 063001: Benjamin Han <benhdj@cs.cmu.edu> Revised after changing the way
//         an FS is addressed in initializing equations: removed FSDict - now
//         the FS name must be X0, X1, ... etc; added a new shell command
//         "SIZE [n]".
// 062201: Benjamin Han <benhdj@cs.cmu.edu> Added a new shell command
//         "INDENT [ON|OFF]".
// 061401: Benjamin Han <benhdj@cs.cmu.edu> Revised due to the change of
//         the prototype of the type 3 constructor of EBlockMain.
// 061201: Benjamin Han <benhdj@cs.cmu.edu> Now (FS path) =c (FS path) is
//         allowed.
// 052801: Benjamin Han <benhdj@cs.cmu.edu> Now we have indented printing
//         of an FStruc thanks to indent.* in Toolbox; changed to all upper-
//         case printing/command matching.
// 052001: Benjamin Han <benhdj@cs.cmu.edu> Revised after adding eBlock.*;
//         better error checking for operator usage.
// 051501: Benjamin Han <benhdj@cs.cmu.edu> Major rewrite.
// 042101: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "ush.hpp"

#include <ctype.h>

using namespace UKernel;

#define DEFAULT_FS_REGS_SIZE 20

bool UShell::isValidFSName (const string &tok, FSRegisters::size_type &fsIdx) const {
  if (tok[0]=='X' && isInteger(tok.substr(1),(int&)fsIdx)) return true;
  else return false;
}

bool UShell::isValue (const TokenString &rhs, Symbol &sym) {
  TokenString::size_type i=0;
  FSRegisters::size_type fsIdx;

  if (rhs[0]=="(")
    if (rhs.size()>1) i=1;
    else return false;                             // only a '('?
  if (isValidFSName(rhs[i],fsIdx)) return false;   // it's an FS
  else {
    // presumably it's a value - but is it a special value?
    SpecialVMap::const_iterator svi=svMap.find(rhs[i]);
    if (svi!=svMap.end()) sym.setSpecial(svi->second);
    else sym.setSpecial(Symbol::UNKNOWN);
    if (i==1)
      // we have parentheses - must be a complex value
      return sym.isSpecial();
    else return rhs.size()==1;  // must be an atomic value then
  }
}

ConvErrCode UShell::convertPath (const TokenString &pathStr, FSRegisters::size_type &fsIdx, Path &path) {
  TokenString::size_type i,s;

  s=pathStr.size()-1;
  if (pathStr[0]=="(") {
    if (pathStr.find("(",1)!=TokenString::npos || 
	pathStr.find(")",1)!=s) return CONV_PAREN;
    i=1;
  }
  else {
    if (s) return CONV_PAREN;  // only an FS name is allowed
    i=0;
  }
  
  if (isValidFSName(pathStr[i],fsIdx)) {
    if (fsIdx>=fsRegs.size()) return CONV_FS_RANGE;
  }
  else return CONV_FS_NAME;
  
  for (++i,path.clear();i<s;++i)
    path.push_back(Feature(Symbol(pathStr[i])));
  
  return CONV_SUCC;
}

ConvErrCode UShell::convertValue (const TokenString &rhs, Value &v) {
  Value::Iterator iter,tmpIter;
  stack<Value::Iterator> iterStack;
  TokenString::size_type i,s;
  string tok;
  Symbol sym;
  SpecialVMap::const_iterator svi;
  bool first=false;

  i=0;
  s=rhs.size();
  do {
    tok=rhs[i++];
    
    if (tok=="(") first=true;
    else if (tok==")")
      if (!iterStack.empty()) {
	do {
	  tmpIter=iterStack.top();
	  iterStack.pop();
	} while (tmpIter!=iter);
	
	iter.up();
      }
      else return CONV_LPAREN;
    else {
      if ((svi=svMap.find(tok))==svMap.end()) sym.set(tok);
      else sym.setSpecial(svi->second);
      
      if (first)
	if (sym.isSpecial()) {
	  if (v.size())
	    iterStack.push(iter=v.insert(iter.end(),sym));
	  else
	    iterStack.push(iter=v.insert(v.begin(),sym));
	  
	  first=false;
	}
	else return CONV_SPECIAL;
      else
	if (!sym.isSpecial())
	  if (v.size())
	    iterStack.push(v.insert(iter.end(),sym));
	  else if (s==1) v.insert(v.begin(),sym);    // singleton value
	  else return CONV_NAKED;
	else return CONV_SPECIAL;
    }      
  } while (i<s);
  
  if (iterStack.empty()) {
    v.compact();
    return CONV_SUCC;
  }
  else return CONV_RPAREN;
}

bool UShell::shellCmd (const string &cmd) {
  TokenString tokStr;
  tokStr.set(cmd);

  TokenString::size_type idx,s=tokStr.size();
  string::size_type i,j;
  FSRegisters::size_type fsIdx,fsSize;
  string first=tokStr[0].substr(1);  // remove the SHELL_CMD_PREFIX

  // uppercase 'first'
  for (i=0,j=first.length();i<j;++i) first[i]=toupper(first[i]);

  if (first=="QUIT" || first=="EXIT" || first=="BYE") return true;
  else if (first=="HELP" || first=="?") printHelp(tokStr);
  else if (first=="SHOW") {
    int iLevel=strlen(PROMPT_SYS);
    if (s==1)
      // print all FS
      for (fsIdx=0,fsSize=fsRegs.size();fsIdx<fsSize;++fsIdx)
	if (FStruc::indentPrint)
	  out<<PROMPT_SYS<<indent(iLevel)<<fsRegs[fsIdx]<<indent(-iLevel)<<endl;
	else out<<PROMPT_SYS<<'X'<<fsIdx<<": "<<fsRegs[fsIdx]<<endl;
    
    else
      for (idx=1,fsSize=fsRegs.size();idx<s;++idx)
	if (isValidFSName(tokStr[idx],fsIdx))
	  if (fsIdx<fsSize)
	    if (FStruc::indentPrint)
	      out<<PROMPT_SYS<<indent(iLevel)<<fsRegs[fsIdx]<<indent(-iLevel)<<endl;
	    else out<<PROMPT_SYS<<'X'<<fsIdx<<": "<<fsRegs[fsIdx]<<endl;
	  else out<<PROMPT_ERR<<"\'"<<tokStr[idx]<<"\' is out of range."<<endl;
	else out<<PROMPT_ERR<<"\'"<<tokStr[idx]<<"\' is an invalid FS name."<<endl;
  }
  else if (first=="INDENT")
    if (s<=2) {
      if (s==2) FStruc::indentPrint=(tokStr[1]=="ON");
      out<<PROMPT_SYS<<"Indentation printing is "<<(FStruc::indentPrint?"on":"off")<<'.'<<endl;
    }
    else
      // error: too many arguments
      out<<PROMPT_ERR<<"Too many arguments."<<endl;
  else if (first=="SIZE")
    if (s<=2) {
      if (s==2)
	if (isInteger(tokStr[1],(int)fsIdx) && fsIdx>0) fsRegs.resize(fsIdx);
	else out<<PROMPT_ERR<<"The argument must be a positive integer."<<endl;
      out<<PROMPT_SYS<<"Current FS registers size is "<<fsRegs.size()<<'.'<<endl;
    }
    else
      // error: too many arguments
      out<<PROMPT_ERR<<"Too many arguments."<<endl;
  else
    // error: the shell command does not exist
    out<<PROMPT_ERR<<"The shell command \'"<<tokStr[0]<<"\' does not exist."<<endl;
  
  return false;
}

void UShell::kernelCmd (string &cmd) {
  OpVec::size_type i,j;
  string::size_type k=0;  // just to suppress the warning of "use without initialized"
  int symCode;
  TokenString lhs,rhs;

  // ensure proper tokenization
  replaceAll(cmd,")"," ) ",0);
  replaceAll(cmd,"("," ( ",0);

  // locate the operator
  for (i=0,j=opVec.size();i<j && (k=cmd.find(opVec[i].str))==string::npos;
       ++i);

  if (i==j)
    // error: no operator inside the kernel command
    out<<PROMPT_ERR<<"No operator inside the kernel command."<<endl;
  else {
    FSRegisters::size_type lFSIdx,rFSIdx;
    int iLevel;
    Path lPath,rPath;
    Symbol sym;
    Value v;
    ConvErrCode convErrCode;
    bool noError=false;
    
    // if the operator is "=t", "=c" or "=i", check if the RHS is empty, if yes the 
    // operator should be "="
    if ((opVec[i].code==Symbol::OP_TEST || opVec[i].code==Symbol::OP_CONSTRAIN ||
	 opVec[i].code==Symbol::OP_ISO)
	&& cmd.find_first_not_of(WHITESPACE_STR,k+2)==string::npos)
      for (i=0;opVec[i].code!=Symbol::OP_PSEUDO_UNIFY;++i);
    
    lhs.set(cmd.substr(0,k),TokenString::none);
    rhs.set(cmd.substr(k+opVec[i].str.length()),TokenString::none);

    if (lhs.size()==0 || rhs.size()==0)
      out<<PROMPT_ERR<<"LHS or RHS is missing."<<endl;    
    // the operator is opVec[i].code and we have lhs and rhs
    else if (convertPath(lhs,lFSIdx,lPath)==CONV_SUCC) {
      if (isValue(rhs,sym)) {
	// RHS is a value
	switch ((symCode=sym.readCode())) {
	case Symbol::UNKNOWN:
	case Symbol::NOT:
	case Symbol::OR:
	case Symbol::MULTIPLE:
	  // ---------- type 2 equation ----------
	  if ((convErrCode=convertValue(rhs,v))==CONV_SUCC)
	    if (opVec[i].code==Symbol::OP_REMOVE_ASSIGN ||
		opVec[i].code==Symbol::OP_PUSH ||
		opVec[i].code==Symbol::OP_POP)
	      out<<PROMPT_ERR<<"Operator \'"<<opVec[i].str<<"\' cannot be used with a value as the RHS."<<endl;
	    else {
	      noError=true;
	      eb.addBlock(new EBlockMain(lFSIdx,lPath,v,opVec[i].code));
	    }
	  else
	    switch (convErrCode) {
	    case CONV_LPAREN:
	      out<<PROMPT_ERR<<"More \'(\' than \')\'."<<endl;
	      break;
	    case CONV_RPAREN:
	      out<<PROMPT_ERR<<"More \')\' than \'(\'."<<endl;
	      break;
	    case CONV_SPECIAL:
	      out<<PROMPT_ERR<<"*NOT*/*OR*/*MULTIPLE* must immediately follow \'(\'."<<endl;
	      break;
	    case CONV_NAKED:
	      out<<PROMPT_ERR<<"Non-atomic value must be surrounded by parentheses."<<endl;
	      break;
	    default: throw;  // just to suppress compiler warnings
	    }
	  break;
	  
	default:  // all special Symbol::SymCode except for UNKNOWN
		  // ---------- type 3 equation ----------
	  if (opVec[i].code==Symbol::OP_PSEUDO_UNIFY) {
	    noError=true;
	    eb.addBlock(new EBlockMain(lFSIdx,lPath,(Symbol::SymCode)symCode));
	  }
	  else
	    // error: special form must use operator '='
	    out<<PROMPT_ERR<<"Special form must use operator \'=\'."<<endl;
	  break;
	}
      }
      
      // ---------- type 1 equation ----------
      else if (convertPath(rhs,rFSIdx,rPath)==CONV_SUCC) {
	noError=true;
	eb.addBlock(new EBlockMain(lFSIdx,lPath,rFSIdx,rPath,opVec[i].code));
      }
      else
	// error: illegal RHS path - we only have one path error for now
	out<<PROMPT_ERR<<"Illegal RHS path."<<endl;
      
      if (noError) {
	if (eb.run(fsRegs)) {
	  iLevel=strlen(PROMPT_SYS)+6;
	  out<<PROMPT_SYS<<"TRUE; ";
	  if (FStruc::indentPrint)
	    out<<indent(iLevel)<<fsRegs[lFSIdx]<<indent(-iLevel)<<endl;
	  else out<<'X'<<lFSIdx<<": "<<fsRegs[lFSIdx]<<endl;
	}
	else out<<PROMPT_SYS<<"FALSE."<<endl;
	
	eb.clearBlocks();
      }
    }  // converting the left path
    else
      // error: illegal LHS path - we only have one path error for now
      out<<PROMPT_ERR<<"Illegal LHS path."<<endl;
  }
}

void UShell::run () {
  string cmd;
  TokenString tokStr;
  string::size_type i;

  fsRegs.resize(DEFAULT_FS_REGS_SIZE);

  printVersion();
  out<<PROMPT_SYS<<"Default FS registers size is "<<DEFAULT_FS_REGS_SIZE<<endl;

  out<<PROMPT_CMD<<flush;  // prompt
  while (getline(in,cmd)) {
    if ((i=cmd.find_first_not_of(WHITESPACE_STR))!=string::npos) {
      cmd.erase(0,i);
      
      if (cmd[0]==SHELL_CMD_PREFIX) {
	if (shellCmd(cmd)) return;
      }
      else kernelCmd(cmd);
    }
    out<<PROMPT_CMD<<flush;  // prompt
  }
}

int main () {
  UShell ush(cin,cout);

  ush.run();
  return 0;
}
