// ======================================================================
// symbol.hpp - A Symbol is an 'integeralized' token, and it has two
//             benefits: (a) fast comparison: internally it's just
//             comparison between two integer IDs; and (b) smart printing:
//             print out real text instead of integer IDs.
//
// 022603: Benjamin Han <benhdj@cs.cmu.edu> Added 3 new SymCode: NOT_NUMBER
//         ("*NOT-NUMBER*"), NOT_INTEGER ("*NOT-INTEGER*"), and NOT_POSITIVE
//         ("*NOT-POSITIVE*")
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 101102: Benjamin Han <benhdj@cs.cmu.edu> Added BOX to Symbol::SymCode for
//         solving the problem of RHS collection with undefined branch.
// 092802: Benjamin Han <benhdj@cs.cmu.edu> Added new operators =c', =t' and
//         =i' (no-filter version of =c, =t and =i); modify printing of symbols
//         so a pair of double quotes will surround empty strings and strings
//         with whitespaces.
// 091602: Benjamin Han <benhdj@cs.cmu.edu> Modify the printing of symbols so
//         empty strings will be surrounded by a pair of double quotes.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Added Symbol::OP_ISO for
//         isomorphism operator.
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Changed Symbol::OP_EQUAL to 
//         Symbol::OP_TEST (testing operator "=t"), added Symbol::OP_CONSTRAIN
//         (constraint operator "=c")
// 081901: Benjamin Han <benhdj@cs.cmu.edu> Updated to work with the new
//         TokenID<>.
// 071701: Benjamin Han <benhdj@cs.cmu.edu> Added global const symOr,symNot
//         and symMultiple (for values and FSs constructions).
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Added global const symGetLex,
//         symGetLexFS, symLexID, symCheck, symAmbiguity (all lexicon
//         related).
// 070601: Benjamin Han <benhdj@cs.cmu.edu> Chagned Symbol::OP_APPEND to
//         Symbol::OP_PUSH
// 062101: Benjamin Han <benhdj@cs.cmu.edu> Shadowed TokenID<>::read() with
//         Symbol::read(), and changed operator<<(ostream&,const Symbol&)
//         into an inline function calling Symbol::read().
// 061401: Benjamin Han <benhdj@cs.cmu.edu> Added global const symWildcard,
//         symValue,symSemValue, symCat and symID.
// 061201: Benjamin Han <benhdj@cs.cmu.edu> Added a global const symNull.
// 052901: Benjamin Han <benhdj@cs.cmu.edu> Minor revision.
// 051501: Benjamin Han <benhdj@cs.cmu.edu> Added isNumber(), isInteger()
//         and isPositive().
// 050801: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef SYMBOL_HPP
#define SYMBOL_HPP

#include "tokID.hpp"

#include <string>
#include <iostream>

#include <stdlib.h>

namespace UKernel {

using namespace Toolbox;

// No assumption about the case of the characters are made - more i18n friendly
class Symbol: public TokenID<std::string> {
public:

  // Negative id is a special value
  // IMPORTANT: UNKNOWN must be the same as UNKNOWN_TOKEN_ID defined in TokenDict<> (-1)
  // ASSUMPTION: TokenDict<> only assigns non-negative IDs
  enum SymCode {NUMBER=-25,INTEGER,POSITIVE,NOT_NUMBER,NOT_INTEGER,NOT_POSITIVE,
		DEFINED,UNDEFINED,REMOVE,MULTIPLE,OR,NOT,
		OP_PSEUDO_UNIFY,OP_CONSTRAIN,OP_CONSTRAIN_NOFILTER,
		OP_TEST,OP_TEST_NOFILTER,OP_ISO,OP_ISO_NOFILTER,
		OP_ASSIGN,OP_REMOVE_ASSIGN,OP_PUSH,OP_POP,BOX,UNKNOWN};

  Symbol () {}
  Symbol (const std::string &token):TokenID<std::string>(token) {}
  Symbol (SymCode sc) { id=sc; }                        // setting special symbols

  const std::string &read () const;

  void setSpecial (SymCode sc) { id=sc; }
  int readCode () const { return id; }

  bool isSpecial () const { return id<UNKNOWN; }        // is this a special symbol? (id<UNKNOWN)
  bool isSpecial (SymCode sc) const { return id==sc; }  // does its code equal to sc?

  // ASSUMPTION: no token is an empty string
  bool isNumber () const;                               // is it a real number?
  bool isInteger () const;                              // is it an integer?
  bool isPositive () const;                             // is it a positive real number?
};

// WARNING: DO NOT USE THESE IN ANY DECLARATION OF GLOBAL/STATIC OBJECTS!
// Reason: initialization order of global/static objects in diff. translation
//         units is undefined, and Symbol has a static TokenDict inside.

extern const Symbol symNot;
extern const Symbol symOr;
extern const Symbol symMultiple;

extern const Symbol symNull;
extern const Symbol symWildcard;
extern const Symbol symValue;
extern const Symbol symSemValue;
extern const Symbol symCat;
extern const Symbol symID;

extern const Symbol symGetLex;
extern const Symbol symGetLexFS;
extern const Symbol symLexID;
extern const Symbol symCheck;
extern const Symbol symAmbiguity;

// ============================ inline functions ============================

inline bool Symbol::isNumber () const {
  if (id<0) return false;

  const char *strPtr=dict().readToken(id).c_str();
  char *endPtr;

  strtod(strPtr,&endPtr);
  return (*endPtr==0);
}

inline bool Symbol::isInteger () const {
  if (id<0) return false;

  const char *strPtr=dict().readToken(id).c_str();
  char *endPtr;

  strtol(strPtr,&endPtr,10);
  return (*endPtr==0);
}

inline bool Symbol::isPositive () const {
  if (id<0) return false;

  double d;
  const char *strPtr=dict().readToken(id).c_str();
  char *endPtr;

  d=strtod(strPtr,&endPtr);
  return (*endPtr==0 && d>0.0);
}

inline std::ostream &operator << (std::ostream &os, const Symbol &sym) {
  std::string str=sym.read();
  if (str.empty() || str.find(' ')!=std::string::npos) return os<<"\""<<str<<"\"";
  else return os<<str;
}

};

#endif
