// ======================================================================
// symbol.cpp - A Symbol is an 'integeralized' token, and it has two
//             benefits: (a) fast comparison: internally it's just
//             comparison between two integer IDs; and (b) smart printing:
//             print out real text instead of integer IDs.
// 
// 022603: Benjamin Han <benhdj@cs.cmu.edu> Added 3 new SymCode: NOT_NUMBER
//         ("*NOT-NUMBER*"), NOT_INTEGER ("*NOT-INTEGER*"), and NOT_POSITIVE
//         ("*NOT-POSITIVE*")
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 102702: Benjamin Han <benhdj@cs.cmu.edu> Changed the various strings returned
//         in Symbol::read() into local constant ones.
// 101102: Benjamin Han <benhdj@cs.cmu.edu> Added BOX to Symbol::SymCode for
//         solving the problem of RHS collection with undefined branch.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Added Symbol::OP_ISO for
//         isomorphism operator.
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Changed Symbol::OP_EQUAL to 
//         Symbol::OP_TEST (testing operator "=t"), added Symbol::OP_CONSTRAIN
//         (constraint operator "=c")
// 081901: Benjamin Han <benhdj@cs.cmu.edu> Updated to work with the new
//         TokenID<>.
// 071701: Benjamin Han <benhdj@cs.cmu.edu> Added global const symOr,symNot
//         and symMultiple (for values and FSs constructions).
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Added global const symGetLex,
//         symGetLexFS, symLexID, symCheck, symAmbiguity (all lexicon
//         related).
// 070601: Benjamin Han <benhdj@cs.cmu.edu> Chagned Symbol::OP_APPEND to
//         Symbol::OP_PUSH.
// 061501: Benjamin Han <benhdj@cs.cmu.edu> For UNKNOWN Symbol now we print
//         "-UNKNOWN-" instead of failing the assertion.
// 061201: Benjamin Han <benhdj@cs.cmu.edu> Added a global const symNull.
// 050801: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "symbol.hpp"

namespace UKernel {

const Symbol symNot(Symbol::NOT);
const Symbol symOr(Symbol::OR);
const Symbol symMultiple(Symbol::MULTIPLE);

const Symbol symNull("NIL");
const Symbol symWildcard("%");
const Symbol symValue("VALUE");
const Symbol symSemValue("SEM-VALUE");
const Symbol symCat("CAT");
const Symbol symID("LEX-ID");

const Symbol symGetLex("GET-LEX");
const Symbol symGetLexFS("GET-LEX-FS");
const Symbol symLexID(":LEX-ID");
const Symbol symCheck(":CHECK");
const Symbol symAmbiguity(":AMBIGUITY");

static const std::string strPseudoUnify="=";
static const std::string strConstrain="=c";
static const std::string strConstrainNoFilter="=c'";
static const std::string strTest="=t";
static const std::string strTestNoFilter="=t'";
static const std::string strIso="=i";
static const std::string strIsoNoFilter="=i'";
static const std::string strAssign="<=";
static const std::string strRemoveAssign="==";
static const std::string strPush=">";
static const std::string strPop="<";
static const std::string strNot="*NOT*";
static const std::string strOr="*OR*";
static const std::string strMultiple="*MULTIPLE*";
static const std::string strRemove="*REMOVE*";
static const std::string strUndefined="*UNDEFINED*";
static const std::string strDefined="*DEFINED*";
static const std::string strNotPositive="*NOT-POSITIVE*";
static const std::string strNotInteger="*NOT-INTEGER*";
static const std::string strNotNumber="*NOT-NUMBER*";
static const std::string strPositive="*POSITIVE*";
static const std::string strInteger="*INTEGER*";
static const std::string strNumber="*NUMBER*";
static const std::string strBox="[]";
static const std::string strUnknown="?";

const std::string &Symbol::read () const {
  switch (id) {
  case Symbol::OP_PSEUDO_UNIFY:return strPseudoUnify;

  case Symbol::OP_CONSTRAIN:return strConstrain;;
  case Symbol::OP_CONSTRAIN_NOFILTER:return strConstrainNoFilter;
  case Symbol::OP_TEST:return strTest;
  case Symbol::OP_TEST_NOFILTER:return strTestNoFilter;
  case Symbol::OP_ISO:return strIso;
  case Symbol::OP_ISO_NOFILTER:return strIsoNoFilter;

  case Symbol::OP_ASSIGN:return strAssign;
  case Symbol::OP_REMOVE_ASSIGN:return strRemoveAssign;
  case Symbol::OP_PUSH:return strPush;
  case Symbol::OP_POP:return strPop;
    
  case Symbol::NOT:return strNot;
  case Symbol::OR:return strOr;
  case Symbol::MULTIPLE:return strMultiple;
  case Symbol::REMOVE:return strRemove;
  case Symbol::UNDEFINED:return strUndefined;
  case Symbol::DEFINED:return strDefined;
  case Symbol::NOT_POSITIVE:return strNotPositive;
  case Symbol::NOT_INTEGER:return strNotInteger;
  case Symbol::NOT_NUMBER:return strNotNumber;
  case Symbol::POSITIVE:return strPositive;
  case Symbol::INTEGER:return strInteger;
  case Symbol::NUMBER:return strNumber;
    
  case Symbol::BOX:return strBox;
  case Symbol::UNKNOWN:return strUnknown;
    
  default: return TokenID<std::string>::read();
  }
}

};
