// ======================================================================
// lexicon.hpp - Providing multiple lexicons
// 
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 110602: Benjamin Han <benhdj@cs.cmu.edu> Get rid of lexiconsPtr, 
//         setActive(), getActiveLexicon() in Lexicons: now Grammar should
//         have the right Lexicons it's referencing.
// 102002: Benjamin Han <benhdj@cs.cmu.edu> Added findFirst() and findNext()
//         to both Lexicon and Lexicons: this makes GET-LEX/GET-LEX-FS with
//         :CHECK more efficient (see eFunc.cpp); removed const-ness of 
//         findLex() so they can reuse the object-wide variables (speeding
//         up things, and they are necessary for findFirst() and findNext()
//         anyway.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Namespace added.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Removed nextLex() methods;
//         all findLex() methods now take 'searchAll' boolean to search
//         all matching entries; also the prototypes of findLex() changed.
// 061501: Benjamin Han <benhdj@cs.cmu.edu> Added Lexicon::nullFS and
//         Lexicons::nullFS (previously nullFS is a global const but it's
//         problematic - see log in fStruc.*); for similar reasons moved
//         the global nullLexicon to Lexicons::nullLexicon.
// 061401: Benjamin Han <benhdj@cs.cmu.edu> Now you can print a Lexicon or
//         Lexicons.
// 061301: Benjamin Han <benhdj@cs.cmu.edu> Added setActive() and 
//         getActiveLexicons().
// 061201: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef LEXICON_HPP
#define LEXICON_HPP

#include "fStruc.hpp"

#include <iostream>

namespace UKernel {

typedef std::list<FStruc> _FSList;
typedef std::map<Symbol,_FSList> _LexEntry;          // (cat,_FSList) pairs

class Lexicon: protected std::map<Symbol,_LexEntry> { // (sem,_LexEntry) pairs
  Symbol id;
  int count;

  FStruc nullFS;
  Path emptyPath,semValuePath,catPath,lexIDPath;
  Value vID;

  // for find*() methods
  Value vSem,vCat;
  const FStruc *fsPtr;   // for findFirst() with filter FS
  _LexEntry::const_iterator lexI;
  _FSList::const_iterator fi;
  int lexCount,fsCount,lexSize,fsListSize;   

  friend std::ostream &operator << (std::ostream &os, const Lexicon &lexicon);
  
public:

  Lexicon (const Symbol &id);

  const Symbol &readID () const { return id; }
  int size () const { return count; }

  void clear () { std::map<Symbol,_LexEntry>::clear(); }

  // use add() to obtain a new FS so that we can initialize it
  FStruc &addLex (const Symbol &sem, const Symbol &cat);
  
  // CAUTION: the FS result is NOT cleared inside the following find*() calls!

  // 1. findLex() returns the # of matching entries, and all of the
  //    entries are merged into 'result', which is an *OR* FS if there's 
  //    more than one entry.
  // 2. findLex() returns 0 iff no matching entry is found, in which
  //    case 'result' is not modified in any way.
  // 3. IMPORTNAT: in the versions with an FS argument, the FS cannot contain
  //    feature 'cat' - if it does the cat feature-value pair needs to be
  //    extracted to be put in the 'cat' argument position.
  int findLex (const Symbol &sem, const Symbol &cat, FStruc &result, 
	       bool searchAll=false);
  int findLex (const Symbol &sem, const Symbol &cat, const FStruc &fs,
	       FStruc &result, bool searchAll=false);
  int findLex (const Symbol &sem, const FStruc &fs, FStruc &result, 
	       bool searchAll=false);

  // the following 3 findFirst() methods are analogous to the 3 findLex()
  // only they will find the first entry but remember the position, so 
  // subsequent findNext() calls return the next matching entry
  bool findFirst (const Symbol &sem, const Symbol &cat, FStruc &result);
  bool findFirst (const Symbol &sem, const Symbol &cat, const FStruc &fs,
		  FStruc &result);
  bool findFirst (const Symbol &sem, const FStruc &fs, FStruc &result);

  // search from the previous point
  // ASSUMPTION: one of findFirst() must have been called before!
  bool findNext (FStruc &result);
};

std::ostream &operator << (std::ostream &os, const Lexicon &lexicon);

// INTERNAL USE: keeps the loading ordering
typedef std::list<Lexicon> _LexiconList;

// multiple lexicons
class Lexicons: protected std::map<Symbol,_LexiconList::iterator> {
  _LexiconList ll;
  _LexiconList::iterator li;
  Lexicon nullLexicon;

  // for findFirst() and findNext()
  const Symbol *semPtr,*catPtr;
  const FStruc *fsPtr;  

  Path semValuePath;

  friend std::ostream &operator << (std::ostream &os, const Lexicons &lexicons);

public:

  // by default every newly declared Lexicons will be the active one
  // use setActive() to switch to another Lexicons if you want
  Lexicons ();

  Lexicon &addLexicon (const Symbol &id);
  void removeLexicon (const Symbol &id);

  Lexicon &findLexicon (const Symbol &id);

  // CAUTION: the FS result is NOT cleared inside the following find*() calls!

  // 1. See the comments for Lexicon::findLex().
  // 2. These versions search through all lexicons, not only one; the ID of the
  //    matching lexicon is returned in 'id' argument.
  int findLex (const Symbol &sem, const Symbol &cat, FStruc &result,
	       bool searchAll=false);
  int findLex (const Symbol &sem, const Symbol &cat, const FStruc &fs, 
	       FStruc &result, bool searchAll=false);
  int findLex (const Symbol &sem, const FStruc &fs, FStruc &result,
	       bool searchAll=false);

  // the following 3 findFirst() methods are analogous to the 3 findLex()
  // only they will find the first entry but remember the position, so 
  // subsequent findNext() calls return the next matching entry
  bool findFirst (const Symbol &sem, const Symbol &cat, FStruc &result);
  bool findFirst (const Symbol &sem, const Symbol &cat, const FStruc &fs,
		  FStruc &result);
  bool findFirst (const Symbol &sem, const FStruc &fs, FStruc &result);

  // search from the previous point
  // ASSUMPTION: one of findFirst() must have been called before!
  bool findNext (FStruc &result);

  int size () const { return ll.size(); }
};

std::ostream &operator << (std::ostream &os, const Lexicons &lexicons);

// ============================ inline functions ============================

inline Lexicon::Lexicon (const Symbol &id):id(id),count(0),nullFS(symNull) {
  semValuePath.push_back(symSemValue);
  catPath.push_back(symCat);
  lexIDPath.push_back(symID);
  vID.insert(vID.begin(),id);
}

inline Lexicons::Lexicons ():nullLexicon(symNull) {
  semValuePath.push_back(symSemValue);
}

inline Lexicon &Lexicons::findLexicon (const Symbol &id) {
  iterator li=find(id);
  if (li!=end()) return *li->second;
  else return nullLexicon;
}

inline int Lexicons::findLex (const Symbol &sem, const Symbol &cat, 
			      FStruc &result, bool searchAll) {
  int i,j;

  for (li=ll.begin(),i=0;li!=ll.end();++li)
    if ((j=li->findLex(sem,cat,result,searchAll))!=0)
      if (searchAll) i+=j;
      else return 1;

  return i;
}

inline int Lexicons::findLex (const Symbol &sem, const Symbol &cat, 
			      const FStruc &fs, FStruc &result, 
			      bool searchAll) {
  int i,j;

  for (li=ll.begin(),i=0;li!=ll.end();++li)
    if ((j=li->findLex(sem,cat,fs,result,searchAll))!=0)
      if (searchAll) i+=j;
      else return 1;

  return i;
}

// only when succeed is sameCat set to false
inline int Lexicons::findLex (const Symbol &sem, const FStruc &fs,
			      FStruc &result, bool searchAll) { 
  int i,j;

  for (li=ll.begin(),i=0;li!=ll.end();++li)
    if ((j=li->findLex(sem,fs,result,searchAll))!=0)
      if (searchAll) i+=j;
      else return 1;

  return i;
}

inline bool Lexicons::findFirst (const Symbol &sem, const Symbol &cat, 
				 FStruc &result) {
  semPtr=&sem;
  catPtr=&cat;
  fsPtr=NULL;

  for (li=ll.begin();li!=ll.end();++li)
    if (li->findFirst(sem,cat,result)) return true;

  return false;
}

inline bool Lexicons::findFirst (const Symbol &sem, const Symbol &cat, 
				 const FStruc &fs, FStruc &result) {
  semPtr=&sem;
  catPtr=&cat;
  fsPtr=&fs;

  for (li=ll.begin();li!=ll.end();++li)
    if (li->findFirst(sem,cat,fs,result)) return true;

  return false;
}

inline bool Lexicons::findFirst (const Symbol &sem, const FStruc &fs,
				 FStruc &result) {
  semPtr=&sem;
  catPtr=NULL;
  fsPtr=&fs;

  for (li=ll.begin();li!=ll.end();++li)
    if (li->findFirst(sem,fs,result)) return true;

  return false;
}

inline bool Lexicons::findNext (FStruc &result) {
  if (li->findNext(result)) return true;
  else
    while (++li!=ll.end())
      if (catPtr)
	if (fsPtr) {
	  if (li->findFirst(*semPtr,*catPtr,*fsPtr,result))
	    return true;
	}
	else {
	  if (li->findFirst(*semPtr,*catPtr,result))
	    return true;
	}
      else if (li->findFirst(*semPtr,*fsPtr,result))
	return true;

  return false;
}

};

#endif
