// ======================================================================
// lexicon.cpp - Providing multiple lexicons
// 
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 110602: Benjamin Han <benhdj@cs.cmu.edu> Get rid of lexiconsPtr.
// 102002: Benjamin Han <benhdj@cs.cmu.edu> Added findFirst() and findNext()
//         to both Lexicon and Lexicons: this makes GET-LEX/GET-LEX-FS with
//         :CHECK more efficient (see eFunc.cpp); removed const-ness of 
//         findLex() so they can reuse the object-wide variables (speeding
//         up things, and they are necessary for findFirst() and findNext()
//         anyway.
// 092802: Benjamin Han <benhdj@cs.cmu.edu> Changed the calls to FStruc::test()
//         to FStruc::testNoFilter().
// 091602: Benjamin Han <benhdj@cs.cmu.edu> Fixed the missing closing 
//         parenthesis in lexicon printing.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Namespace added.
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Changed calls to FStruc::equal()
//         to FStruc::test().
// 082801: Benjamin Han <benhdj@cs.cmu.edu> Use list<>::back() whenever 
//         possible.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Removed nextLex() methods;
//         all findLex() methods now take 'searchAll' boolean to search
//         all matching entries; also the prototypes of findLex() changed.
// 062101: Benjamin Han <benhdj@cs.cmu.edu> Fixed bugs in
//         Lexicon::nextLex(const FStruc&,Symbol&).
// 061401: Benjamin Han <benhdj@cs.cmu.edu> Now you can print a Lexicon or
//         Lexicons.
// 061201: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "lexicon.hpp"

#include <sstream>

namespace UKernel {

FStruc &Lexicon::addLex (const Symbol &sem, const Symbol &cat) {
  iterator li=find(sem);
  _FSList *fslPtr;

  ++count;
  if (li==end()) fslPtr=&(operator[](sem)[cat]);
  else {
    _LexEntry::iterator lexI=li->second.find(cat);
    
    if (lexI==li->second.end()) fslPtr=&(li->second[cat]);
    else fslPtr=&(lexI->second);
  }
  
  fslPtr->push_back(nullFS);
  return fslPtr->back();
}

int Lexicon::findLex (const Symbol &sem, const Symbol &cat, FStruc &result,
		      bool searchAll) {
  const_iterator li=find(sem);
  
  if (li!=end() && (lexI=li->second.find(cat))!=li->second.end()) {
    const _FSList &fsList=lexI->second;
    fsListSize=fsList.size();

    vSem.clear();
    vCat.clear();
    vSem.insert(vSem.begin(),sem);
    vCat.insert(vCat.begin(),cat);
    if (fsListSize>1 && searchAll) {
      for (fsCount=0,fi=fsList.begin();fsCount<fsListSize;++fsCount,++fi) {
	result.mergeOrMulti(*fi);
	result.assignComplexLast(semValuePath,vSem);
	result.assignComplexLast(catPath,vCat);
	result.assignComplexLast(lexIDPath,vID);
      }
      return fsListSize;
    }
    else {
      result.mergeOrMulti(*fsList.begin());
      result.assignComplexLast(semValuePath,vSem);
      result.assignComplexLast(catPath,vCat);
      result.assignComplexLast(lexIDPath,vID);
      return 1;
    }
  }
  return 0;
}

int Lexicon::findLex (const Symbol &sem, const Symbol &cat,  const FStruc &fs,
		      FStruc &result, bool searchAll) {
  const_iterator li=find(sem);

  if (li!=end() && (lexI=li->second.find(cat))!=li->second.end()) {
    const _FSList &fsList=lexI->second;
    int i=0;

    vSem.clear();
    vCat.clear();    
    vSem.insert(vSem.begin(),sem);
    vCat.insert(vCat.begin(),cat);
    for (fi=fsList.begin();fi!=fsList.end();++fi)
      if (fi->testNoFilter(emptyPath,fs,emptyPath)) {
	result.mergeOrMulti(*fi);
	result.assignComplexLast(semValuePath,vSem);
	result.assignComplexLast(catPath,vCat);
	result.assignComplexLast(lexIDPath,vID);
	
	if (searchAll) ++i;
	else return 1;
      }
    
    return i;
  }
  return 0;
}

int Lexicon::findLex (const Symbol &sem, const FStruc &fs, FStruc &result,
		      bool searchAll) {
  const_iterator li=find(sem);
   
  if (li!=end()) {
    int i=0;
    
    vSem.clear();
    vSem.insert(vSem.begin(),sem);
    lexI=li->second.begin();
    do {
      const _FSList &fsList=lexI->second;
      
      vCat.clear();
      vCat.insert(vCat.begin(),lexI->first);
      for (fi=fsList.begin();fi!=fsList.end();++fi)
	if (fi->testNoFilter(emptyPath,fs,emptyPath)) {
	  result.mergeOrMulti(*fi);
	  result.assignComplexLast(semValuePath,vSem);
	  result.assignComplexLast(catPath,vCat);
	  result.assignComplexLast(lexIDPath,vID);
	  
	  if (searchAll) ++i;
	  else return 1;
	}
    } while ((++lexI)!=li->second.end());
    
    return i;
  }
  return 0;
}

bool Lexicon::findFirst (const Symbol &sem, const Symbol &cat, FStruc &result) {
  const_iterator li=find(sem);
  
  if (li!=end() && (lexI=li->second.find(cat))!=li->second.end()) {
    const _FSList &fsList=lexI->second;

    fsPtr=NULL;

    lexSize=1;    
    fsListSize=fsList.size();
    lexCount=fsCount=0;
    
    vSem.clear();
    vCat.clear();
    vSem.insert(vSem.begin(),sem);
    vCat.insert(vCat.begin(),cat);

    fi=fsList.begin();
    result.mergeOrMulti(*fi);
    result.assignComplexLast(semValuePath,vSem);
    result.assignComplexLast(catPath,vCat);
    result.assignComplexLast(lexIDPath,vID);

    return true;
  }
  return false;
}

bool Lexicon::findFirst (const Symbol &sem, const Symbol &cat, const FStruc &fs,
			 FStruc &result) {
  const_iterator li=find(sem);

  if (li!=end() && (lexI=li->second.find(cat))!=li->second.end()) {
    const _FSList &fsList=lexI->second;

    fsPtr=&fs;

    lexSize=1;
    fsListSize=fsList.size();
    lexCount=fsCount=0;

    vSem.clear();
    vCat.clear();    
    vSem.insert(vSem.begin(),sem);
    vCat.insert(vCat.begin(),cat);

    for (fi=fsList.begin();fsCount<fsListSize;++fi,++fsCount)
      if (fi->testNoFilter(emptyPath,fs,emptyPath)) {
	result.mergeOrMulti(*fi);
	result.assignComplexLast(semValuePath,vSem);
	result.assignComplexLast(catPath,vCat);
	result.assignComplexLast(lexIDPath,vID);
	
	return true;
      }
  }
  return false;
}

bool Lexicon::findFirst (const Symbol &sem, const FStruc &fs, FStruc &result) {
  const_iterator li=find(sem);
   
  if (li!=end()) {
    fsPtr=&fs;

    vSem.clear();
    vSem.insert(vSem.begin(),sem);

    lexI=li->second.begin();
    lexSize=li->second.size();
    lexCount=0;
    do {
      const _FSList &fsList=lexI->second;

      fsListSize=fsList.size();
      fsCount=0;
      
      vCat.clear();
      vCat.insert(vCat.begin(),lexI->first);
      for (fi=fsList.begin();fsCount<fsListSize;++fi,++fsCount)
	if (fi->testNoFilter(emptyPath,fs,emptyPath)) {
	  result.mergeOrMulti(*fi);
	  result.assignComplexLast(semValuePath,vSem);
	  result.assignComplexLast(catPath,vCat);
	  result.assignComplexLast(lexIDPath,vID);
	  
	  return true;
	}
      
      if (++lexCount<lexSize) ++lexI;
      else break;
    } while (true);
  }
  return false;
}

bool Lexicon::findNext (FStruc &result) {

  do {
    if (++fsCount<fsListSize)
      // advance to the next entry with the same category
      ++fi;
    else if (++lexCount<lexSize) {
      // advance to the next category
      const _FSList &fsList=(++lexI)->second;
      
      vCat.clear();
      vCat.insert(vCat.begin(),lexI->first);
      fsListSize=fsList.size();
      fi=fsList.begin();
    }
    else return false;
  } while (fsPtr && !fi->testNoFilter(emptyPath,*fsPtr,emptyPath));

  result.mergeOrMulti(*fi);
  result.assignComplexLast(semValuePath,vSem);
  result.assignComplexLast(catPath,vCat);
  result.assignComplexLast(lexIDPath,vID);
  return true;
}

std::ostream &operator << (std::ostream &os, const Lexicon &lexicon) {
  Lexicon::const_iterator li;
  _LexEntry::const_iterator lexI;
  _FSList::const_iterator fsI;
  std::ostringstream prefix,middle,suffix;
  bool backup,notFirst;

  backup=FStruc::indentPrint;
  FStruc::indentPrint=false;
  for (li=lexicon.begin(),notFirst=false;li!=lexicon.end();
       ++li,prefix.str("")) {
    prefix<<'('<<li->first<<' ';
    for (lexI=li->second.begin();lexI!=li->second.end();
	 ++lexI,middle.str("")) {
      middle<<"((CAT "<<lexI->first<<") ";
      for (fsI=lexI->second.begin();fsI!=lexI->second.end();
	   ++fsI,suffix.str("")) {
	suffix<<*fsI;
	if (notFirst) os<<std::endl;
	else notFirst=true;
	// substr(1) because printing an FS will have its own opening parenthesis
	os<<prefix.str()<<middle.str()<<suffix.str().substr(1)<<')';
      }
    }
  }
  
  FStruc::indentPrint=backup;
  return os;
}

std::ostream &operator << (std::ostream &os, const Lexicons &lexicons) {
  _LexiconList::const_iterator li;
  bool notFirst;

  for (li=lexicons.ll.begin(),notFirst=false;li!=lexicons.ll.end();++li) {
    if (notFirst) os<<std::endl;
    else notFirst=true;
    os<<"LEX-ID = "<<li->readID()<<std::endl<<*li;
  }
  
  return os;
}

Lexicon &Lexicons::addLexicon (const Symbol &id) {
  iterator li=find(id);

  if (li==end()) {
    ll.push_back(Lexicon(id));
    return *(operator[](id)=(--ll.end()));
  }
  else return *li->second;
}

void Lexicons::removeLexicon (const Symbol &id) {
  iterator li=find(id);

  if (li!=end()) {
    ll.erase(li->second);
    erase(li);
  }
}

};
