// ======================================================================
// fsRegs.hpp - Feature Structure Registers.
// 
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed; use FSRegisters::_Parent.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> FSIdxSet is now a nested class
//         of FSRegisters; changed several int declarations into size_type;
//         namespace added.
// 090501: Benjamin Han <benhdj@cs.cmu.edu> Revised FSregisters::copy() (the
//         1st form) to copy/remove only the additional/redundant FSs - faster.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Created (moved from equation.*).
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef FSREGS_HPP
#define FSREGS_HPP

#include "fStruc.hpp"

#include "misc.hpp"

#include <vector>
#include <list>
#include <set>

namespace UKernel {

// Feature structure registers - for running equations
// TO-DO: it's designed to be fast for add(), but slower for operator[] -
//        is this the case during run time?
class FSRegisters: protected std::vector<std::list<FStruc>::iterator> {
  typedef std::vector<std::list<FStruc>::iterator> _Parent;

  std::list<FStruc> fsList;

public:
  typedef _Parent::size_type size_type;

  class FSIdxSet: public std::set<size_type> {
  public:
    
    FSIdxSet &operator += (const FSIdxSet &fsIdxSet);
  };

  FSRegisters () {}
  FSRegisters (size_type s);  // specify the size

  // the 1st form resizes *this to s; if s is larger than the original size *ONLY*
  //   those additional FSs will be copied from fsRegs while the others are left
  //   intact; otherwise only those redundant FSs will be removed and the others
  //   are left intact
  // the 2nd form copies the elements whose indices are in fsIdxSet from fsRegs
  void copy (const FSRegisters &fsRegs, size_type s);
  void copy (const FSRegisters &fsRegs, const FSIdxSet &fsIdxSet);

  void add (size_type newFSIdx);  // resize *this to accomodate the newFSIdx
  void resize (size_type s);

  FStruc &operator [] (size_type i);
  const FStruc &operator [] (size_type i) const;

  size_type size () const { return _Parent::size(); }
};

std::ostream &operator << (std::ostream &os, const FSRegisters &fsRegs);

// ============================ inline functions ============================

inline void FSRegisters::copy (const FSRegisters &fsRegs, size_type s) {
  size_type i,j=size();

  if (j<s) {
    _Parent::resize(s);
    for (i=j;i<s;++i) {
      fsList.push_back(fsRegs[i]);
      _Parent::operator[](i)=(--fsList.end());
    }
  }
  else if (s<j) {
    for (i=s;i<j;++i)
      fsList.erase(_Parent::operator[](i));
    _Parent::resize(s);
  }
}

inline void FSRegisters::copy (const FSRegisters &fsRegs, const FSIdxSet &fsIdxSet) {
  FSIdxSet::const_iterator fi;

  // ASSUMPTION: all FSs in *this already have names
  for (fi=fsIdxSet.begin();fi!=fsIdxSet.end();++fi) {
    assert(*fi<size());
    operator[](*fi)=fsRegs[*fi];
  }
}

inline FStruc &FSRegisters::operator [] (size_type i) {
  assert(i>=0 && i<size());
  return *_Parent::operator[](i);
}

inline const FStruc &FSRegisters::operator [] (size_type i) const {
  assert(i>=0 && i<size());
  return *_Parent::operator[](i);
}

};

#endif
