// ======================================================================
// fStruc.hpp - Feature Structure code
// 
// 061604: Benjamin Han <benhdj@cs.cmu.edu> fix in Path::compare(): missing "return"
//         in "return hasPrefix"!
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 112002: Benjamin Han <benhdj@cs.cmu.edu> nullTI -> nullTI().
// 102002: Benjamin Han <benhdj@cs.cmu.edu> Changed some comments: equivalence
//         tests (=t) should be "unifiability tests" now; removed the bool 
//         argument 'first' in all filter(): it's no longer needed for GET-LEX/
//         GET-LEX-FS with :CHECK; changed Feature::vFlag (bool) into one bit in
//         bitset 'flags' to accommodate possible future flags
// 101802: Benjamin Han <benhdj@cs.cmu.edu> Modifyed collect*() in FStruc
//         so undefined branches get boxes in the collected FS/V; also modified
//         various operations so that boxes work correctly with others;
//         added FStruc::removeBox() to remove possible boxes (e.g., when
//         assigning RHS to LHS).
// 101102: Benjamin Han <benhdj@cs.cmu.edu> Remove succ ref arg from 
//         FStruc::filter() - the semantics of GET-LEX/GET-LEX-FS
//         with an undefined :CHECK arg is equivalent to no checking
//         (filtering) at all.
// 100902: Benjamin Han <benhdj@cs.cmu.edu> Added the 'first' bool argument
//         to all FStruc::filter(): if it's true only the first branch of
//         an *OR* FS after filtering is kept (for the modified GET-LEX*
//         with :CHECK behavior - see eFunc.cpp).
// 100402: Benjamin Han <benhdj@cs.cmu.edu> Fixed FStruc::isUndefined() so
//         given an empty LHS FS it'll return true instead of false.
// 092802: Benjamin Han <benhdj@cs.cmu.edu> Changed the behavior of =c 
//         (FStruc::constrain() ): it's testing the isomorphism up to the
//         unifiability of values (so =i, i.e., FStruc::isomorphism() are still
//         stricter), and it also filters out in an *OR* FS the branches that
//         do not pass the tests; changed the behavior of =t and =i so they now
//         too become filtering operators given an *OR* FS; added *NoFilter() 
//         in FStruc for the new operators =c', =t', and =i'; changed the 
//         behavior of =i so that it's in line with =c and =t when RHS is empty 
//         or not defined;
//         changed isDefined(), isNumber(), isInteger(), isPositive(), and 
//         added isUndefined() in FStruc: now they are destructive and can
//         be used as a filter against an *OR* FS;
//         moved all mp-, rp-, and cb- series of functions together with their
//         supporting code to a new file fStrucFunc.cpp;
//         added FStruc::isomorphicSameRootRelaxed() to test isomorphism up to
//         the unifiability of values.
// 092102: Benjamin Han <benhdj@cs.cmu.edu> Modified both FStruc::constrain()
//         so the thing unified is the thing that's defined (discovered by
//         Donna Gates)
// 091602: Benjamin Han <benhdj@cs.cmu.edu> Added one protected method
//         FStruc::copyDaughter() (basically make a call to 
//         FStruc::operator=()).
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Changed several int declarations
//         into the rightful size_type; namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Major revision for support of
//         penetrating paths: added modifyPath(), runPath(), collect*(),
//         reduceToAtomFSIfPossible(), compact(), etc in FStruc, removed 
//         assignPath(), walkPath(), assignComplexLast() etc, changed 
//         isAtom() to isAtomValue(), removeAtom() to removeAtomValue(),
//         and revised almost all of the rest of the code; added 
//         FStruc::operator==(), and a new UKernel operator "=i" (isomorphism,
//         Symbol::OP_ISO); also in the process fixed several bugs in
//         detecting isomorphic daughter FSs of *OR* FSs.
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Renamed all equal() to test();
//         revised comments: changed "=c" to "=t"; added two forms of
//         FStruc::constrain() for constraint equations ("=c").
// 090301: Benjamin Han <benhdj@cs.cmu.edu> Renamed Equivalence*<> to 
//         TIsomorphism*<>.
// 080701: Benjamin Han <benhdj@cs.cmu.edu> Renamed dSize() to size() in 
//         _TreeIteratorBase<>; renamed Homomorphic* function objects to
//         Equivalence*; revised extendPath(), assignPath() and walkPath()
//         in FStruc - now _PathIterator<> embeds the path so we don't need
//         to pass Path anymore.
// 080201: Benjamin Han <benhdj@cs.cmu.edu> Renamed dBegin() and dEnd() to 
//         begin() and end() in _TreeIteratorBase<>.
// 072001: Benjamin Han <benhdj@cs.cmu.edu> Revised using the new ways to
//         declare tree iterators.
// 071701: Benjamin Han <benhdj@cs.cmu.edu> Revised FStruc::reduceToAtomFS():
//         now the argument is non-const and will be updated correctly.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::mergeOrMulti(), 
//         FStruc::isComplex(), FStruc::assignComplex(), 
//         FStruc::assignComplexLast() and FStruc::filter(); revised 
//         FStruc::equal() so now the equivalence test can be relaxed; 
//         simplified FStruc::reduceToAtomFS().
// 071001: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::multiplePU() and
//         FStruc::multipleEq().
// 070601: Benjamin Han <benhdj@cs.cmu.edu> Changed FStruc::append() to
//         FStruc::push().
// 070201: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::isValue(); added
//         FStruc::mergeOr().
// 062901: Benjamin Han <benhdj@cs.cmu.edu> Revised all FS operator-related
//         methods to conform with the descriptions in Operators.txt.
// 062801: Benjamin Han <benhdj@cs.cmu.edu> Added Path::compare().
// 062401: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::reduceToAtomFS(),
//         FStruc::isAtom() and FStruc::removeAtom(); moved walkPath() from
//         public back to private again; moved
//         operator=(TreeConstIteratorBase(Feature)) to private; removed
//         FStruc::prune().
// 062301: Benjamin Han <benhdj@cs.cmu.edu> Removed FStruc::removeVal() and
//         made FStruc::remove() public - they're the same.
// 062201: Benjamin Han <benhdj@cs.cmu.edu> Now operator '*' and '->' only
//         return const object; added _TreeIteratorBase<>::update() to update
//         a node's data while preserving the integrity of the indices;
//         added _TreeIteratorBase<>::get() - dangerous use it only when you
//         know what you are doing.
// 062101: Benjamin Han <benhdj@cs.cmu.edu> Added
//         operator=(TreeConstIteratorBase(Feature)).
// 061501: Benjamin Han <benhdj@cs.cmu.edu> Removed nullFS - init order of
//         global objects of diff translation units is undefined - it conflicts
//         with the static component of Symbol (i.e., TokenDict); revised
//         extendPath() again to allow it fails; added assignPath().
// 061301: Benjamin Han <benhdj@cs.cmu.edu> Moved two walkPath() from private
//         to public; and now you can turn on/off indented printing by setting
//         FStruc::indentPrint.
// 061201: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::empty(); added
//         FStruc::equal() for an FS+path RHS; added a global const nullFS.
// 051701: Benjamin Han <benhdj@cs.cmu.edu> Added operator << for Path;
//         added a new data member name, new methods setName() and getName(),
//         and a new constructor FStruc(const Symbol&) to FStruc.
// 051501: Benjamin Han <benhdj@cs.cmu.edu> Added isNumber(), isInteger()
//         and isPositive(); code cleanup.
// 051401: Benjamin Han <benhdj@cs.cmu.edu> Moved the static 
//         Feature::vList to FStruc.
// 050801: Benjamin Han <benhdj@cs.cmu.edu> Now a feature name is a Symbol
//         instead of an int.
// 042101: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef FSTRUC_HPP
#define FSTRUC_HPP

#include "value.hpp"

#include "indent.hpp"

#include <list>
#include <bitset>

namespace UKernel {

using namespace Toolbox;

// INTERNAL USE: internal storage of values for each FStruc
struct _VList: public std::list<Value> {
  _VList::iterator add (const _VList::const_iterator vi) {
    push_back(*vi);
    return --end();
  }

  _VList::iterator add (const Value &v) {
    push_back(v);
    return --end();
  }
};

struct Feature {
  // vFlag: true iff this node carries a value
  enum FFlags { vFlag, numFlags };
  typedef std::bitset<numFlags> Flags;

  Flags flags;

  Symbol name;          // feature name
  _VList::iterator vi;  // points to a particular value

  Feature () {}  // by default name.id is UNKNOWN_TOKEN_ID
  Feature (const Symbol &name, _VList::iterator vi):name(name),vi(vi) { flags[vFlag]=true; }
  Feature (const Symbol &name):name(name) {}

  void setValue (bool t) { flags[vFlag]=t; }

  bool isValue () const { return flags[vFlag]; }

  bool operator < (const Feature &f) const { return name<f.name; }
  bool operator == (const Feature &f) const { return name==f.name; }
};

typedef Tree<Feature>::PathIterator _FSPathIterator;
struct Path: public _FSPathIterator::Path {
  enum CmpResult { isPrefix,hasPrefix,equal,none };

  CmpResult compare (const Path &p) const;
};

std::ostream &operator << (std::ostream &os, const Path &path);

// INTERNAL USE: entries of a Blueprint
struct _BlueprintEntry {
  bool endTreeFlag; // true iff the entry signals the end of the current subtree
  Symbol sym;       // signals the type of a complex FS, if it's not UNKNOWN
  Tree<Feature>::ConstIterator iter;

  _BlueprintEntry (bool endTreeFlag=false):endTreeFlag(endTreeFlag) {}
  _BlueprintEntry (Symbol::SymCode c):endTreeFlag(false),sym(c) {}
  _BlueprintEntry (Tree<Feature>::ConstIterator &iter):endTreeFlag(false),iter(iter) {}
};

// struct Blueprint: For reconstructing a collected value/FS via traversing a 
//                   penetrating path
struct Blueprint: public std::list<_BlueprintEntry> {
  typedef std::list<_BlueprintEntry> _Parent;

  bool setFlag; // true iff vFlag is set
  bool vFlag;   // true iff the blueprint is for a value

  void clear () { _Parent::clear(); setFlag=false; }
};

// acronyms: FS == feature structure
//           LHS/RHS == left/right-hand side
class FStruc: protected Tree<Feature> {
  typedef bool (FStruc::*MPFuncPtr)(PathIterator&);
  typedef bool (FStruc::*RPFuncPtr)(ConstIterator&) const;
  typedef void (FStruc::*CBFuncPtr)(ConstIterator&) const;
  
  enum CollectCode {CL_FAIL,CL_BOX,CL_SUCC};

  Symbol name;
  _VList vList;   // internal storage of all Values
                  // to avoid non-terminal node in FStruc tree
                  // from carrying actual values

  // IMPORTANT: for all the following static members they imply that at 
  //   any point only one 'type' of path modification/path running/blueprint
  //   collection can take place - this is to save memory and speed up the
  //   operations

  // --------------- For use with modifyPath()/runPath() ---------------
  static const Value *vPtr;
  static ConstIterator rIter;

  // -------------------- For use with modifyPath() --------------------
  // the 3 function pointers are called in modifyPath(), when the path
  // doesn't exist, exists, or is blocked by a value
  // IMPORTANT ASSUMPTION: re-entrancy is NOT allowed!
  //                       (since there's only one copy of vPtr/rIter)
  static MPFuncPtr mpPathNotExistFuncPtr;  // called in modifyPath()
  static MPFuncPtr mpPathExistFuncPtr;
  static MPFuncPtr mpVBlockFuncPtr;

  bool mpDefaultVBlockFunc (PathIterator &pIter) { return false; }
  bool mpPUVPathNotExistFunc (PathIterator &pIter);
  bool mpPUVPathExistFunc (PathIterator &pIter);
  bool mpPUFSPathNotExistFunc (PathIterator &lPIter);
  bool mpPUFSPathExistFunc (PathIterator &lPIter);
  bool mpConstrainVPathExistFunc (PathIterator &lPIter);
  bool mpConstrainFSPathExistFunc (PathIterator &lPIter);
  bool mpTestFSPathExistFunc (PathIterator &lPIter);
  bool mpIsoVPathExistFunc (PathIterator &lPIter);
  bool mpIsoFSPathExistFunc (PathIterator &lPIter);

  bool mpDefinedPathExistFunc (PathIterator &pIter) { return true; }
  bool mpUndefinedVBlockFunc (PathIterator &pIter) { return true; }
  bool mpUndefinedPathExistFunc (PathIterator &pIter) { return false; }
  bool mpNumberPathExistFunc (PathIterator &pIter);
  bool mpIntegerPathExistFunc (PathIterator &pIter);
  bool mpPositivePathExistFunc (PathIterator &pIter);

  bool mpAssignVFunc (PathIterator &pIter);
  bool mpAssignFSFunc (PathIterator &pIter);
  bool mpPushVFunc (PathIterator &pIter);
  bool mpPushFSFunc (PathIterator &lPIter);
  // -------------------------------------------------------------------

  // -------------------- For use with runPath() --------------------
  static RPFuncPtr rpFuncPtr;  // called in runPath()

  // IMPORTANT ASSUMPTION: re-entrancy is NOT allowed!
  //                       (since there's only one copy of vPtr/rIter)
  bool rpConstrainVFunc (ConstIterator &iter) const;
  bool rpConstrainFSFunc (ConstIterator &iter) const;
  bool rpTestFSFunc (ConstIterator &iter) const;
  bool rpIsoVFunc (ConstIterator &iter) const;
  bool rpIsoFSFunc (ConstIterator &iter) const;
  // ----------------------------------------------------------------

  // ------------- For use with collectBlueprint() and collect() -------------
  static CBFuncPtr cbFuncPtr;  // called in collect()

  void cbDefaultFunc (ConstIterator &iter) const {}
  void cbPopFunc (ConstIterator &iter) const;
  // -------------------------------------------------------------------------

  // 1. The first version returns true iff the two FS trees are isomorphic
  // 2. The second version returns true iff the two FS trees are isomorphic, 
  //    up to the unifiability of values
  // ASSUMPTION: the two nodes themselves (pointed by iter1 and iter2) are equal
  //             (so no test will be done for these two nodes)
  bool isomorphicSameRoot (ConstIterator iter1, ConstIterator iter2) const;
  bool isomorphicSameRootRelaxed (ConstIterator iter1, ConstIterator iter2) const;

  // NOTES: 
  // 1. The methods modifyPath(), runPath(), remove() and collect() are all
  //    similar in their implementation for penetrating paths.
  // 2. runPath() and collect() are non-destructive; modifyPath() and remove()
  //    are.

  // TO-DO: maybe consolidate remove() with modifyPath(), and collect() with
  //        runPath()?

  // destructively run the path specified in lPIter - can penetrate complex FSs
  // ASSUMPTIONS:
  //   1. lPIter is properly initialized
  //   2. the 3 MPFuncPtr's must be pointed to appropriate functions
  //   3. in the functions pointed by 3 MPFuncPtr's do *not* prune the iterators
  //   4. set vPtr/rIter appropriately
  //   5. *this is *not* empty
  bool modifyPath (PathIterator &lPIter);

  // non-destructively run the path specified in lPIter - can penetrate complex FSs
  // ASSUMPTIONS:
  //   1. lPIter is properly initialized
  //   2. rpFuncPtr must be pointed to an appropriate function
  //   3. set vPtr/rIter appropriately
  //   4. *this is *not* empty
  bool runPath (ConstPathIterator &lPIter) const;

  // modify LHS path using the RHS (calling modifyPath() within)
  // used by =c, =t, and =i
  bool modifyByRHS (const Path &lPath, const FStruc &rFStruc, const Path &rPath,
		    MPFuncPtr vFuncPtr, MPFuncPtr fsFuncPtr);

  // run LHS path using the RHS (calling runPath() within)
  // used by =c', =t', and =i' (non-filtering versions)
  bool runByRHS (const Path &lPath, const FStruc &rFStruc, const Path &rPath,
		 RPFuncPtr vFuncPtr, RPFuncPtr fsFuncPtr) const;

  // remove from fromIter, also remove any resulting mother-with-zero-daughter 
  // upward until hitting toIter; try to reduce to an atomic FS whenever possible
  void removeIter (Iterator &fromIter, Iterator toIter);

  // remove a path; return true iff lPIter has at least one daughter left
  // ASSUMPTIONS: 
  //   1. lPIter is properly initialized w.r.t a non-empty path
  //   2. *this is *not* empty
  bool remove (PathIterator &pIter);

  // 1. it's only possible to have boxes (undefined branches) inside an *OR* FS/V
  //    and if the entire collection is reduced to box(es) then CL_BOX is returned,
  //    with *no* new entry inserted in bp; otherwise it returns CL_SUCC or
  //    CL_FAIL (due to type mismatch or blocking by a value in the middle of the 
  //    path)
  // 2. set boxSucc to false iff box is NOT allowed (so collect() will never return
  //    CL_BOX, nor will it insert a box inside bp - it returns CL_FAIL whenever an
  //    undefined branch is found)
  // 3. Don't call this directly - this is meant to be called by collectBlueprint()
  // ASSUMPTIONS:
  //   1. pIter is properly initialized w.r.t a penetrating path
  //   2. cbFuncPtr is pointed to an appropriate function
  //   3. *this is *not* empty
  CollectCode collect (ConstPathIterator &pIter, Blueprint &bp, bool boxSucc) const;

  // 1. collect a blueprint of a penetrating path; once a blueprint is collected
  //    the actual FS or V can then be collected using collectValue() or collectFS()
  // 2. boxes can only occur directly under an *OR* component, and if they occur they
  //    will be the *first* (left-most) daughter, and there'll be only one copy of
  //    them within each such *OR* component.
  // 3. see collect() for comments on boxSucc
  // 4. both collectValue() and collectFS() return true iff a box is in the collected
  //    value/FS, provided boxSucc is set true when calling collectBlueprint().
  // ASSUMPTIONS:
  //  1. For all *this is *not* empty  
  //  2. For collectBlueprint() cbFuncPtr must be pointed to an appropriate function
  //  3. For collectValue() and collectFS() bp must have size>1, which implies
  //     bp must be collected using a penetrating path
  CollectCode collectBlueprint (const Path &path, Blueprint &bp, bool boxSucc=true) const;
  bool collectValue (const Blueprint &bp, Value &v) const;
  bool collectFS (const Blueprint &bp, FStruc &fs) const;

  // extend the FS for the rest of the untraversed path; after calling pIter
  // points to the leaf node and will lose all its traversal history.
  // ASSUMPTION: pIter is set properly before calling
  void extendPath (PathIterator &pIter);

  // lIter points to a non-*MULTIPLE* FS and rIter points to a *MULTIPLE* FS
  bool multiplePU (Iterator &lIter, const ConstIterator &rIter);
  bool multipleEq (ConstIterator &lIter, const ConstIterator &rIter) const;

  // unify two FS; *lPIter is modified iff unification succeeds
  // ASSUMPTION: both iterators do not point to a *OR*/*MULTIPLE* node
  bool pseudoUnify (Iterator &lIter, const ConstIterator &rIter);

  // unifiability test; set relaxed to true if you want the test to return true
  // when a RHS feature doesn't exist in the LHS (used in filter())
  // ASSUMPTION: both iterators do not point to a *MULTIPLE* node
  bool test (ConstIterator &lIter, const ConstIterator &rIter,
	     bool relaxed=false) const;

  void copy (const FStruc &fs);

  // delete all daughters of 'iter' and the value (if any) in 'iter'
  // iter->flags[vFlag] and iter->name are not changed
  // IMPORTANT: iter itself is not deleted!
  void clear (Iterator iter);

  // reduce a complex FS pointed to by motherIter to an atom FS pointed
  // to by its first daughterIter
  void reduceToAtomFS (Iterator &motherIter);
  void reduceToAtomFSIfPossible (Iterator &motherIter);

  // remove all isomorphic *OR* daughter FSs
  void compact (Iterator &iter);

  // 1. Filter out the daughter FSs which do not agree with the FS rooted in
  //    rIter, if *this is a complex FS. If it's an atomic FS, filter against
  //    the entire FS.
  // 2. An FS agrees with fs/(fs rPath) iff for each feature existing in 
  //    both FSs, the two FSs have equal values ('=t'). For those features
  //    that exist in either but not in the other, they do NOT matter.
  // 3. filter() returns true iff the filtered FS is not empty.
  bool filter (ConstIterator &rIter);

  // remove possible box under iter; return true iff such removal takes place
  bool removeBox (Iterator iter);

  void printIter (std::ostream &os, ConstIterator iter) const;

  // IMPORTANT: 'name' is NOT copied!
  // operator=(ConstIterator iter) copies all daughters of
  // 'iter' to the cleared *this; *this could be empty after the operation.
  // ASSUMPTION: iter must not be a value node (!iter->isValue())
  FStruc &operator = (ConstIterator iter);

  friend struct FEqVIsomorphism;
  friend struct FEqVUnifiable;
  friend struct SharedArc;
  friend std::ostream &operator << (std::ostream &os, const FStruc &fs);

protected:
  
  // copy the FStruc rooted at iter to fs
  // see operator= (ConstIterator)
  void copyDaughter (FStruc &fs, ConstIterator iter) { fs=iter; }

public:

  typedef Tree<Feature>::size_type size_type;

  static bool indentPrint;  // true iff you want indented printing (default), 
                            // otherwise it's printed in one line without the name

  // the root node represents this entire FS
  FStruc () { insert(begin(),Feature()); }
  FStruc (const Symbol &sym) { name=sym; insert(begin(),Feature()); }
  FStruc (const FStruc &fs) { name=fs.name; copy(fs); }

  ~FStruc () { clear(begin()); }

  // vList is automatically cleaned up
  void clear () { clear(begin()); begin().get().setValue(false); }

  void setName (const Symbol &sym) { name=sym; }
  const Symbol &readName () const { return name; }

  bool empty () const { return (size()==1 && !begin()->isValue()); }

  // return true iff the lPath leads to an atomic value, in that case
  // sym is that atomic value; return false if lPath is a penetrating path
  bool isAtomValue (const Path &lPath, Symbol &sym) const;

  // return true iff this FS is a value
  bool isValue () const { return begin()->isValue(); }

  // return true iff this FS is a complex FS (*MULTIPLE* or *OR*)
  bool isComplex () const;

  // remove possible box inside the FS; return true iff such removal takes place
  bool removeBox () { return removeBox(begin()); }

  // 1. Remove the atomic value lPath leads to;
  // 2. Do nothing if lPath doesn't lead to an atomic value (including the
  //    case where lPath is penetrating);
  // 3. Return true iff the removal succeeds;
  // 4. 'sym' is the atomic value removed iff the removal succeeds.
  bool removeAtomValue (const Path &lPath, Symbol &sym);

  // return true iff lPath leads to an FS/V; 'fs'/'v' will be updated with the
  // FS/V (lPath can be a penetrating path)
  bool getFS (const Path &lPath, FStruc &fs) const;
  bool getValue (const Path &lPath, Value &v) const;

  // these two are for debugging purposes - they don't omit boxes
  bool getFSBox (const Path &lPath, FStruc &fs) const;
  bool getValueBox (const Path &lPath, Value &v) const;

  // merge in 'fs' to become an *OR* FS; the 2nd form doesn't check if
  // multiple same FSs are merged in (all are kept - faster)
  // CAUTION: may let RHS box features creep into LHS! BE CAREFUL!
  // ASSUMPTIONS:
  //   1. for mergeOr() 'fs' is of the same type as that of *this, i.e.,
  //      either both are values or both are non-value FSs
  //   2. for mergeOrMulti() both *this and 'fs' must be non-value FSs,
  //      and fs is *not* empty
  void mergeOr (const FStruc &fs);
  void mergeOrMulti (const FStruc &fs);

  // assign v to *this if it's an atomic FS, or to the *last* daughter FS
  // of *this if it's a complex FS; lPath starts from the root node in the
  // former case, or from the root node of the daughter FS in the latter one
  // CAUTION: may let RHS box values creep into LHS! BE CAREFUL!
  // ASSUMPTION: lPath can *not* exist in *this, even partially
  void assignComplexLast (const Path &lPath, const Value &v);

  // See the comments for filter(ConstIterator&).
  bool filter (const FStruc &fs) { ConstIterator iter; return filter(iter=fs.begin()); }
  bool filter (const FStruc &fs, const Path &rPath);

  // remove all isomorphic *OR* daughter FSs
  void compact () { Iterator iter; compact(iter=begin());  }

  // IMPORTANT: 'name' is NOT copied!
  FStruc &operator = (const FStruc &fs) { clear(begin()); copy(fs); return *this; }

  bool operator == (const FStruc &fs) const;

  // ----------------------------------------------------------------------
  // all of the following methods will NOT modify *this if the returning
  // value is false
  // ASSUMPTIONS:
  //   1. When operations involve both the left path and right path
  //      one should not be the prefix of the other; e.g. (f 1 2) = (f 1) is 
  //      not allowed!
  //   2. For all methods involving a RHS value, the value cannot have any
  //      box.
  // ----------------------------------------------------------------------

  // operator = (all paths can be penetrating paths)
  // for the 2nd version set rPathFail to true iff you want it to return false
  // when RHS is undefined (it's for removeAssign())
  bool pseudoUnify (const Path &lPath, const Value &v);
  bool pseudoUnify (const Path &lPath, const FStruc &rFStruc, const Path &rPath,
		    bool rPathFail=false);

  // operator =c (constraint equations; all paths can be penetrating paths)
  // for *OR* FS they filter out the branches that do not pass the tests
  bool constrain (const Path &lPath, const Value &v);
  bool constrain (const Path &lPath, const FStruc &rFStruc, const Path &rPath);

  // operator =c' (constraint equations; all paths can be penetrating paths)
  bool constrainNoFilter (const Path &lPath, const Value &v) const;
  bool constrainNoFilter (const Path &lPath, const FStruc &rFStruc, const Path &rPath) const;

  // operator =t (testing equations; all paths can be penetrating paths)
  // for *OR* FS they filter out the branches that do not pass the tests
  bool test (const Path &lPath, const Value &v);
  bool test (const Path &lPath, const FStruc &rFStruc, const Path &rPath);

  // operator =t' (testing equations; all paths can be penetrating paths)
  bool testNoFilter (const Path &lPath, const Value &v) const;
  bool testNoFilter (const Path &lPath, const FStruc &rFStruc, const Path &rPath) const;

  // operator =i (isomorphism equations; all paths can be penetrating paths)
  // for *OR* FS they filter out the branches that do not pass the tests
  bool isomorphism (const Path &lPath, const Value &v);
  bool isomorphism (const Path &lPath, const FStruc &rFStruc, const Path &rPath);

  // operator =i' (isomorphism equations; all paths can be penetrating paths)
  bool isomorphismNoFilter (const Path &lPath, const Value &v) const;
  bool isomorphismNoFilter (const Path &lPath, const FStruc &rFStruc, const Path &rPath) const;

  // operator <= (all paths can be penetrating paths)
  // ASSUMPTION: v has no box
  bool assign (const Path &lPath, const Value &v);   // always returns true
  bool assign (const Path &lPath, const FStruc &rFStruc, const Path &rPath);

  // operator == (all paths can be penetrating paths)
  bool removeAssign (const Path &lPath, FStruc &rFStruc, const Path &rPath);

  // operator > and < - for < 'rFStruc' is not modified if failed
  // (all paths can be penetrating paths)
  bool push (const Path &lPath, const FStruc &rFStruc, const Path &rPath);
  bool pop (const Path &lPath, FStruc &rFStruc, const Path &rPath);

  // all paths below in is*() can be penetrating paths
  // for *OR* FS it filters out the branches that do not pass the tests
  bool isDefined (const Path &lPath);   // *DEFINED*
  bool isUndefined (const Path &lPath); // *UNDEFINED*
  bool isNumber (const Path &lPath);    // is the value pointed to a real number?
  bool isInteger (const Path &lPath);   // is the value pointed to an integer?
  bool isPositive (const Path &lPath);  // is the value pointed to a positive real number?

  // *REMOVE* - always return true (lPath can be a penetrating path)
  bool remove (const Path &lPath);
};

typedef std::list<FStruc::size_type> _IndexList;

// Indented printing: you must take care of the first line (see equTst.cpp) for
// examples
std::ostream &operator << (std::ostream &os, const FStruc &fs);

// ============================ inline functions ============================

inline Path::CmpResult Path::compare (const Path &p) const {
  const_iterator pi1,pi2;

  for (pi1=begin(),pi2=p.begin();
       pi1!=end() && pi2!=p.end() && *pi1==*pi2;
       ++pi1,++pi2);

  if (pi1==end())
    if (pi2==end()) return equal;
    else return isPrefix;
  else if (pi2==end()) return hasPrefix;
  else return none;
}

inline void FStruc::removeIter (Iterator &fromIter, Iterator toIter) {
  Iterator tmpIter;

  do {
    tmpIter=fromIter;
    tmpIter.up();
    clear(fromIter);
    prune(fromIter);
    fromIter=tmpIter;
  } while (fromIter!=toIter && fromIter.size()==0);
  reduceToAtomFSIfPossible(fromIter);
}

inline FStruc::CollectCode FStruc::collectBlueprint (const Path &path, Blueprint &bp, 
						     bool boxSucc) const {
  assert(!empty());

  ConstPathIterator pIter(path);

  bp.clear();
  return collect(pIter=begin(),bp,boxSucc);
}

inline void FStruc::extendPath (FStruc::PathIterator &pIter) {
  Iterator ti;
  Path::const_iterator pi;
  Path::size_type s=pIter.pathDist();

  for (pi=pIter.readPI(),ti=pIter;s;--s,++pi) {
#ifndef NDEBUG
    DIterator dIter;
#endif
    assert(!ti.find(Feature(pi->name),dIter));
    ti=insert(ti.end(),Feature(pi->name));
  }
  
  pIter=ti;
}

inline void FStruc::reduceToAtomFS (FStruc::Iterator &motherIter) {
  assert(motherIter.size());

  DIterator daughterIter=motherIter.begin();

  assert(daughterIter->name.isSpecial(Symbol::MULTIPLE) ||
	 daughterIter->name.isSpecial(Symbol::OR));

  Feature f=*daughterIter;

  f.name=motherIter->name;
  promote(daughterIter);
  daughterIter.update(f);

  motherIter=daughterIter;
}

inline void FStruc::reduceToAtomFSIfPossible (FStruc::Iterator &motherIter) {
  if (motherIter.size()==1) {
    Iterator daughterIter=motherIter.begin();
    if (daughterIter->name.isSpecial(Symbol::OR) || 
	daughterIter->name.isSpecial(Symbol::MULTIPLE)) reduceToAtomFS(motherIter);
  }
}

inline FStruc &FStruc::operator = (FStruc::ConstIterator iter) {
  assert(!iter->isValue());

  Iterator root=begin();
  ConstDIterator dIter;
  FrontIterator fIter;

  clear();
  for (dIter=iter.begin();dIter!=iter.end();++dIter)
    for (fIter=insert(root.end(),dIter);fIter!=fIter.nullTI();++fIter) {
      assert(fIter->isValue());
      fIter.get().vi=vList.add(fIter->vi);
    }
  return *this;
}

inline bool FStruc::isAtomValue (const Path &lPath, Symbol &sym) const {
  if (empty()) return false;
  else {
    ConstPathIterator pIter(lPath);
    Path::size_type s=lPath.size();
    
    for (pIter=begin();s && pIter!=pIter.nullTI();--s,++pIter);
    if (pIter==pIter.nullTI()) return false;
    
    if (pIter->isValue() && pIter->vi->isAtom()) {
      sym=*pIter->vi->begin();
      return true;
    }
    else return false;
  }
}

inline bool FStruc::isComplex () const {
  ConstDIterator dIter=begin().begin();

  return (size()>1 && (dIter->name.isSpecial(Symbol::OR) || 
		       dIter->name.isSpecial(Symbol::MULTIPLE)));
}

inline bool FStruc::removeAtomValue (const Path &lPath, Symbol &sym) {
  if (empty()) return false;
  else {
    PathIterator pIter(lPath);
    Path::size_type s=lPath.size();

    for (pIter=begin();s && pIter!=pIter.nullTI();--s,++pIter);
    if (pIter==pIter.nullTI()) return false;
    
    if (pIter->isValue() && pIter->vi->isAtom()) {
      sym=*pIter->vi->begin();
      removeIter(pIter,begin());
      return true;
    }
    else return false;
  }
}

inline bool FStruc::getFS (const Path &lPath, FStruc &fs) const {
  Blueprint bp;
  
  cbFuncPtr=&FStruc::cbDefaultFunc;

  // box is not allowed
  if (collectBlueprint(lPath,bp,false) && !bp.vFlag) {
    if (bp.size()==1) fs=bp.begin()->iter;
    else collectFS(bp,fs);
    return true;
  }
  else return false;
}

inline bool FStruc::getValue (const Path &lPath, Value &v) const {
  Blueprint bp;
  
  cbFuncPtr=&FStruc::cbDefaultFunc;

  // box is not allowed
  if (collectBlueprint(lPath,bp,false) && bp.vFlag) {
    if (bp.size()==1) v=*(bp.begin()->iter->vi);
    else collectValue(bp,v);
    return true;
  }
  else return false;
}

inline bool FStruc::getFSBox (const Path &lPath, FStruc &fs) const {
  Blueprint bp;
  
  cbFuncPtr=&FStruc::cbDefaultFunc;

  // box is not allowed
  if (collectBlueprint(lPath,bp) && !bp.vFlag) {
    if (bp.size()==1) fs=bp.begin()->iter;
    else collectFS(bp,fs);
    return true;
  }
  else return false;
}

inline bool FStruc::getValueBox (const Path &lPath, Value &v) const {
  Blueprint bp;
  
  cbFuncPtr=&FStruc::cbDefaultFunc;

  if (collectBlueprint(lPath,bp) && bp.vFlag) {
    if (bp.size()==1) v=*(bp.begin()->iter->vi);
    else collectValue(bp,v);
    return true;
  }
  else return false;
}


inline bool FStruc::operator == (const FStruc &fs) const {
  if (begin()->isValue()!=fs.begin()->isValue()) return false;  // type mismatch
  else if (begin()->isValue()) return *(begin()->vi)==*(fs.begin()->vi);
  else if (begin()->name==fs.begin()->name)
    return isomorphicSameRoot(begin(),fs.begin());
  else return false;
}

inline bool FStruc::pseudoUnify (const Path &lPath, const Value &v) {
  PathIterator lPIter(lPath);

  vPtr=&v;
  lPIter=begin();

  mpPathNotExistFuncPtr=&FStruc::mpPUVPathNotExistFunc;
  mpPathExistFuncPtr=&FStruc::mpPUVPathExistFunc;
  mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;

  return (empty()?mpPUVPathNotExistFunc(lPIter):modifyPath(lPIter));
}

// operator =c (constraint equations) for values
inline bool FStruc::constrain (const Path &lPath, const Value &v) {
  // the only difference here compared to pseudoUnify(const Path&, const Value&)
  // is we use a *different* mpPathNotExistFuncPtr here
  PathIterator lPIter(lPath);

  vPtr=&v;
  lPIter=begin();

  mpPathNotExistFuncPtr=&FStruc::mpDefaultVBlockFunc;    // just return false from modifyPath()
  mpPathExistFuncPtr=&FStruc::mpConstrainVPathExistFunc; // NOTE: =c is only testing unifiability at values
  mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;

  return (empty()?false:modifyPath(lPIter));
}

// operator =c (constraint equations) for FSs
inline bool FStruc::constrain (const Path &lPath, const FStruc &rFStruc, const Path &rPath) {
  return modifyByRHS(lPath,rFStruc,rPath,
		     &FStruc::mpConstrainVPathExistFunc,
		     &FStruc::mpConstrainFSPathExistFunc);
}

// operator =c' (constraint equations) for values
inline bool FStruc::constrainNoFilter (const Path &lPath, const Value &v) const {
  ConstPathIterator lPIter(lPath);

  vPtr=&v;
  rpFuncPtr=&FStruc::rpConstrainVFunc;
  return (empty()?false:runPath(lPIter=begin()));
}

// operator =c' (constraint equations) for FSs
inline bool FStruc::constrainNoFilter (const Path &lPath, const FStruc &rFStruc, 
				       const Path &rPath) const {
  return runByRHS(lPath,rFStruc,rPath,
		  &FStruc::rpConstrainVFunc,
		  &FStruc::rpConstrainFSFunc);
}

// operator =t (testing equations) for values
inline bool FStruc::test (const Path &lPath, const Value &v) {
  PathIterator lPIter(lPath);

  vPtr=&v;
  lPIter=begin();

  mpPathNotExistFuncPtr=&FStruc::mpDefaultVBlockFunc;    // just return false from modifyPath()
  mpPathExistFuncPtr=&FStruc::mpConstrainVPathExistFunc; // just like =c for when RHS is a value
  mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;

  return (empty()?false:modifyPath(lPIter));
}

// operator =t (testing equations) for FSs
inline bool FStruc::test (const Path &lPath, const FStruc &rFStruc, const Path &rPath) {
  return modifyByRHS(lPath,rFStruc,rPath,
		     &FStruc::mpConstrainVPathExistFunc,
		     &FStruc::mpTestFSPathExistFunc);
}

// operator =t' (testing equations) for values
inline bool FStruc::testNoFilter (const Path &lPath, const Value &v) const {
  ConstPathIterator lPIter(lPath);

  vPtr=&v;
  rpFuncPtr=&FStruc::rpConstrainVFunc;   // same as =c for values
  return (empty()?false:runPath(lPIter=begin()));
}

// operator =t' (testing equations) for FSs
inline bool FStruc::testNoFilter (const Path &lPath, const FStruc &rFStruc, 
				  const Path &rPath) const {
  return runByRHS(lPath,rFStruc,rPath,
		  &FStruc::rpConstrainVFunc,
		  &FStruc::rpTestFSFunc);
}

// operator =i (isomorphism equations) for values
inline bool FStruc::isomorphism (const Path &lPath, const Value &v) {
  PathIterator lPIter(lPath);

  vPtr=&v;
  lPIter=begin();

  mpPathNotExistFuncPtr=&FStruc::mpDefaultVBlockFunc;    // just return false from modifyPath()
  mpPathExistFuncPtr=&FStruc::mpIsoVPathExistFunc;
  mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;

  return (empty()?false:modifyPath(lPIter));
}

// operator =i (isomorphism equations) for FSs
inline bool FStruc::isomorphism (const Path &lPath, const FStruc &rFStruc, const Path &rPath) {
  return modifyByRHS(lPath,rFStruc,rPath,
		     &FStruc::mpIsoVPathExistFunc,
		     &FStruc::mpIsoFSPathExistFunc);
}

// operator =i' (isomorphism equations) for values
inline bool FStruc::isomorphismNoFilter (const Path &lPath, const Value &v) const {
  ConstPathIterator lPIter(lPath);

  vPtr=&v;
  rpFuncPtr=&FStruc::rpIsoVFunc;
  return (empty()?false:runPath(lPIter=begin()));
}

// operator =i' (isomorphism equations) for FSs
inline bool FStruc::isomorphismNoFilter (const Path &lPath, const FStruc &rFStruc, 
					 const Path &rPath) const {
  return runByRHS(lPath,rFStruc,rPath,
		  &FStruc::rpIsoVFunc,
		  &FStruc::rpIsoFSFunc);
}

inline bool FStruc::removeAssign (const Path &lPath, FStruc &rFStruc, const Path &rPath) {
  if (pseudoUnify(lPath,rFStruc,rPath,true)) {
    rFStruc.remove(rPath);
    return true;
  }
  else return false;
}

inline bool FStruc::isDefined (const Path &lPath) {
  PathIterator lPIter(lPath);

  lPIter=begin();

  mpPathNotExistFuncPtr=&FStruc::mpDefaultVBlockFunc;  // just return false from modifyPath()
  mpPathExistFuncPtr=&FStruc::mpDefinedPathExistFunc;  // just return true from modifyPath()
  mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;
  
  return (empty()?false:modifyPath(lPIter));
}

inline bool FStruc::isUndefined (const Path &lPath) {
  PathIterator lPIter(lPath);

  lPIter=begin();

  mpPathNotExistFuncPtr=&FStruc::mpUndefinedVBlockFunc;  // just return true from modifyPath()
  mpPathExistFuncPtr=&FStruc::mpUndefinedPathExistFunc;  // just return false from modifyPath()
  mpVBlockFuncPtr=&FStruc::mpUndefinedVBlockFunc;
  
  return (empty()?true:modifyPath(lPIter));
}

inline bool FStruc::isNumber (const Path &lPath) {
  PathIterator lPIter(lPath);

  lPIter=begin();

  mpPathNotExistFuncPtr=&FStruc::mpDefaultVBlockFunc;  // just return false from modifyPath()
  mpPathExistFuncPtr=&FStruc::mpNumberPathExistFunc;
  mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;
  
  return (empty()?false:modifyPath(lPIter));
}

inline bool FStruc::isInteger (const Path &lPath) {
  PathIterator lPIter(lPath);

  lPIter=begin();

  mpPathNotExistFuncPtr=&FStruc::mpDefaultVBlockFunc;  // just return false from modifyPath()
  mpPathExistFuncPtr=&FStruc::mpIntegerPathExistFunc;
  mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;
  
  return (empty()?false:modifyPath(lPIter));
}

inline bool FStruc::isPositive (const Path &lPath) {
  PathIterator lPIter(lPath);

  lPIter=begin();

  mpPathNotExistFuncPtr=&FStruc::mpDefaultVBlockFunc;  // just return false from modifyPath()
  mpPathExistFuncPtr=&FStruc::mpPositivePathExistFunc;
  mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;
  
  return (empty()?false:modifyPath(lPIter));
}

};

#endif
