// ======================================================================
// fStruc.cpp - Feature Structure code
// 
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 120202: Benjamin Han <benhdj@cs.cmu.edu> Changed the return statement
//         of FStruc::remove() to avoid MS VC6 warning.
// 112002: Benjamin Han <benhdj@cs.cmu.edu> nullTI -> nullTI().
// 102102: Benjamin Han <benhdj@cs.cmu.edu> Made FStruc::test() work with
//         boxes; changed code with Feature::vFlag to use Feature::isValue() 
//         and Feature::setValue(); modified modifyByRHS() and runByRHS()
//         to NOT run over collected FS/V but directly over the bluepring,
//         so they correctly modeled =c, =t, =i and thir primed versions.
// 102002: Benjamin Han <benhdj@cs.cmu.edu> Changed some comments: equivalence
//         tests (=t) should be "unifiability tests" now; removed the bool 
//         argument 'first' in all filter(): it's no longer needed for GET-LEX/
//         GET-LEX-FS with :CHECK
// 101802: Benjamin Han <benhdj@cs.cmu.edu> Modifyed collect*() in FStruc
//         so undefined branches get boxes in the collected FS/V; also modified
//         various operations so that boxes work correctly with others;
//         added FStruc::removeBox() to remove possible boxes (e.g., when
//         assigning RHS to LHS).
// 101102: Benjamin Han <benhdj@cs.cmu.edu> Remove succ ref arg from 
//         FStruc::filter() - the semantics of GET-LEX/GET-LEX-FS
//         with an undefined :CHECK arg is equivalent to no checking
//         (filtering) at all.
// 100902: Benjamin Han <benhdj@cs.cmu.edu> Added the 'first' bool argument
//         to all FStruc::filter(): if it's true only the first branch of
//         an *OR* FS after filtering is kept (for the modified GET-LEX*
//         with :CHECK behavior - see eFunc.cpp).
// 092802: Benjamin Han <benhdj@cs.cmu.edu> Changed the behavior of =c 
//         (FStruc::constrain() ): it's testing the isomorphism up to the
//         unifiability of values (so =i, i.e., FStruc::isomorphism() are still
//         stricter), and it also filters out in an *OR* FS the branches that
//         do not pass the tests; changed the behavior of =t and =i so they now
//         too become filtering operators given an *OR* FS; added *NoFilter() 
//         in FStruc for the new operators =c', =t', and =i'; changed the 
//         behavior of =i so that it's in line with =c and =t when RHS is empty 
//         or not defined;
// 092102: Benjamin Han <benhdj@cs.cmu.edu> Modified both FStruc::constrain()
//         so the thing unified is the thing that's defined (discovered by
//         Donna Gates); again modified FStruc::test(), but this time make it
//         in line with FStruc::constrain() (the former one is non-destructive
//         while the latter is destructive).
// 072202: Benjamin Han <benhdj@cs.cmu.edu> Reverted the 'fix' for 
//         FStruc::test() - it was actually an intended behavior!!
// 020502: Benjamin Han <benhdj@cs.cmu.edu> bugfix in FStruc::test(): should
//         return false when either the LHS or the RHS is empty (but not both).
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Changed several int declarations
//         into the rightful size_type; namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Major revision for support of
//         penetrating paths, and other stuff, see 092601 comments in 
//         fStruc.hpp.
// 091001: Benjamin Han <benhdj@cs.cmu.edu> Revised code for unifying *OR* and
//         atomic FSs - no FS copying now.
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Renamed all equal() to test();
//         revised comments: changed "=c" to "=t".
// 090301: Benjamin Han <benhdj@cs.cmu.edu> Renamed Equivalence*<> to 
//         TIsomorphism*<>.
// 080901: Benjamin Han <benhdj@cs.cmu.edu> Fixed a potential bug in 
//         FStruc::push() - rDIter points before rPIter.begin().
// 080701: Benjamin Han <benhdj@cs.cmu.edu> Renamed dSize() to size() in 
//         _TreeIteratorBase<>; renamed Homomorphic* function objects to 
//         Equivalence*; revised extendPath(), assignPath() and walkPath()
//         in FStruc - now _PathIterator<> embeds the path so we don't need 
//         to pass Path anymore.
// 080201: Benjamin Han <benhdj@cs.cmu.edu> Renamed dBegin() and dEnd() to 
//         begin() and end() in _TreeIteratorBase<>.
// 080201: Benjamin Han <benhdj@cs.cmu.edu> Minor revision for the renaming
//         of _TreeIteratorBase<>::dFindFirst to find().
// 072001: Benjamin Han <benhdj@cs.cmu.edu> Revised using the new ways to
//         declare tree iterators.
// 071701: Benjamin Han <benhdj@cs.cmu.edu> Completed FStruc::pseudoUnify()
//         and FStruc::equal() for interactions between *OR* and the other
//         types of FSs; code cleanup.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::mergeOrMulti(), 
//         FStruc::assignComplex(), FStruc::assignComplexLast() and 
//         FStruc::filter(); revised FStruc::pseudoUnify() and FStruc::equal() 
//         (both are iterator versions) to model the behaviors of interactions
//          between atomic/complex FSs in the 1988 GenKit; revised 
//         FStruc::equal() so now the equivalence test can be relaxed. 
// 071001: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::multiplePU() and
//         FStruc::multipleEq().
// 070601: Benjamin Han <benhdj@cs.cmu.edu> Changed FStruc::append() to
//         FStruc::push(), and it's now pushing the RHS on top of the LHS.
// 070301: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::isHomomorphic();
//         operator<<(ostream&,const FStruc&) now prints *OR* FSs.
// 070201: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::mergeOr().
// 070101: Benjamin Han <benhdj@cs.cmu.edu> Fixed a brain-dead bug: moved the
//         code for dealing with *MULTIPLE* FSs from the frontend pseudoUnify()
//         and equal() to the private ones so they are invoked recursively.
// 062901: Benjamin Han <benhdj@cs.cmu.edu> Revised all FS operator-related
//         methods to conform with the descriptions in Operators.txt.
// 062501: Benjamin Han <benhdj@cs.cmu.edu> FStruc::pseudoUnify() and
//         FStruc::equal() now supports *MULTIPLE* FS; Revised
//         FStruc::extendPath() and FStruc::assignPath() to detect *MULTIPLE*
//         nodes; fixed a bug in FStruc::extendPath() - set succ to true
//         when the FS and the path are both empty; fixed a bug in both
//         FStruc::walkPath() - if the FS is empty they should return false.
// 062301: Benjamin Han <benhdj@cs.cmu.edu> FStruc::pop() now supports
//         *MULTIPLE* FS.
// 062201: Benjamin Han <benhdj@cs.cmu.edu> FStruc::append() now supports
//         *MULTIPLE* FS.
// 062101: Benjamin Han <benhdj@cs.cmu.edu> Fixed FS printing: for an FS node
//         an additional pair of parentheses is added around the printout
//         of its daughters, provided it's not a *MULTIPLE* FS node.
// 061501: Benjamin Han <benhdj@cs.cmu.edu> Fixed a bug in 
//         FStruc::extendPath() - with an empty FS and an empty path it
//         should return false; removed nullFS - init order of global
//         objects of diff translation units is undefined - it conflicts
//         with the static component of Symbol (i.e., TokenDict); revised
//         extendPath() again to allow it fails; added assignPath().
// 061301: Benjamin Han <benhdj@cs.cmu.edu> Now you can turn on/off indented
//         printing by setting FStruc::indentPrint.
// 061201: Benjamin Han <benhdj@cs.cmu.edu> Added FStruc::equal() for an 
//         FS+path RHS; added a global const nullFS.
// 052801: Benjamin Han <benhdj@cs.cmu.edu> Now we have indented printing
//         of an FStruc thanks to indent.* in Toolbox.
// 051701: Benjamin Han <benhdj@cs.cmu.edu> Removed the dependency on sstream
//         library; added operator << for Path.
// 051401: Benjamin Han <benhdj@cs.cmu.edu> Moved the static 
//         Feature::vList to FStruc.
// 050801: Benjamin Han <benhdj@cs.cmu.edu> Now a feature name is a Symbol
//         instead of an int.
// 042101: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "fStruc.hpp"

#include <stack>

namespace UKernel {

const Value *FStruc::vPtr=NULL;
FStruc::ConstIterator FStruc::rIter;
FStruc::MPFuncPtr FStruc::mpPathNotExistFuncPtr=NULL;
FStruc::MPFuncPtr FStruc::mpPathExistFuncPtr=NULL;
FStruc::MPFuncPtr FStruc::mpVBlockFuncPtr=NULL;
FStruc::RPFuncPtr FStruc::rpFuncPtr=NULL;
FStruc::CBFuncPtr FStruc::cbFuncPtr=NULL;
bool FStruc::indentPrint=true;

bool FStruc::modifyPath (FStruc::PathIterator &lPIter) {
  assert(!empty());

  Path::size_type s=lPIter.pathDist();

  for (;s && lPIter!=lPIter.nullTI();--s,++lPIter);

  // now lPIter should point to the updating point, if it exists

  if (lPIter==lPIter.nullTI()) {
    DIterator dIter;

    // path doesn't exist, or blocked by value/*OR*/*MULTIPLE*
    --lPIter;
    if (lPIter->isValue()) return (this->*(mpVBlockFuncPtr))(lPIter);
    
    assert(lPIter.size());
    
    dIter=lPIter.begin();
    if (dIter->name.isSpecial(Symbol::OR)) {
      _IndexList eraseList;
      _IndexList::const_iterator ei;
      DIterator dIter2;
      DIterator::size_type i,s=lPIter.size();
      PathIterator newPIter;
      
      for (i=0;i<s;++i,++dIter) {
	newPIter=lPIter;
	newPIter.moveTo(dIter);
	if (!modifyPath(newPIter)) eraseList.push_back(i);
      }
      
      if (eraseList.size()!=s) {
	for (ei=eraseList.begin(),i=0,dIter=lPIter.begin();ei!=eraseList.end();++ei,++i) {
	  for (;i<*ei;++i,++dIter);
	  dIter=erase(dIter);
	}
	      
	// remove isomorphic daughter FSs
	for (dIter=lPIter.begin();dIter!=lPIter.end();++dIter)
	  for (++(dIter2=dIter);dIter2!=lPIter.end();)
	    if (isomorphicSameRoot(dIter,dIter2)) dIter2=erase(dIter2);
	    else ++dIter2;
	
	// reduce to an atom if only one daughter is left
	if (lPIter.size()==1) reduceToAtomFS(lPIter);
	
	return true;
      }
      else return false;
    }
    else if (dIter->name.isSpecial(Symbol::MULTIPLE)) {
      DIterator newDIter,dBackup=dIter;  // dBackup points to the 1st daughter
      FrontIterator fIter;
      PathIterator newPIter;
	  
      for (;dIter!=lPIter.end();++dIter) {
	newPIter=lPIter;
	for (fIter=(newDIter=insert(dBackup,dIter));
	     fIter!=fIter.nullTI();++fIter) {
	  assert(fIter->isValue());
	  fIter.get().vi=vList.add(fIter->vi);
	}
	      
	newPIter.moveTo(newDIter);
	if (!modifyPath(newPIter)) {
	  // recursive call fails - remove the new daughter FSs
	  for (dIter=lPIter.begin();dIter!=dBackup;dIter=erase(dIter));
	  return false;
	}
      }
      
      // recursive call succeeds - remove the old daughter FSs
      for (dIter=dBackup;dIter!=lPIter.end();dIter=erase(dIter));
      return true;
    }
    else return (this->*(mpPathNotExistFuncPtr))(lPIter);  // path doesn't exist
  }
  else return (this->*(mpPathExistFuncPtr))(lPIter);         // path exists
}

bool FStruc::runPath (FStruc::ConstPathIterator &lPIter) const {
  assert(!empty());

  Path::size_type s=lPIter.pathDist();

  for (;s && lPIter!=lPIter.nullTI();--s,++lPIter);

  // now lPIter should point to the testing point, if it exists

  if (lPIter==lPIter.nullTI()) {
    ConstDIterator dIter;

    // path doesn't exist, or blocked by value/*OR*/*MULTIPLE*
    --lPIter;
    if (lPIter->isValue()) return false;
    
    assert(lPIter.size());
    
    dIter=lPIter.begin();
    if (dIter->name.isSpecial(Symbol::OR)) {
      bool result;
      ConstPathIterator newPIter;
      
      for (result=false;dIter!=lPIter.end();++dIter) {
	newPIter=lPIter;
	newPIter.moveTo(dIter);
	if (runPath(newPIter)) result=true;
      }
      return result;
    }
    else if (dIter->name.isSpecial(Symbol::MULTIPLE)) {
      ConstPathIterator newPIter;
	  
      for (;dIter!=lPIter.end();++dIter) {
	newPIter=lPIter;
	newPIter.moveTo(dIter);
	if (!runPath(newPIter)) return false;
      }
      return true;
    }
    else return false;  // path doesn't exist
  }
  else return (this->*(rpFuncPtr))(lPIter);         // path exists
}

bool FStruc::modifyByRHS (const Path &lPath, const FStruc &rFStruc, const Path &rPath,
			  MPFuncPtr vFuncPtr, MPFuncPtr fsFuncPtr) {
  // deal with the case where RHS FS is empty
  if (rFStruc.empty()) return true;
  else {
    Blueprint bp;
    CollectCode cc;
    
    cbFuncPtr=&FStruc::cbDefaultFunc;
    cc=rFStruc.collectBlueprint(rPath,bp);
    if (cc==CL_SUCC) {	
      // now we know rFstruc is not empty, and rPath is valid
      // if LHS is empty this must fail
      if (empty()) return false;

      // mpDefaultVBlockFunc() just returns false from modifyPath()
      mpPathNotExistFuncPtr=&FStruc::mpDefaultVBlockFunc;
      mpPathExistFuncPtr=(bp.vFlag?vFuncPtr:fsFuncPtr);
      mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;

      PathIterator lPIter(lPath);
      lPIter=begin();
      if (bp.size()==1) {
	if (bp.vFlag) vPtr=&(*(bp.begin()->iter->vi));
	else rIter=bp.begin()->iter;
	return modifyPath(lPIter);
      }
      else {
	// a penetrating path leading to a V/FS; we'll run over the
	// blueprint directly INSTEAD OF the collected V/FS
	std::stack<Symbol> symStack;
	Symbol current;
	Blueprint::const_iterator bi;
	
	bi=bp.begin();
	assert(!bi->sym.isSpecial(Symbol::UNKNOWN));
	
	for (symStack.push(bi->sym);bi!=bp.end();++bi) {
	  if (bi->sym.isSpecial(Symbol::UNKNOWN)) {
	    if (bp.vFlag) vPtr=&(*(bi->iter->vi));
	    else rIter=bi->iter;

	    if (modifyPath(lPIter)) {
	      if (current.isSpecial(Symbol::OR)) return true;
	    }
	    else if (current.isSpecial(Symbol::MULTIPLE)) return false;
	  }
	  else if (bi->endTreeFlag) {
	    current=symStack.top();
	    symStack.pop();
	  }
	  else {
	    // bi->sym is either *OR* or *MULTIPLE*
	    if (!current.isSpecial(Symbol::UNKNOWN))
	      symStack.push(current);
	    current=bi->sym;	      
	  }
	}
	
	// we run through the blueprint, which means true for *MULTIPLE*
	// and false for *OR*
	return current.isSpecial(Symbol::MULTIPLE);
      }
    }

    // return:
    // true:  iff RHS is undefined
    // false: iff type mismatch occurs or path blocked by values on RHS
    else return (cc==CL_BOX);
  } 
}

bool FStruc::runByRHS (const Path &lPath, const FStruc &rFStruc, const Path &rPath,
		       RPFuncPtr vFuncPtr, RPFuncPtr fsFuncPtr) const {
  // deal with the case where RHS FS is empty
  if (rFStruc.empty()) return true;
  else {
    Blueprint bp;
    CollectCode cc;
  
    cbFuncPtr=&FStruc::cbDefaultFunc;
    cc=rFStruc.collectBlueprint(rPath,bp);
    if (cc==CL_SUCC) {
      // now we know rFstruc is not empty, and rPath is valid
      // if LHS is empty this must fail
      if (empty()) return false;

      FStruc::rpFuncPtr=(bp.vFlag?vFuncPtr:fsFuncPtr);
      
      ConstPathIterator lPIter(lPath);
      lPIter=begin();
      if (bp.size()==1) {
	if (bp.vFlag) vPtr=&(*(bp.begin()->iter->vi));
	else rIter=bp.begin()->iter;
	return runPath(lPIter);
      }
      else {
	// a penetrating path leading to a V/FS; we'll run over the
	// blueprint directly INSTEAD OF the collected V/FS
	std::stack<Symbol> symStack;
	Symbol current;
	Blueprint::const_iterator bi;
	
	bi=bp.begin();
	assert(!bi->sym.isSpecial(Symbol::UNKNOWN));
	
	for (symStack.push(bi->sym);bi!=bp.end();++bi) {
	  if (bi->sym.isSpecial(Symbol::UNKNOWN)) {
	    if (bp.vFlag) vPtr=&(*(bi->iter->vi));
	    else rIter=bi->iter;

	    if (runPath(lPIter)) {
	      if (current.isSpecial(Symbol::OR)) return true;
	    }
	    else if (current.isSpecial(Symbol::MULTIPLE)) return false;
	  }
	  else if (bi->endTreeFlag) {
	    current=symStack.top();
	    symStack.pop();
	  }
	  else {
	    // bi->sym is either *OR* or *MULTIPLE*
	    if (!current.isSpecial(Symbol::UNKNOWN))
	      symStack.push(current);
	    current=bi->sym;	      
	  }
	}
	
	// we run through the blueprint, which means true for *MULTIPLE*
	// and false for *OR*
	return current.isSpecial(Symbol::MULTIPLE);
      }
    }

    // return:
    // true:  iff RHS is undefined
    // false: iff type mismatch occurs or path blocked by values on RHS
    else return (cc==CL_BOX);
  }
}

bool FStruc::remove (FStruc::PathIterator &pIter) {
  assert(!empty());

  Path::size_type s=pIter.pathDist();
  PathIterator start,tmpPIter;

  start=pIter;
  for (;s && pIter!=pIter.nullTI();--s,++pIter);

  // now pIter should point to the updating point, if it exists

  if (pIter==pIter.nullTI()) {
    DIterator dIter;
    bool isOr;

    // path doesn't exist, or blocked by value/*OR*/*MULTIPLE*
    --pIter;
    if (pIter->isValue()) return true;
    
    assert(pIter.size());
    
    dIter=pIter.begin();
    if ((isOr=dIter->name.isSpecial(Symbol::OR)) || 
	dIter->name.isSpecial(Symbol::MULTIPLE)) {
      PathIterator newPIter;
      
      while (dIter!=pIter.end()) {
	newPIter=pIter;
	newPIter.moveTo(dIter);
	if (remove(newPIter)) ++dIter;
	else dIter=erase(dIter);
      }

      if (pIter.size()) {
	// remove isomorphic daughter FSs	      
	if (isOr) {
	  DIterator dIter2;

	  for (dIter=pIter.begin();dIter!=pIter.end();++dIter)
	    for (++(dIter2=dIter);dIter2!=pIter.end();)
	      if (isomorphicSameRoot(dIter,dIter2)) dIter2=erase(dIter2);
	      else ++dIter2;
	}
	
	// reduce to an atom if only one daughter is left
	if (pIter.size()==1) reduceToAtomFS(pIter);
	return true;
      }
    }
    else return true;   // path doesn't exist
  }
  
  // code reaches here when a complex FS is wiped out entirely, or
  // the path doesn't exist
  removeIter(pIter,start);

  return (start.size()>0);
}

FStruc::CollectCode FStruc::collect (ConstPathIterator &pIter, Blueprint &bp,
				     bool boxSucc) const
{
  assert(!empty());
  assert(cbFuncPtr);

  Path::size_type s=pIter.pathDist();

  for (;s && pIter!=pIter.nullTI();--s,++pIter);

  // now pIter should point to the collecting point, if it exists

  if (pIter==pIter.nullTI()) {
    ConstDIterator dIter;

    // path doesn't exist, or blocked by value/*OR*/*MULTIPLE*
    --pIter;      
    if (pIter->isValue()) return CL_FAIL;
    
    // ================================================
    // IMPORTANT NOTE: BOX ONLY EXISTS IN AN *OR* FS/V!
    // ================================================
    dIter=pIter.begin();
    if (dIter->name.isSpecial(Symbol::OR)) {
      ConstPathIterator newPIter;
      CollectCode cc,result=CL_FAIL;
      bool atLeastOneSucc=false;
      
      bp.push_back(_BlueprintEntry(Symbol::OR));
      for (;dIter!=pIter.end();++dIter) {
	newPIter=pIter;
	newPIter.moveTo(dIter);	
	cc=collect(newPIter,bp,boxSucc);

	// cc         result    new result
	// -------    -------   ----------
	// CL_FAIL    CL_FAIL   CL_FAIL
        //            CL_SUCC   CL_SUCC
        //            CL_BOX    CL_BOX
        // CL_SUCC    CL_FAIL   CL_SUCC
        //            CL_SUCC   CL_SUCC
        //            CL_BOX    CL_BOX
        // CL_BOX     CL_FAIL   CL_BOX
        //            CL_SUCC   CL_BOX
        //            CL_BOX    CL_BOX

	if (cc==CL_SUCC) {
	  atLeastOneSucc=true;
	  if (result!=CL_BOX) result=CL_SUCC;
	}
	else if (cc==CL_BOX) result=CL_BOX;
      }
      switch (result) {
      case CL_SUCC: bp.push_back(_BlueprintEntry(true)); break;
      case CL_BOX:
	// only append a box if it's NOT the only thing at this level
	if (atLeastOneSucc) {
	  bp.push_back(_BlueprintEntry(Symbol::BOX)); // box is always the last in *OR*
	  bp.push_back(_BlueprintEntry(true));
	  result=CL_SUCC;          // at the next higher level it's NOT a simple box
	  break;
	}
	// if we only have a box do the following... (leaves nothing at this level)
      default: bp.pop_back();   // pop the *OR* node
      }
      
      return result;
    }
    else if (dIter->name.isSpecial(Symbol::MULTIPLE)) {
      ConstPathIterator newPIter;
      Blueprint::iterator bi;   // backup the iterator pointing to the last element
      CollectCode cc,result=CL_FAIL;	
      
      bp.push_back(_BlueprintEntry(Symbol::MULTIPLE));
      --(bi=bp.end());
      for (;dIter!=pIter.end();++dIter) {
	newPIter=pIter;
	newPIter.moveTo(dIter);
	cc=collect(newPIter,bp,boxSucc);

	// cc         result    new result
	// -------    -------   ----------
	// CL_FAIL    ******************** (return with CL_FAIL)
        // CL_SUCC    CL_FAIL   CL_SUCC
        //            CL_SUCC   CL_SUCC
        //            CL_BOX    CL_SUCC
        // CL_BOX     CL_FAIL   CL_BOX
        //            CL_SUCC   CL_SUCC
        //            CL_BOX    CL_BOX

	if (cc==CL_FAIL)  {
	  bp.erase(bi,bp.end());
	  return CL_FAIL;
	}
	else if (cc==CL_SUCC) result=CL_SUCC;
	else if (result!=CL_SUCC) result=CL_BOX;
      }

      // as long as we have at least one successful sub-collection, a box (if any)
      // will be ignored
      if (result==CL_SUCC)
	bp.push_back(_BlueprintEntry(true));
      else bp.erase(bi,bp.end());
      
      return result;
    }

    // path doesn't exist - return a box iff boxSucc is true
    else if (boxSucc) return CL_BOX;
    else return CL_FAIL;
  }
  else {
    // path exists
    (this->*(cbFuncPtr))(pIter);  // possible pIter movement
    if (bp.setFlag) {
      if (pIter->isValue()!=bp.vFlag) return CL_FAIL;   // type mismatch
    }
    else {
      bp.setFlag=true;
      bp.vFlag=pIter->isValue();
    }
    
    bp.push_back(_BlueprintEntry(pIter));
    return CL_SUCC;
  }
}

bool FStruc::collectValue (const Blueprint &bp, Value &v) const {
  assert(!empty());
  assert(bp.vFlag);
  assert(bp.size()>1 && !bp.begin()->endTreeFlag);

  Value::Iterator iter;
  Blueprint::const_iterator bi;
  bool hasBox=false;

  v.clear();
  bi=bp.begin();
  iter=v.insert(v.begin(),bi->sym);
  for (++bi;bi!=bp.end();++bi)
    if (bi->endTreeFlag) iter.up();
    else if (bi->sym.isSpecial(Symbol::UNKNOWN))
      v.insert(iter.end(),bi->iter->vi->begin());
    else if (bi->sym.isSpecial(Symbol::BOX)) {
      // box can only happen within an *OR* value
      assert(iter->isSpecial(Symbol::OR));
      v.insert(iter.begin(),bi->sym);
      hasBox=true;
    }
    else iter=v.insert(iter.end(),bi->sym);

  v.compact();

  assert(iter==iter.nullTI());
  return hasBox;
}

bool FStruc::collectFS (const Blueprint &bp, FStruc &fs) const {
  assert(!empty());
  assert(!bp.vFlag);
  assert(bp.size()>1 && !bp.begin()->endTreeFlag);

  Iterator iter;
  Blueprint::const_iterator bi;
  FrontIterator fIter;
  std::stack<Symbol> symStack;
  bool hasBox=false;

  fs.clear();
  iter=fs.begin();
  for (bi=bp.begin();bi!=bp.end();++bi)
    if (bi->endTreeFlag) {
      fs.reduceToAtomFSIfPossible(iter);
      symStack.pop();
      iter.up();
    }
    else if (bi->sym.isSpecial(Symbol::UNKNOWN)) {
      Symbol &sym=symStack.top();
	
      // for *OR* FS check if the new daughter FS is redundant
      if (sym.isSpecial(Symbol::OR)) {
	ConstDIterator dIter;

	for (dIter=iter.begin();dIter!=iter.end();++dIter)
	  if (isomorphicSameRoot(dIter,bi->iter)) break;
	if (dIter!=iter.end()) continue;
      }
      
      ConstDIterator rDIter;
      
      iter=fs.insert(iter.end(),Feature(sym));
      for (rDIter=bi->iter.begin();rDIter!=bi->iter.end();++rDIter)
	for (fIter=fs.insert(iter.end(),rDIter);fIter!=fIter.nullTI();++fIter) {
	  assert(fIter->isValue());
	  fIter.get().vi=fs.vList.add(fIter->vi); 
	}
      iter.up();  // point back to *OR* or *MULTIPLE* node
    }
    else if (bi->sym.isSpecial(Symbol::BOX)) {
      Symbol &sym=symStack.top();
      Value v;

      // Box can only happen within an *OR* FS
      assert(sym.isSpecial(Symbol::OR));

      iter=fs.insert(iter.begin(),Feature(sym));                // the *OR* node
      v.insert(v.begin(),bi->sym);                              // the BOX value
      fs.insert(iter.end(),Feature(bi->sym,fs.vList.add(v)));   // the BOX feature
      iter.up();  // point back to *OR* node

      hasBox=true;
    }
    else {
      if (!symStack.empty()) iter=fs.insert(iter.end(),symStack.top());
      symStack.push(bi->sym);
    }

  assert(symStack.empty());
  assert(iter==iter.nullTI());
  return hasBox;
}

// lIter points to a non-*MULTIPLE* FS and rIter points to a *MULTIPLE* FS
bool FStruc::multiplePU (FStruc::Iterator &lIter, 
			 const FStruc::ConstIterator &rIter) {
  assert(!lIter->isValue() && !lIter.begin()->name.isSpecial(Symbol::MULTIPLE));
  assert(!rIter->isValue() && rIter.begin()->name.isSpecial(Symbol::MULTIPLE));

  ConstDIterator rDIter;
  DIterator newDIter,lDIter,dBackup=lIter.begin();
  FrontIterator fIter;
  Feature multipleHead(symMultiple);
  
  for (rDIter=rIter.begin();rDIter!=rIter.end();++rDIter) {
    // copy the entire subtree rooted at lIter to the first daughter of
    // lIter - DO NOT copy the new daughter, i.e., do not dive into infinite
    // recursion
    newDIter=insert(lIter.begin(),multipleHead);
    for (lDIter=dBackup;lDIter!=lIter.end();++lDIter)
      for (fIter=insert(newDIter.end(),lDIter);fIter!=fIter.nullTI();++fIter) {
	assert(fIter->isValue());
	fIter.get().vi=vList.add(fIter->vi);
      }
    
    if (!pseudoUnify(newDIter,rDIter)) {
      for (lDIter=lIter.begin();lDIter!=dBackup;lDIter=erase(lDIter));
      return false;
    }
  }
  
  for (lDIter=dBackup;lDIter!=lIter.end();lDIter=erase(lDIter));
  return true;
}

// lIter points to a non-*MULTIPLE* FS and rIter points to a *MULTIPLE* FS
bool FStruc::multipleEq (FStruc::ConstIterator &lIter, 
			 const FStruc::ConstIterator &rIter) const {
  assert(!lIter->isValue() && !lIter.begin()->name.isSpecial(Symbol::MULTIPLE));
  assert(!rIter->isValue() && rIter.begin()->name.isSpecial(Symbol::MULTIPLE));

  ConstDIterator rDIter;

  for (rDIter=rIter.begin();rDIter!=rIter.end();++rDIter)
    if (!test(lIter,rDIter)) return false;

  return true;
}

struct SharedArc {
  FStruc::DIterator dIter;
  FStruc fs;
};

// ASSUMPTION: both iterators do not point to a *MULTIPLE* node
bool FStruc::pseudoUnify (FStruc::Iterator &lIter,
			  const FStruc::ConstIterator &rIter) {
  typedef std::list<SharedArc> SharedArcs;
  typedef std::list<ConstDIterator> NewArcs;
  
  DIterator lDIter,dIter;
  ConstDIterator rDIter;
  FrontIterator fIter;
  Iterator seRoot;
  SharedArc sharedArc;
  SharedArcs sharedArcs;
  SharedArcs::iterator si;
  NewArcs newArcs;
  NewArcs::const_iterator ni;
  size_type i,s;

  // -----------------------------------------------------------------
  // Step 0 - fork sub-unifications for complex FSs
  // -----------------------------------------------------------------

  lDIter=lIter.begin();
  rDIter=rIter.begin();

  switch (lDIter->name.readCode()) {
  case Symbol::MULTIPLE: {         // ---------- LHS is a *MULTIPLE*
    DIterator dBackup=lDIter;   // dBackup points to the 1st daughter

    if (rDIter->name.readCode()!=Symbol::MULTIPLE) {      
      // the RHS is NOT a *MULTIPLE*

      // we don't worry about possible RHS boxes here - it's delegated to the
      // recursive pseudoUnify() calls

      for (;lDIter!=lIter.end();++lDIter) {
	// note if the unification is successful the result is pushed
	// to the *front* of the final FS
	for (fIter=(dIter=insert(lIter.begin(),lDIter));fIter!=fIter.nullTI();++fIter) {
	  assert(fIter->isValue());
	  fIter.get().vi=vList.add(fIter->vi);
	}

	if (!pseudoUnify(dIter,rIter)) {
	  for (lDIter=lIter.begin();lDIter!=dBackup;lDIter=erase(lDIter));
	  return false;
	}
      }
      
      for (lDIter=dBackup;lDIter!=lIter.end();lDIter=erase(lDIter));
      return true;
    }
    else {
      // the RHS is a *MULTIPLE*
      for (;lDIter!=lIter.end();++lDIter)
	for (rDIter=rIter.begin();rDIter!=rIter.end();++rDIter) {
	  // note if the unification is successful the result is pushed
	  // to the *front* of the final FS
	  for (fIter=(dIter=insert(lIter.begin(),lDIter));fIter!=fIter.nullTI();++fIter) {
	    assert(fIter->isValue());
	    fIter.get().vi=vList.add(fIter->vi);
	  }
	  
	  if (!pseudoUnify(dIter,rDIter)) {
	    for (lDIter=lIter.begin();lDIter!=dBackup;lDIter=erase(lDIter));
	    return false;
	  }
	}
	    
      for (lDIter=dBackup;lDIter!=lIter.end();lDIter=erase(lDIter));
      return true;
    }
  }

  case Symbol::OR:                // ---------- LHS is an *OR*
    switch (rDIter->name.readCode()) {
    case Symbol::MULTIPLE: return multiplePU(lIter,rIter);

    case Symbol::OR: {
      DIterator di,dBackup=lDIter;   // dBackup points to the 1st daughter
      ConstDIterator rhsStart;
      bool hasBox=rDIter.begin()->name.isSpecial(Symbol::BOX);

      // this excludes the leading RHS daughter from unifications if it's a box	
      rhsStart=rIter.begin();
      if (hasBox) ++rhsStart;
      
      for (lDIter=dBackup;lDIter!=lIter.end();++lDIter) {
	// dIter points to the new daughter FS
	for (fIter=(dIter=insert(dBackup,lDIter));fIter!=fIter.nullTI();++fIter) {
	  assert(fIter->isValue());
	  fIter.get().vi=vList.add(fIter->vi);
	}
		
	rDIter=rhsStart;
	do {
	  if (pseudoUnify(dIter,rDIter)) {
	    // check to see if dIter is isomorphic to any of the
	    // new daughter FSs
	    for (di=lIter.begin();di!=dIter && !isomorphicSameRoot(dIter,di);++di);
	    if (di!=dIter) prune(dIter);
	    
	    if (++rDIter!=rIter.end()) {
	      // prepare the next fresh copy
	      for (fIter=(dIter=insert(dBackup,lDIter));
		   fIter!=fIter.nullTI();++fIter) {
		assert(fIter->isValue());
		fIter.get().vi=vList.add(fIter->vi);
	      }
	    }
	    else break;
	  }
	  // unification fails
	  else if (++rDIter==rIter.end()) {
	    prune(dIter);
	    break;
	  }
	} while (true);
      }
      
      if (dBackup!=lIter.begin()) {
	// unification succeeded - erase the old FS
	for (lDIter=dBackup;lDIter!=lIter.end();lDIter=erase(lDIter));
	if (lIter.size()==1) reduceToAtomFS(lIter);
	return true;
      }
      // if all unifications fail, return true iff we have a box
      else return hasBox;
    }
      
    default: {
      _IndexList eraseList;
      _IndexList::const_iterator ei;
      DIterator::size_type i,s=lIter.size();
      
      for (i=0;i<s;++i,++lDIter)
	if (!pseudoUnify(lDIter,rIter)) eraseList.push_back(i);
      
      if (eraseList.size()!=s) {
	for (ei=eraseList.begin(),i=0,lDIter=lIter.begin();ei!=eraseList.end();++ei,++i) {
	  for (;i<*ei;++i,++lDIter);
	  lDIter=erase(lDIter);
	}
		
	// remove isomorphic daughter FSs
	for (lDIter=lIter.begin();lDIter!=lIter.end();++lDIter)
	  for (++(dIter=lDIter);dIter!=lIter.end();)
	    if (isomorphicSameRoot(lDIter,dIter)) dIter=erase(dIter);
	    else ++dIter;
	
	// reduce to an atom if only one daughter is left
	if (lIter.size()==1) reduceToAtomFS(lIter);
	
	return true;
      }
      else return false;
    }
    }
    
  default:                        // ---------- LHS is an atom
    switch (rDIter->name.readCode()) {
    case Symbol::MULTIPLE: return multiplePU(lIter,rIter);

    case Symbol::OR: {
      Feature orHead(symOr);
      DIterator dBackup=lDIter;   // dBackup points to the 1st daughter
      bool hasBox=rDIter.begin()->name.isSpecial(Symbol::BOX);

      // copy the entire subtree rooted at lIter to the first daughter of
      // lIter - DO NOT copy the new daughter, i.e., do not dive into infinite
      // recursion
      dIter=insert(dBackup,orHead);
      for (lDIter=dBackup;lDIter!=lIter.end();++lDIter)
	for (fIter=insert(dIter.end(),lDIter);fIter!=fIter.nullTI();++fIter) {
	  assert(fIter->isValue());
	  fIter.get().vi=vList.add(fIter->vi);
	}

      // this excludes the leading RHS daughter from unifications if it's a box	
      rDIter=rIter.begin();
      if (hasBox) ++rDIter;
      do {
	if (pseudoUnify(dIter,rDIter)) {
	  // check to see if dIter is isomorphic to any of the
	  // new daughter FSs
	  for (lDIter=lIter.begin();
	       lDIter!=dIter && !isomorphicSameRoot(dIter,lDIter);++lDIter);
	  if (lDIter!=dIter) prune(dIter);
	  
	  if (++rDIter!=rIter.end()) {
	    // prepare the next fresh copy
	    dIter=insert(dBackup,orHead);
	    for (lDIter=dBackup;lDIter!=lIter.end();++lDIter)
	      for (fIter=insert(dIter.end(),lDIter);fIter!=fIter.nullTI();++fIter) {
		assert(fIter->isValue());
		fIter.get().vi=vList.add(fIter->vi);
	      }
	  }
	  else break;
	}
	// unification fails
	else if (++rDIter==rIter.end()) {
	  prune(dIter);
	  break;
	}
      } while (true);
      
      if (dBackup!=lIter.begin()) {
	// unification succeeded - erase the old FS
	for (lDIter=dBackup;lDIter!=lIter.end();lDIter=erase(lDIter));
	if (lIter.size()==1) reduceToAtomFS(lIter);
	return true;
      }
      // if all unifications fail, return true iff we have a box
      else return hasBox;
    }
    }
  }

  // -----------------------------------------------------------------
  // Step 1 - Atomic FS vs. atomic FS: collect all unified shared arcs
  //          and new arcs
  // -----------------------------------------------------------------

  seRoot=sharedArc.fs.begin();
  for (;rDIter!=rIter.end();++rDIter)
    if (lIter.find(*rDIter,lDIter)) {
      if (lDIter->isValue()!=rDIter->isValue()) return false; // value type mismatch
      else {
	sharedArc.dIter=lDIter;
	if (lDIter->isValue()) {
	  // both are values
	  seRoot.get().setValue(true);
	  seRoot.get().vi=vList.add(lDIter->vi);
	  if (!seRoot->vi->pseudoUnify(*rDIter->vi)) {
	    sharedArc.fs.clear();
	    return false;
	  }
	}
	else {
	  // both are FS
	  
	  // copy *lDIter into sharedArc.fs
	  seRoot.get().setValue(false);
	  for (dIter=lDIter.begin(),i=0,s=lDIter.size();i<s;++dIter,++i)
	    sharedArc.fs.insert(seRoot.end(),dIter);

	  // var duplication
	  for (fIter=seRoot;fIter!=fIter.nullTI();++fIter) {
	    assert(fIter->isValue());
	    fIter.get().vi=vList.add(fIter->vi);
	  }

	  if (!sharedArc.fs.pseudoUnify(seRoot,rDIter)) {
	    sharedArc.fs.clear();
	    return false;
	  }
	}
	
	sharedArcs.push_back(sharedArc);	    
	sharedArc.fs.clear();
      }
    }
    else newArcs.push_back(rDIter);

  // -----------------------------------------------------------------
  // Step 2 - unification succeeds - update the FS
  // -----------------------------------------------------------------

  for (si=sharedArcs.begin();si!=sharedArcs.end();++si) {
    seRoot=si->fs.begin();

    clear(si->dIter);
    si->dIter.get().setValue(seRoot->isValue());
    si->dIter.get().vi=seRoot->vi;
    
    for (dIter=seRoot.begin(),i=0,s=seRoot.size();i<s;++dIter,++i)
      insert(si->dIter.end(),dIter);
    
    // var duplication
    for (fIter=si->dIter;fIter!=fIter.nullTI();++fIter) {
      assert(fIter->isValue());
      fIter.get().vi=vList.add(fIter->vi);
    }
  }
  for (ni=newArcs.begin();ni!=newArcs.end();++ni)
    // var duplication
    for (fIter=insert(lIter.end(),*ni);fIter!=fIter.nullTI();++fIter) {
      assert(fIter->isValue());
      fIter.get().vi=vList.add(fIter->vi);
    }
  
  return true;
}

// ASSUMPTION: both iterators do not point to a *MULTIPLE* node
bool FStruc::test (FStruc::ConstIterator &lIter,
		   const FStruc::ConstIterator &rIter,
		   bool relaxed) const {
  ConstDIterator lDIter,rDIter;
  
  // fork sub unifiability tests for complex FS
  lDIter=lIter.begin();
  rDIter=rIter.begin();

  switch (lDIter->name.readCode()) {
  case Symbol::MULTIPLE:          // ---------- LHS is a *MULTIPLE*


    if (rDIter->name.readCode()!=Symbol::MULTIPLE) {
      // the RHS is NOT a *MULTIPLE*
      // we don't worry about possible RHS boxes here - it's delegated to the
      // recursive test() calls
      for (;lDIter!=lIter.end();++lDIter)
	if (!test(lDIter,rIter,relaxed)) return false;
      return true;
    }
    else {
      // the RHS is a *MULTIPLE*
      for (;lDIter!=lIter.end();++lDIter)
	for (rDIter=rIter.begin();rDIter!=rIter.end();++rDIter)
	  if (!test(lDIter,rDIter,relaxed)) return false;
      return true;
    }

  case Symbol::OR:                // ---------- LHS is an *OR*
    switch (rDIter->name.readCode()) {
    case Symbol::MULTIPLE: return multipleEq(lIter,rIter);

    case Symbol::OR:
      // a box in the leading RHS daughter shortcuts the test
      if (rDIter.begin()->name.isSpecial(Symbol::BOX)) return true;

      for (lDIter=lIter.begin();lDIter!=lIter.end();++lDIter)
	for (rDIter=rIter.begin();rDIter!=rIter.end();++rDIter)
	  if (test(lDIter,rDIter)) return true;
      return false;
      
    default:
      for (lDIter=lIter.begin();lDIter!=lIter.end();++lDIter)
	if (test(lDIter,rIter)) return true;
      return false;
    }

  default:                        // ---------- LHS is an atom
    switch (rDIter->name.readCode()) {
    case Symbol::MULTIPLE: return multipleEq(lIter,rIter);

    case Symbol::OR:
      // a box in the leading RHS daughter shortcuts the test
      if (rDIter.begin()->name.isSpecial(Symbol::BOX)) return true;

      for (rDIter=rIter.begin();rDIter!=rIter.end();++rDIter)
	if (test(lIter,rDIter)) return true;
      return false;
      
    default:
      for (;rDIter!=rIter.end();++rDIter)
	if (lIter.find(*rDIter,lDIter))
	  if (lDIter->isValue()!=rDIter->isValue()) 
	    return false; // value type mismatch
	  else
	    // both are values
	    if (lDIter->isValue()) {
	      if (!lDIter->vi->test(*rDIter->vi)) return false;
	    }
            // both are FS
	    else {
	      if (!test(lDIter,rDIter,relaxed)) return false;
	    }
	else return relaxed;
      
      return true;
    }
  }
}

void FStruc::copy (const FStruc &fs) {
  FrontIterator fIter;

  Tree<Feature>::operator=(fs);

  // duplicate all vars
  if (fs.size()==1) {
    Iterator iter=begin();
    if (iter->isValue()) iter.get().vi=vList.add(iter->vi);
  }
  else
    for (fIter=begin();fIter!=fIter.nullTI();++fIter) {
      assert(fIter->isValue());
      fIter.get().vi=vList.add(fIter->vi);
    }
}

void FStruc::clear (FStruc::Iterator iter) {
  if (iter->isValue()) {
    assert(iter.size()==0);
    vList.erase(iter->vi);
  }
  else if (iter.size()) {
    DIterator dIter;
    FrontIterator fIter;

    // delete all vars
    for (dIter=iter.begin();dIter!=iter.end();dIter=erase(dIter))
      for (fIter=dIter;fIter!=fIter.nullTI();++fIter) {
	assert(fIter->isValue());
	vList.erase(fIter->vi);
      }
  }
}

void FStruc::compact (FStruc::Iterator &iter) {
  // no compaction for atoms
  if (!iter.size()) return;
  
  DIterator iter1,iter2;

  for (iter1=iter.begin();iter1!=iter.end();++iter1) compact(iter1);

  // Only try to remove isomorphic *OR* daughter FSs
  if ((iter1=iter.begin())->name.isSpecial(Symbol::OR)) {
    for (;iter1!=iter.end();++iter1)
      for (++(iter2=iter1);iter2!=iter.end();)
	if (isomorphicSameRoot(iter1,iter2)) iter2=erase(iter2);
	else ++iter2;
    
    if (iter.size()==1) reduceToAtomFS(iter);
  }
}

bool FStruc::filter (ConstIterator &rIter) {
  if (empty()) return false;

  ConstIterator lConstDIter;
  Iterator lIter=begin();

  if (isComplex()) {
    DIterator lDIter;
    Iterator::size_type s;
    
    for (lDIter=lIter.begin();lDIter!=lIter.end();)
      if (test(lConstDIter=lDIter,rIter,true)) ++lDIter;
      else lDIter=erase(lDIter);

    if ((s=lIter.size())==0) return false;
    else {
      if (s==1) reduceToAtomFS(lIter);
      return true;
    }
  }
  else
    if (test(lConstDIter=lIter,rIter,true)) return true;
    else {
      clear();
      return false;
    }
}


// INTERNAL USE: for traversing each and every box
template <class T, class Ref, class Ptr, class TNode, class Cmp>
struct _BoxAgenda {
  _TreeIteratorBase<T,Ref,Ptr,TNode,Cmp> start;
  
  void init (_TreeIteratorBase<T,Ref,Ptr,TNode,Cmp> &i) {
    if ((start=i)->isValue() || !i.begin()->name.isSpecial()) {
      // impossible to have boxes
      i=i.nullTI();
      return;
    }

    _TreeDIterator<T,Ref,Ptr,TNode,Cmp> di=i.begin().begin();
    do {
      switch (di->name.readCode()) {
      case Symbol::OR:
      case Symbol::MULTIPLE: di=di.begin(); break;

      case Symbol::BOX: i=di.up(); return;

      default:  
	// for atomic FSs: move to a node to the right
	while (++di==di.nullTI())
	  if ((di=di.up())==start) {
	    i=i.nullTI();
	    return;
	  }
      }
    } while (true);
  }
  
  void visitNext (_TreeIteratorBase<T,Ref,Ptr,TNode,Cmp> &i) {
    _TreeDIterator<T,Ref,Ptr,TNode,Cmp> di=i;

    // move to a node to the right
    while (++di==di.nullTI())
      if ((di=di.up())==start) {
	i=i.nullTI();
	return;
      }

    do {
      switch (di->name.readCode()) {
      case Symbol::OR:
      case Symbol::MULTIPLE: di=di.begin(); break;

      case Symbol::BOX: i=di.up(); return;

      default:  
	// for atomic FSs: move to a node to the right
	while (++di==di.nullTI())
	  if ((di=di.up())==start) {
	    i=i.nullTI();
	    return;
	  }
      }
    } while (true);
  }

  void remove (const _TreeIteratorBase<T,Ref,Ptr,TNode,Cmp> &ti) {}
};

bool FStruc::removeBox (Iterator iter) { 
  // to workaround "macro uses ',' as delimiter" problem
  typedef _TreeNode<Feature,std::less<Feature> > FSNode;
  TreeIterator(Feature,_BoxAgenda,FSNode,std::less<Feature>) bi;
  bool result=false;

  for (bi=iter;bi!=bi.nullTI();result=true,bi=erase(bi));
  return result;

}

void FStruc::mergeOr (const FStruc &fs) {
  DIterator lIter=begin();
  ConstIterator rIter=fs.begin();
  Feature f;

  assert(lIter->isValue()==rIter->isValue());

  if (lIter->isValue()) lIter->vi->mergeOr(*rIter->vi);
  else if (empty()) *this=fs;
  else if (fs.empty()) return;
  else {
    ConstDIterator lDIter,rDIter;
    FrontIterator fIter;
    _IndexList mergeList;
    size_type i;

    lDIter=lIter.begin();
    rDIter=rIter.begin();

    // collect all RHS FSs to be merged
    if (rDIter->name.isSpecial(Symbol::OR)) {
      // the RHS is an *OR* FS
      if (lDIter->name.isSpecial(Symbol::OR))
	for (i=0;rDIter!=rIter.end();++i,++rDIter) {
	  // see if rDIter points to an existing daughter FS of the LHS
	  for (lDIter=lIter.begin();
	       lDIter!=lIter.end() && !isomorphicSameRoot(lDIter,rDIter);
	       ++lDIter);
	  if (lDIter==lIter.end()) mergeList.push_back(i);
	}
      else
	for (i=0;rDIter!=rIter.end();++i,++rDIter)
	  if (!isomorphicSameRoot(lIter,rDIter)) mergeList.push_back(i);
      
      if (!mergeList.size()) return;         // nothing to merge
    }
    else
      // the RHS is NOT an *OR* FS	
      if (lDIter->name.isSpecial(Symbol::OR)) {
	for (;lDIter!=lIter.end();++lDIter)
	  if (isomorphicSameRoot(lDIter,rIter)) return;  // nothing to merge
      }
      else if (isomorphicSameRoot(lIter,rIter)) return;  // nothing to merge
    
    // if LHS is not an *OR* FS, make it so
    if (!lIter.begin()->name.isSpecial(Symbol::OR)) {
      // default Feature is a non-value node
      demote(lIter,Feature(lIter->name));
      
      // change the name of *lIter via _TreeIteratorBase<>::update()
      // must do this way to ensure correct indexing
      f=*lIter;
      f.name.setSpecial(Symbol::OR);
      lIter.update(f);
      
      lIter.up();
    }

    // the actual merging starts here
    if (mergeList.size()) {
      // the RHS is an *OR* FS
      assert(rIter.begin()->name.isSpecial(Symbol::OR));
      
      _IndexList::const_iterator mi;
      
      for (mi=mergeList.begin(),i=0,rDIter=rIter.begin();
	   mi!=mergeList.end();++mi,++i,++rDIter) {
	for (;i<*mi;++i,++rDIter);
	for (fIter=insert(lIter.end(),rDIter);fIter!=fIter.nullTI();++fIter) {
	  assert(fIter->isValue());
	  fIter.get().vi=vList.add(fIter->vi);
	}
      }
    }
    else {
      // the RHS is not an *OR* FS
      assert(!rIter.begin()->name.isSpecial(Symbol::OR));
      
      Iterator iter=insert(lIter.end(),rIter);
      
      // change the name of *iter via _TreeIteratorBase<>::update()
      // must do this way to ensure correct indexing
      f=*iter;
      f.name.setSpecial(Symbol::OR);
      iter.update(f);
      
      for (fIter=iter;fIter!=fIter.nullTI();++fIter) {
	assert(fIter->isValue());
	fIter.get().vi=vList.add(fIter->vi);
      }
    }
  }
}

void FStruc::mergeOrMulti (const FStruc &fs) {
  assert(!fs.empty());

  DIterator lIter=begin();
  ConstIterator rIter=fs.begin();
  Feature f;

  assert(lIter->isValue()==rIter->isValue() && !lIter->isValue());

  if (empty()) *this=fs;
  else {
    ConstDIterator lDIter,rDIter;
    FrontIterator fIter;
    
    lDIter=lIter.begin();
    rDIter=rIter.begin();
    
    // if LHS is not an *OR* FS, make it so
    if (!lIter.begin()->name.isSpecial(Symbol::OR)) {
      // default Feature is a non-value node
      demote(lIter,Feature(lIter->name));
      
      // change the name of *lIter via _TreeIteratorBase<>::update()
      // must do this way to ensure correct indexing
      f=*lIter;
      f.name.setSpecial(Symbol::OR);
      lIter.update(f);
      
      lIter.up();
    }
    
    // the actual merging starts here
    if (rDIter->name.isSpecial(Symbol::OR)) {
      // the RHS is an *OR* FS
      assert(rIter.size());
      
      for (;rDIter!=rIter.end();++rDIter)
	for (fIter=insert(lIter.end(),rDIter);fIter!=fIter.nullTI();++fIter) {
	  assert(fIter->isValue());
	  fIter.get().vi=vList.add(fIter->vi);
	}
    }
    else {
      // the RHS is not an *OR* FS
      Iterator iter=insert(lIter.end(),rIter);
      
      // change the name of *iter via _TreeIteratorBase<>::update()
      // must do this way to ensure correct indexing
      f=*iter;
      f.name.setSpecial(Symbol::OR);
      iter.update(f);
      
      for (fIter=iter;fIter!=fIter.nullTI();++fIter) {
	assert(fIter->isValue());
	fIter.get().vi=vList.add(fIter->vi);
      }
    }
  }
}

void FStruc::assignComplexLast (const Path &lPath, const Value &v) {
  PathIterator lPIter(lPath);

  if (isComplex()) lPIter=(--(begin().end()));
  else lPIter=begin();

  extendPath(lPIter);
  lPIter.get().setValue(true);
  lPIter.get().vi=vList.add(v);
}

bool FStruc::filter (const FStruc &fs, const Path &rPath) {

  if (fs.empty()) return (!empty());
  else {
    Blueprint bp;
    CollectCode cc;
    
    cbFuncPtr=&FStruc::cbDefaultFunc;
    cc=fs.collectBlueprint(rPath,bp,false); // don't allow box in the filter FS
    if (cc==CL_SUCC && !bp.vFlag) {
      if (bp.size()==1) return filter(bp.begin()->iter);
      else {
	FStruc tmpFS;
	ConstIterator iter;
	
	fs.collectFS(bp,tmpFS);
	return filter(iter=tmpFS.begin());
      }
    }
    else return false;
  }
}

// operator =
bool FStruc::pseudoUnify (const Path &lPath, const FStruc &rFStruc, const Path &rPath,
			  bool rPathFail) {
  // deal with the case where RHS FS is empty
  if (rFStruc.empty()) return (rPath.size()?!rPathFail:true);    
  else {
    Blueprint bp;
    CollectCode cc;
    
    cbFuncPtr=&FStruc::cbDefaultFunc;
    cc=rFStruc.collectBlueprint(rPath,bp);
    if (cc==CL_SUCC)
      if (bp.vFlag)
	// TO-DO: if bp has only one element, the RHS value can't have box;
	//        maybe we can have an alternative version of pseudoUnify()
	//        which assumes no-box property (thus doesn't do removeBox()
	//        within FStruc::mpPUVPathNotExistFunc() and can run faster)
	if (bp.size()==1) return pseudoUnify(lPath,*(bp.begin()->iter->vi));
	else {
	  // a penetrating path leading to a value
	  Value v;
	  
	  rFStruc.collectValue(bp,v);
	  return pseudoUnify(lPath,v);
	}
      else {
	PathIterator lPIter(lPath);

	lPIter=begin();
	mpPathNotExistFuncPtr=&FStruc::mpPUFSPathNotExistFunc;
	mpPathExistFuncPtr=&FStruc::mpPUFSPathExistFunc;
	mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;
	
	// TO-DO: if bp has only one element, the RHS FS can't have box;
	//        maybe we can have an alternative version of pseudoUnify()
	//        which assumes no-box property (thus doesn't do removeBox()
	//        within FStruc::mpPUFSPathNotExistFunc() and can run faster)
	if (bp.size()==1) {
	  rIter=bp.begin()->iter;
	  return (empty()?mpPUFSPathNotExistFunc(lPIter):
		  modifyPath(lPIter));
	}
	else {
	  // a penetrating path leading to an FS
	  FStruc fs;
	  
	  rFStruc.collectFS(bp,fs);
	  rIter=fs.begin();	  
	  return (empty()?mpPUFSPathNotExistFunc(lPIter):
		  modifyPath(lPIter));
	}
      }

    // return:
    // !rPathFail: iff RHS is undefined
    // false:      iff type mismatch occurs or path blocked by values on RHS
    else if (cc==CL_BOX) return (!rPathFail);
    else return false;
  }
}

// operator <=
// ASSUMPTION: v has no box
bool FStruc::assign (const Path &lPath, const Value &v) {
  PathIterator lPIter(lPath);

  vPtr=&v;
  lPIter=begin();
  mpPathNotExistFuncPtr=mpPathExistFuncPtr=mpVBlockFuncPtr=
    &FStruc::mpAssignVFunc;

  if (empty()) mpAssignVFunc(lPIter);
  else modifyPath(lPIter);

  return true;
}

// operator <=
bool FStruc::assign (const Path &lPath, const FStruc &rFStruc, const Path &rPath) {
  if (rFStruc.empty()) remove(lPath);  // RHS is empty
  else {
    Blueprint bp;
    CollectCode cc;
  
    cbFuncPtr=&FStruc::cbDefaultFunc;
    cc=rFStruc.collectBlueprint(rPath,bp,false);
    if (cc==CL_SUCC)
      
      if (bp.vFlag)
	if (bp.size()==1) return assign(lPath,*(bp.begin()->iter->vi));
	else {
	  // a penetrating path leading to a value
	  Value v;
	  
	  if (rFStruc.collectValue(bp,v) && v.removeBox())
	    v.compact();
	  return assign(lPath,v);
	}
      else {
	PathIterator lPIter(lPath);
	
	lPIter=begin();
	mpPathNotExistFuncPtr=mpPathExistFuncPtr=mpVBlockFuncPtr=
	  &FStruc::mpAssignFSFunc;
	
	if (bp.size()==1) {
	  rIter=bp.begin()->iter;
	  return (empty()?mpAssignFSFunc(lPIter):modifyPath(lPIter));
	}
	else {
	  // a penetrating path leading to an FS
	  FStruc fs;
	  
	  if (rFStruc.collectFS(bp,fs) && fs.removeBox())
	    fs.compact();
	  rIter=fs.begin();	  
	  return (empty()?mpAssignFSFunc(lPIter):modifyPath(lPIter));
	}
      }
    
    else remove(lPath);  // RHS is undefined
  }
  
  return true;
}

// operator >
bool FStruc::push (const Path &lPath, const FStruc &rFStruc, const Path &rPath) {
  if (rFStruc.empty()) return true;  // RHS is empty
  else {
    Blueprint bp;
    CollectCode cc;
  
    cbFuncPtr=&FStruc::cbDefaultFunc;
    cc=rFStruc.collectBlueprint(rPath,bp,false);
    if (cc==CL_SUCC) {
      PathIterator lPIter(lPath);
      
      lPIter=begin();	
      if (bp.vFlag) {
	mpPathNotExistFuncPtr=mpPathExistFuncPtr=&FStruc::mpPushVFunc;
	mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;
	
	if (bp.size()==1) {
	  vPtr=&(*(bp.begin()->iter->vi));
	  return (empty()?mpAssignVFunc(lPIter):modifyPath(lPIter));
	}
	else {
	  // a penetrating path leading to a value
	  Value v;
	  
	  if (rFStruc.collectValue(bp,v) && v.removeBox())
	    v.compact();
	  vPtr=&v;
	  return (empty()?mpAssignVFunc(lPIter):modifyPath(lPIter));
	}
      }
      else {
	mpPathNotExistFuncPtr=mpPathExistFuncPtr=&FStruc::mpPushFSFunc;
	mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;
	
	if (bp.size()==1) {
	  rIter=bp.begin()->iter;
	  return (empty()?mpAssignFSFunc(lPIter):modifyPath(lPIter));
	}
	else {
	  // a penetrating path leading to an FS
	  FStruc fs;
	  
	  if (rFStruc.collectFS(bp,fs) && fs.removeBox())
	    fs.compact();
	  rIter=fs.begin();	  
	  return (empty()?mpAssignFSFunc(lPIter):modifyPath(lPIter));
	}
      }
    }
  }
  
  return true;   // RHS is undefined
}

// operator <
bool FStruc::pop (const Path &lPath, FStruc &rFStruc, const Path &rPath) {
  if (rFStruc.empty()) return false;  // RHS is empty
  else {
    Blueprint bp;
    bool result;
    
    cbFuncPtr=&FStruc::cbPopFunc;
    if (rFStruc.collectBlueprint(rPath,bp,false)==CL_SUCC) {
      if (bp.vFlag) {
	Value v;
	
	if (bp.size()==1) bp.begin()->iter->vi->top(v);
	else {
	  // a penetrating path leading to a value
	  Value tmpV;
		  
	  rFStruc.collectValue(bp,tmpV);
	  tmpV.top(v);
	}
	result=pseudoUnify(lPath,v);  // v may have boxes
      }
      else {
	PathIterator lPIter(lPath);
	
	lPIter=begin();
	mpPathNotExistFuncPtr=&FStruc::mpPUFSPathNotExistFunc;
	mpPathExistFuncPtr=&FStruc::mpPUFSPathExistFunc;
	mpVBlockFuncPtr=&FStruc::mpDefaultVBlockFunc;
	
	if (bp.size()==1) {
	  rIter=bp.begin()->iter;
	  result=(empty()?mpPUFSPathNotExistFunc(lPIter):
		  modifyPath(lPIter));
	}
	else {
	  // a penetrating path leading to an FS
	  FStruc fs;
	  
	  rFStruc.collectFS(bp,fs);
	  rIter=fs.begin();
	  result=(empty()?mpPUFSPathNotExistFunc(lPIter):
		  modifyPath(lPIter));
	}
      }
      
      if (result) {
	// unification succeeded - need to pop the RHS
	Blueprint::const_iterator bi=bp.begin();
	Iterator iter;
	
	if (bp.vFlag)
	  
	  // ========== pop values ==========
	  if (bp.size()==1) {
	    bi->iter.makeNonConstIterator(iter);
	    if (!iter->vi->pop())
	      // remove the empty value
	      if (iter==rFStruc.begin()) {
		rFStruc.clear();
		return true;  // no need for compaction
	      }
	      else rFStruc.removeIter(iter,rFStruc.begin());
	  }
	  else
	    for (;bi!=bp.end();++bi) {
	      if (!bi->endTreeFlag && bi->sym.isSpecial(Symbol::UNKNOWN)) {
		bi->iter.makeNonConstIterator(iter);
		if (!iter->vi->pop())
		  // remove the empty value
		  rFStruc.removeIter(iter,rFStruc.begin());
	      }
	    }
	else
	  
	  // ========== pop FSs ==========
	  if (bp.size()==1) {
	    bi->iter.makeNonConstIterator(iter);
	    if (iter==rFStruc.begin()) {
	      rFStruc.clear();
	      return true;  // no need for compaction
	    }
	    else rFStruc.removeIter(iter,rFStruc.begin());
	  }
	  else
	    for (;bi!=bp.end();++bi)
	      if (!bi->endTreeFlag && bi->sym.isSpecial(Symbol::UNKNOWN)) {
		bi->iter.makeNonConstIterator(iter);
		rFStruc.removeIter(iter,rFStruc.begin());
	      }
	
	rFStruc.compact();	     
	return true;
      }
    }
  }

  return false;   // RHS is undefined, or the unification failed
}

// *REMOVE* - always return true
bool FStruc::remove (const Path &lPath) {
  if (!empty())
    if (lPath.size()) {
      PathIterator lPIter(lPath);
      remove(lPIter=begin());
    }
    else clear();
  
  return true;
}

std::ostream &operator << (std::ostream &os, const Path &path) {
  Path::const_iterator pi;
  bool first;
  for (pi=path.begin(),first=true;pi!=path.end();++pi)
    if (first) {
      os<<pi->name;
      first=false;
    }
    else os<<' '<<pi->name;
  return os;
}

// ASSUMPTION: iter points to a non-value node
void FStruc::printIter (std::ostream &os, FStruc::ConstIterator iter) const {
  ConstDIterator dIter;
  bool notFirst;

  if (iter->name.isSpecial(Symbol::BOX)) {
    os<<iter->name;
    return;
  }

  for (dIter=iter.begin(),notFirst=false;dIter!=iter.end();++dIter) {
    if (notFirst)
      if (FStruc::indentPrint) {
	os<<std::endl;
	indent::print(os);
      }
      else os<<' ';
    else notFirst=true;
    if (dIter->isValue()) os<<'('<<dIter->name<<' '<<*dIter->vi<<')';
    else if (dIter->name.isSpecial(Symbol::MULTIPLE) ||
	     dIter->name.isSpecial(Symbol::OR)) {
      if (dIter.isFirstD()) {
	os<<dIter->name;
	if (FStruc::indentPrint) {
	  os<<std::endl;
	  indent::print(os);
	}
	else os<<' ';
      }
      os<<'('<<indent(1);
      printIter(os,dIter);
      os<<')'<<indent(-1);
    }
    else {
      os<<'('<<dIter->name;
      if (FStruc::indentPrint) {
	os<<std::endl;
	indent::print(os)<<' ';
      }
      else os<<' ';
      
      os<<'('<<indent(2);
      printIter(os,dIter);
      os<<"))"<<indent(-2);
    }
  }
}

std::ostream &operator << (std::ostream &os, const FStruc &fs) {
  int iLevel;

  if (FStruc::indentPrint) {
    os<<fs.name<<": ";
    if (fs.begin()->isValue()) os<<*fs.begin()->vi;
    else {
      iLevel=fs.name.read().length()+3;	  
      os<<'('<<indent(iLevel);
      fs.printIter(os,fs.begin());
      os<<')'<<indent(-iLevel);
    }
  }
  else if (fs.begin()->isValue()) os<<*fs.begin()->vi;
  else {
    os<<'(';
    fs.printIter(os,fs.begin());
    os<<')';
  }
  
  return os;
}

};
