// ======================================================================
// equation.hpp - This models a single equation.
// 
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 110602: Benjamin Han <benhdj@cs.cmu.edu> _EquType4 has a new reference member
//         g of type Grammar: extension functions now know where to get the
//         global resource from the corresponding grammar.
// 103002: Benjamin Han <benhdj@cs.cmu.edu> Added _EquType5 (type 5 equations)
//         to deal with "(lFS lPath) op QFS" (QFS = quoted FS).
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Changed several int declarations
//         into the FSRegisters::size_type; namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Added _Equation::runTracing().
// 090601: Benjamin Han <benhdj@cs.cmu.edu> Changed Symbol::OP_EQUAL to 
//         Symbol::OP_TEST.
// 090301: Benjamin Han <benhdj@cs.cmu.edu> Minor revisions to supress warnings
//         under -pedantic -Wall.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Revised the equation type 4 to be
//         the interface to the extension functions, via which the lexicon
//         facility is now implemented; removed equation type 5-10 (no more
//         next-lex but much more flexible get-lex/get-lex-fs); moved
//         FSRegisters code to fsRegs.*.
// 063001: Benjamin Han <benhdj@cs.cmu.edu> Added FSRegisters::copy(); added
//         class FSIdxSet; FSRegisters::add() now returns nothing.
// 062901: Benjamin Han <benhdj@cs.cmu.edu> Changed the way an FS is addressed -
//         now at compile time only the index of an FS in FSRegisters is kept,
//         and when running one needs to suppy a copy of FSRegisters; moved
//         FSRegsiters from grammar.* to here; added FSRegisters(int).
// 062401: Benjamin Han <benhdj@cs.cmu.edu> Revised to use FStruc::isAtom()
//         in Slot::isAtom() and to use FStruc::removeAtom() in the constructors
//         for _EquType7 and _EquType8.
// 062101: Benjamin Han <benhdj@cs.cmu.edu> Added equation type 9 and 10
//         (for get-lex-fs equations with unquoted (FS PATH)); augmented
//         _EquType4 accordingly.
// 061401: Benjamin Han <benhdj@cs.cmu.edu> Added all lexical equations
//         (type 4 to 8); changed type 3 EBlockMain constructor prototype
//         - the ending argument is changed from const Symbol& to
//         Symbol::SymCode.
// 052001: Benjamin Han <benhdj@cs.cmu.edu> Removed Equations and added
//         eBlock.* for "equation blocks" (and, or, eor, case) - now
//         equation.* models a single equation.
// 051801: Benjamin Han <benhdj@cs.cmu.edu> Now each type has its own run()
//         and print(), which are two virtual functions (today's compilers
//         should make this as efficient as using switch statements).
// 051701: Benjamin Han <benhdj@cs.cmu.edu> Changed Equations::run() into
//         a const method.
// 051501: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef EQUATION_HPP
#define EQUATION_HPP

#include "eFunc.hpp"

#include <algorithm>
#include <iostream>

namespace UKernel {

// INTERNAL USE: an equation
class _Equation {
protected:

  FSPath lFSPath;

public:

  _Equation (FSRegisters::size_type lFSIdx, const Path &lPath):lFSPath(lFSIdx,lPath) {}
  virtual ~_Equation () {}

  virtual bool run (FSRegisters &fsRegs) const=0;
  virtual void print (std::ostream &os) const=0;

  bool runTracing (FSRegisters &fsRegs, std::ostream &out, int iLevel) const;
};

std::ostream &operator << (std::ostream &os, const _Equation &e);

// INTERNAL USE: a type 1 equation of form "(lFS lPath) op (rFS rPath)"
class _EquType1: public _Equation {
  Symbol::SymCode opCode;
  FSPath rFSPath;

public:

  _EquType1 (FSRegisters::size_type lFSIdx, const Path &lPath, 
	     FSRegisters::size_type rFSIdx, const Path &rPath, Symbol::SymCode opCode)
    :_Equation(lFSIdx,lPath),opCode(opCode),rFSPath(rFSIdx,rPath) {}

  bool run (FSRegisters &fsRegs) const;
  void print (std::ostream &os) const;
};

// INTERNAL USE: a type 2 equation of form "(lFS lPath) op v"
class _EquType2: public _Equation {
  Symbol::SymCode opCode;
  Value v;

public:

  _EquType2 (FSRegisters::size_type lFSIdx, const Path &lPath, const Value &v, Symbol::SymCode opCode)
    :_Equation(lFSIdx,lPath),opCode(opCode),v(v) {}

  bool run (FSRegisters &fsRegs) const;
  void print (std::ostream &os) const;
};

// INTERNAL USE: a type 3 equation of form "(lFS lPath) = sym"
class _EquType3: public _Equation {
  Symbol::SymCode symCode;

public:

  _EquType3 (FSRegisters::size_type lFSIdx, const Path &lPath, Symbol::SymCode symCode)
    :_Equation(lFSIdx,lPath),symCode(symCode) {}

  bool run (FSRegisters &fsRegs) const;
  void print (std::ostream &os) const;
};

// INTERNAL USE: a type 4 equation of form "(lFS lPath) op (EFunc req opt)"
//               where EFunc is an extension function, req and opt are two
//               EFArgs for required and optional arguments
class _EquType4: public _Equation {
  static _EFTable efTable;
  static Path emptyPath;

  EFunc *efPtr;
  EFArgs req,opt;         // nominal args
  Symbol::SymCode opCode;

  Grammar &g;

public:

  // ASSUMPTION: each EFArg in 'opt' must have a name
  _EquType4 (FSRegisters::size_type lFSIdx, const Path &lPath, const Symbol &funcName, 
	     const EFArgs &req, const EFArgs &opt,
	     Symbol::SymCode opCode,Grammar &g);

  bool run (FSRegisters &fsRegs) const;
  void print (std::ostream &os) const;
};

// INTERNAL USE: a type 5 equation of form "(lFS lPath) op QFS" (QFS = quoted FS)
class _EquType5: public _Equation {
  Symbol::SymCode opCode;
  FStruc fs;

public:

  _EquType5 (FSRegisters::size_type lFSIdx, const Path &lPath, const FStruc &fs, Symbol::SymCode opCode)
    :_Equation(lFSIdx,lPath),opCode(opCode),fs(fs) {}

  bool run (FSRegisters &fsRegs) const;
  void print (std::ostream &os) const;
};


// ============================ inline functions ============================

inline bool _Equation::runTracing (FSRegisters &fsRegs, std::ostream &out, int iLevel) const {
  bool result;

  fill_n(std::ostream_iterator<char>(out),iLevel,' ');
  out<<"Run \'";
  print(out);
  out<<"\' : ";
  result=run(fsRegs);
  out<<(result?"T":"NIL");
  return result;
}

};

#endif
