// ======================================================================
// eFunc.hpp - Extension Function for easier system extension - a kind of
//            callback mechanism.
// 
// 011403: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 122502: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 110602: Benjamin Han <benhdj@cs.cmu.edu> All EFunc::operator() receives
//         an additional argument 'g' (Grammar) so the extension functions
//         can get global resources from the corresponding grammar.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Changed several int declarations
//         into the rightful size_type; namespace added.
// 090301: Benjamin Han <benhdj@cs.cmu.edu> Minor revisions to supress warnings
//         under -pedantic -Wall.
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef EFUNC_HPP
#define EFUNC_HPP

#include "lexicon.hpp"
#include "eFArgs.hpp"

namespace UKernel {

class Grammar;   // forward declaration since all extension 

// ==================== EFunc-related declarations ====================

// inherit EFunc to implement your extension
struct EFunc {
  Symbol name;

  EFunc (const Symbol &name):name(name) {}
  virtual ~EFunc () {}

  // default processArgs() does nothing
  // default print() prints out name, req, and opt, in that order
  virtual void processArgs (EFArgs &req, EFArgs &opt) {}
  virtual void print (std::ostream &os, const EFArgs &req, const EFArgs &opt);

  // return true and update result, iff the execution is successful
  // returning false will cause the host equation to fail
  // ASSUMPTION: each EFArg in 'opt' must have a name
  virtual bool operator () (Grammar &g, const FSRegisters &fsRegs, 
			    const EFArgs &req, const EFArgs &opt,
			    FStruc &result)=0;
};

// ==================== _EFTable-related declarations ====================

// INTERNAL USE: an entry for an EFunc - opt args conversion table and a pointer
// to the function obj
struct _EFuncEntry {
  EFArgConvTable table;
  EFunc *efPtr;

  _EFuncEntry (const EFArgConvTable &table, EFunc *efPtr):table(table),efPtr(efPtr) {}
};

// INTERNAL USE
typedef std::set<Symbol> _EFOptKey;
struct _EFOptEntry: public std::map<_EFOptKey,_EFuncEntry> {
  typedef std::map<_EFOptKey,_EFuncEntry> _Parent;
  void insert (const EFArgConvTable &table, EFunc *efPtr);
};
typedef std::map<int,_EFOptEntry> _EFReqEntry;  // (# of req arguments, signature of opt)

// INTERNAL USE: table storing the prototype info (signature) of each EFunc
class _EFTable: protected std::map<Symbol,_EFReqEntry> {
  typedef std::map<Symbol,_EFReqEntry> _Parent;

  bool initFlag;

public:

  void clear ();

  _EFTable ():initFlag(false) {}
  ~_EFTable () { clear(); }

  bool isInit () const { return initFlag; }

  // call this before looking up in the table; this is to avoid a static init
  // in _EquFunc, because Symbol has a static component in it.
  void init ();

  // find the correct function to call: return NULL iff none can be found
  // both req and opt will be processed and opt will be reordered if necessary
  EFunc *find (const Symbol &funcName, EFArgs &req, EFArgs &opt) const;
};

// ============================ inline functions ============================

inline void EFunc::print (std::ostream &os, const EFArgs &req, const EFArgs &opt) {
  EFArgs::size_type i,s=opt.size();

  os<<name<<' '<<req;
  if (s) {
    for (i=0;i<s && !opt.readFlags(i)[EFArg::isSet];++i);
    if (i<s) os<<' '<<opt;
  }
}

};

#endif
