// ======================================================================
// eFunc.cpp - Extension Function for easier system extension - a kind of
//            callback mechanism.
// 
// 011403: Benjamin Han <benhdj@cs.cmu.edu> removed "using namespace std;" 
//         and inserted std:: prefix where needed.
// 110602: Benjamin Han <benhdj@cs.cmu.edu> All EFunc::operator() receives
//         an additional argument 'g' (Grammar) so the extension functions
//         can get global resources from the corresponding grammar; replaced
//         Lexicons::getActiveLexicon() call to g.lex (now that we have 'g'
//         argument).
// 102402: Benjamin Han <benhdj@cs.cmu.edu> Bug fix in NULL checkFSPtr:
//         don't call FStruc::filter() with NULL!
// 102002: Benjamin Han <benhdj@cs.cmu.edu> Reflect the non-const find*()
//         methods (see lexicon.hpp); changed the implementation of 
//         GET-LEX/GET-LEX-FS with :CHECK to use findFirst()/findNext() so
//         they are now faster.
// 101102: Benjamin Han <benhdj@cs.cmu.edu> Remove 'succ' reference argument
//         from _GetLexBase::check() - the semantics of GET-LEX/GET-LEX-FS
//         with an undefined :CHECK arg is equivalent to no checking
//         (filtering) at all.
// 100902: Benjamin Han <benhdj@cs.cmu.edu> Changed GetLex*() behavior: if
//         :CHECK is present but not :AMBIGUITY, we need to return the
//         first FS of an *OR* result after filtering.
// 120601: Benjamin Han <benhdj@cs.cmu.edu> Changed several int declarations
//         into the rightful size_type; namespace added.
// 092601: Benjamin Han <benhdj@cs.cmu.edu> Changed calls to 
//         FStruc::removeAtom() to FStruc::removeAtomValue().
// 071301: Benjamin Han <benhdj@cs.cmu.edu> Created.
// ======================================================================

//    Copyright (C) 2000-2004 Benjamin Han <benhdj@cs.cmu.edu>
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "eFunc.hpp"
#include "grammar.hpp"  // thie caused cyclic inclusion, blocked by #ifdef

namespace UKernel {

struct _GetLexBase: public EFunc {
  Symbol sem,cat,lexID;
  Path semValuePath,emptyPath;
  Value vSem;

  // for setCheckFS()
  FStruc checkFS;
  const FStruc *checkFSPtr;

  _GetLexBase (const Symbol &name):EFunc(name) { semValuePath.push_back(symSemValue); }

  // set checkFSPtr according to args[argIdx] (whether it's quoted or an FSPATH)
  void setCheckFS (const EFArgs &args, EFArgs::size_type argIdx, 
		   const FSRegisters &fsRegs) {
    assert(args.readFlags(argIdx)[EFArg::isSet] && 
	   (args.readFlags(argIdx)[EFArg::tFS] || args.readFlags(argIdx)[EFArg::tFSPath]));

      if (args.readFlags(argIdx)[EFArg::tFS]) checkFSPtr=&args.readFS(argIdx);
      else {
	const FSPath &fsPath=args.readFSPath(argIdx);

	if (fsRegs[fsPath.fsIdx].getFS(fsPath.path,checkFS))
	  checkFSPtr=&checkFS;
	else checkFSPtr=NULL;
      }
  }

  // return true iff the filtered result is not empty
  bool check (const EFArgs &args, EFArgs::size_type argIdx, const FSRegisters &fsRegs, 
	      FStruc &result) {
    setCheckFS(args,argIdx,fsRegs);
    return (checkFSPtr?result.filter(*checkFSPtr):true);
  }

  void setFailedResult (FStruc &result, const Symbol &sem) {
    vSem.insert(vSem.begin(),sem);
    result.assign(semValuePath,vSem);
    vSem.clear();
  }
};

// get-lex:
// req: (sem, cat)
// opt: ()
struct GetLex0: public _GetLexBase {
  GetLex0 (const Symbol &name):_GetLexBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && req.isAtom(1,fsRegs,cat)) {
      if (!g.lex.findLex(sem,cat,result))
	setFailedResult(result,sem);	
      return true;
    }
    
    return false;
  }
};

// get-lex:
// req: (sem, cat)
// opt: (lexID)
struct GetLex1_1: public _GetLexBase {
  GetLex1_1 (const Symbol &name):_GetLexBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && req.isAtom(1,fsRegs,cat) && 
	opt.isAtom(0,fsRegs,lexID)) {
      Lexicon &lexicon=g.lex.findLexicon(lexID);
      
      if (lexicon.readID()==symNull || !lexicon.findLex(sem,cat,result))
	setFailedResult(result,sem);
      return true;
    }
    return false;
  }
};

// get-lex:
// req: (sem, cat)
// opt: (check)
struct GetLex1_2: public _GetLexBase {
  GetLex1_2 (const Symbol &name):_GetLexBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && req.isAtom(1,fsRegs,cat)) {
      Lexicons &l=g.lex;      

      setCheckFS(opt,0,fsRegs);
      if (l.findFirst(sem,cat,result))
	do {
	  // result will be cleared if filter() returns false
	  if (checkFSPtr==NULL || result.filter(*checkFSPtr))
	    return true;
	} while (l.findNext(result));

      setFailedResult(result,sem);
      return true;
    }
    return false;
  }
};

// get-lex:
// req: (sem, cat)
// opt: (ambiguity)
struct GetLex1_3: public _GetLexBase {
  GetLex1_3 (const Symbol &name):_GetLexBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && req.isAtom(1,fsRegs,cat)) {
      if (!g.lex.findLex(sem,cat,result,opt.readBool(0)))
	setFailedResult(result,sem);
      return true;
    }    
    return false;
  }
};

// get-lex:
// req: (sem, cat)
// opt: (lexID, check)
struct GetLex2_1: public _GetLexBase {
  GetLex2_1 (const Symbol &name):_GetLexBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && req.isAtom(1,fsRegs,cat) && 
	opt.isAtom(0,fsRegs,lexID)) {
      Lexicon &lexicon=g.lex.findLexicon(lexID);

      setCheckFS(opt,1,fsRegs);
      if (lexicon.readID()!=symNull && lexicon.findFirst(sem,cat,result))
	do {
	  // result will be cleared if filter() returns false
	  if (checkFSPtr==NULL || result.filter(*checkFSPtr))
	    return true;
	} while (lexicon.findNext(result));
      
	setFailedResult(result,sem);
	return true;
    }
    return false;
  }
};

// get-lex:
// req: (sem, cat)
// opt: (lexID, ambiguity)
struct GetLex2_2: public _GetLexBase {
  GetLex2_2 (const Symbol &name):_GetLexBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && req.isAtom(1,fsRegs,cat) && 
	opt.isAtom(0,fsRegs,lexID)) {
      Lexicon &lexicon=g.lex.findLexicon(lexID);

      if (lexicon.readID()==symNull || !lexicon.findLex(sem,cat,result,opt.readBool(1)))
	setFailedResult(result,sem);
      return true;
    }
    return false;
  }
};

// get-lex:
// req: (sem, cat)
// opt: (check, ambiguity)
struct GetLex2_3: public _GetLexBase {
  GetLex2_3 (const Symbol &name):_GetLexBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && req.isAtom(1,fsRegs,cat)) {
      if (g.lex.findLex(sem,cat,result,opt.readBool(1)) &&
	  check(opt,0,fsRegs,result))
	return true;
      else {
	setFailedResult(result,sem);
	return true;
      }
    }    
    return false;
  }
};

// get-lex:
// req: (sem, cat)
// opt: (lexID, check, ambiguity)
struct GetLex3: public _GetLexBase {
  GetLex3 (const Symbol &name):_GetLexBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && req.isAtom(1,fsRegs,cat) && 
	opt.isAtom(0,fsRegs,lexID)) {
      Lexicon &lexicon=g.lex.findLexicon(lexID);
      
      if (lexicon.readID()!=symNull && lexicon.findLex(sem,cat,result,opt.readBool(2)) &&
	  check(opt,1,fsRegs,result))
	return true;
      else {
	setFailedResult(result,sem);
	return true;
      }
    }
    return false;
  }
};

// INTERNAL USE: Base func obj for GET-LEX-FS
// 1. For args preprocessing: we need to pull out the CAT value from the 
//    required FS argument, if any, and put it in the 3rd required arg.
// 2. For printing: we need to print the removed CAT in the required args,
//    as if the preprocessing is never done.
struct _GetLexFSBase: public _GetLexBase {
  Path catPath;

  _GetLexFSBase (const Symbol &name):_GetLexBase(name) { catPath.push_back(symCat); }

  void processArgs (EFArgs &req, EFArgs &opt) {
    // if the 2nd req arg is a quoted FS, pull out the CAT feature if the
    // arg has it and it's an atomic value; store the CAT in the 3rd arg

    const EFArg::Flags &flags=req.readFlags(1);

    assert(flags[EFArg::isSet] && (flags[EFArg::tFS] || flags[EFArg::tFSPath]));

    if (flags[EFArg::tFS]) {
      // quoted FS
      if (req.getFS(1).removeAtomValue(catPath,cat)) {
	// store the CAT value in the 3rd req arg
	Value &v=req.newValue();
	v.insert(v.begin(),cat);
      }
    }
  }

  void print (std::ostream &os, const EFArgs &req, const EFArgs &opt) {
    EFArgs::size_type i,s=opt.size();
    
    os<<name<<' ';
    if (req.readFlags(0)[EFArg::tValue]) os<<req.readValue(0);
    else os<<req.readFSPath(0);
    os<<' ';
    if (req.readFlags(1)[EFArg::tFSPath]) os<<req.readFSPath(1);
    else {
      bool backup=FStruc::indentPrint;
      
      FStruc::indentPrint=false;
      if (req.size()==2) os<<"\'"<<req.readFS(1);
      else {
	// we need to put CAT back in
	FStruc fs=req.readFS(1);
	fs.assign(catPath,req.readValue(2));
	os<<"\'"<<fs;
      }
      FStruc::indentPrint=backup;
    }
    
    if (s) {
      for (i=0;i<s && !opt.readFlags(i)[EFArg::isSet];++i);
      if (i<s) os<<' '<<opt;
    }
  }
};

// get-lex-fs:
// req: (sem, fs)
// opt: ()
struct GetLexFS0: public _GetLexFSBase {
  GetLexFS0 (const Symbol &name):_GetLexFSBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem)) {
      Lexicons &l=g.lex;
      const EFArg::Flags &flags=req.readFlags(1);
      
      assert(flags[EFArg::isSet] && (flags[EFArg::tFS] || flags[EFArg::tFSPath]));
      
      if (flags[EFArg::tFS]) {
	// quoted FS
	if ((req.size()==3?
	     l.findLex(sem,*req.readValue(2).begin(),req.readFS(1),result):
	     l.findLex(sem,req.readFS(1),result))==0)
	  setFailedResult(result,sem);
      }
      else {
	// (FS Path)
	const FSPath &fsPath=req.readFSPath(1);
	FStruc fs;
	
	if (fsRegs[fsPath.fsIdx].getFS(fsPath.path,fs)) {
	  if ((fs.removeAtomValue(catPath,cat)?
	       l.findLex(sem,cat,fs,result):l.findLex(sem,fs,result))==0)
	    setFailedResult(result,sem);
	}
	else return false;
      }
      return true;
    }
    return false;
  }
};

// get-lex-fs:
// req: (sem, fs)
// opt: (lexID)
struct GetLexFS1_1: public _GetLexFSBase {
  GetLexFS1_1 (const Symbol &name):_GetLexFSBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && opt.isAtom(0,fsRegs,lexID)) {
      Lexicon &lexicon=g.lex.findLexicon(lexID);
      
      if (lexicon.readID()==symNull) setFailedResult(result,sem);
      else {
	const EFArg::Flags &flags=req.readFlags(1);
	    
	assert(flags[EFArg::isSet] && (flags[EFArg::tFS] || flags[EFArg::tFSPath]));
	
	if (flags[EFArg::tFS]) {
	  // quoted FS
	  if ((req.size()==3?
	       lexicon.findLex(sem,*req.readValue(2).begin(),req.readFS(1),result):
	       lexicon.findLex(sem,req.readFS(1),result))==0)
	    setFailedResult(result,sem);
	}
	else {
	  // (FS Path)
	  const FSPath &fsPath=req.readFSPath(1);
	  FStruc fs;
	  
	  if (fsRegs[fsPath.fsIdx].getFS(fsPath.path,fs)) {
	    if ((fs.removeAtomValue(catPath,cat)?
		 lexicon.findLex(sem,cat,fs,result):
		 lexicon.findLex(sem,fs,result))==0)
	      setFailedResult(result,sem);
	  }
	  else return false;
	}
      }
      return true;
    }
    return false;
  }
};

// get-lex-fs:
// req: (sem, fs)
// opt: (check)
struct GetLexFS1_2: public _GetLexFSBase {
  GetLexFS1_2 (const Symbol &name):_GetLexFSBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem)) {
      Lexicons &l=g.lex;
      const EFArg::Flags &flags=req.readFlags(1);

      assert(flags[EFArg::isSet] && (flags[EFArg::tFS] || flags[EFArg::tFSPath]));

      setCheckFS(opt,0,fsRegs);

      if (flags[EFArg::tFS]) {
	// quoted FS
	if ((req.size()==3?
	     l.findFirst(sem,*req.readValue(2).begin(),req.readFS(1),result):
	     l.findFirst(sem,req.readFS(1),result))>0)
	  do {
	    // result will be cleared if filter() returns false	    
	    if (checkFSPtr==NULL || result.filter(*checkFSPtr))
	      return true;
	  } while (l.findNext(result));
      }
      else {
	// (FS Path)
	const FSPath &fsPath=req.readFSPath(1);
	FStruc fs;
	
	if (fsRegs[fsPath.fsIdx].getFS(fsPath.path,fs)) {
	  if ((fs.removeAtomValue(catPath,cat)?
	       l.findFirst(sem,cat,fs,result):
	       l.findFirst(sem,fs,result))>0)
	    do {
	      // result will be cleared if filter() returns false	    
	      if (checkFSPtr==NULL || result.filter(*checkFSPtr))
		return true;
	    } while (l.findNext(result));
	}
	else return false;
      }

      setFailedResult(result,sem);
      return true;
    }
    return false;
  }
};

// get-lex-fs:
// req: (sem, fs)
// opt: (ambiguity)
struct GetLexFS1_3: public _GetLexFSBase {
  GetLexFS1_3 (const Symbol &name):_GetLexFSBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem)) {
      Lexicons &l=g.lex;
      const EFArg::Flags &flags=req.readFlags(1);
      
      assert(flags[EFArg::isSet] && (flags[EFArg::tFS] || flags[EFArg::tFSPath]));
      
      if (flags[EFArg::tFS]) {
	// quoted FS
	if ((req.size()==3?
	     l.findLex(sem,*req.readValue(2).begin(),req.readFS(1),result,opt.readBool(0)):
	     l.findLex(sem,req.readFS(1),result,opt.readBool(0)))==0)
	  setFailedResult(result,sem);
      }
      else {
	// (FS Path)
	const FSPath &fsPath=req.readFSPath(1);
	FStruc fs;
	
	if (fsRegs[fsPath.fsIdx].getFS(fsPath.path,fs)) {
	  if ((fs.removeAtomValue(catPath,cat)?
	       l.findLex(sem,cat,fs,result,opt.readBool(0)):
	       l.findLex(sem,fs,result,opt.readBool(0)))==0)
	    setFailedResult(result,sem);
	}
	else return false;
      }
      return true;
    }
    return false;
  }
};

// get-lex-fs:
// req: (sem, fs)
// opt: (lexID,check)
struct GetLexFS2_1: public _GetLexFSBase {
  GetLexFS2_1 (const Symbol &name):_GetLexFSBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && opt.isAtom(0,fsRegs,lexID)) {
      Lexicon &lexicon=g.lex.findLexicon(lexID);
      
      if (lexicon.readID()!=symNull) {
	const EFArg::Flags &flags=req.readFlags(1);
	
	assert(flags[EFArg::isSet] && (flags[EFArg::tFS] || flags[EFArg::tFSPath]));

	setCheckFS(opt,1,fsRegs);

	if (flags[EFArg::tFS]) {
	  // quoted FS
	  if ((req.size()==3?
	       lexicon.findFirst(sem,*req.readValue(2).begin(),req.readFS(1),result):
	       lexicon.findFirst(sem,req.readFS(1),result))>0)
	    do {
	      // result will be cleared if filter() returns false	    
	      if (checkFSPtr==NULL || result.filter(*checkFSPtr)) 
		return true;
	    } while (lexicon.findNext(result));
	}
	else {
	  // (FS Path)
	  const FSPath &fsPath=req.readFSPath(1);
	  FStruc fs;
	  
	  if (fsRegs[fsPath.fsIdx].getFS(fsPath.path,fs)) {
	    if ((fs.removeAtomValue(catPath,cat)?
		 lexicon.findFirst(sem,cat,fs,result):
		 lexicon.findFirst(sem,fs,result))>0)
	      do {
		// result will be cleared if filter() returns false	    
		if (checkFSPtr==NULL || result.filter(*checkFSPtr))
		  return true;
	      } while (lexicon.findNext(result));
	  }
	  else return false;
	}
      }

      setFailedResult(result,sem);
      return true;
    }
    return false;
  }
};

// get-lex-fs:
// req: (sem, fs)
// opt: (lexID,ambiguity)
struct GetLexFS2_2: public _GetLexFSBase {
  GetLexFS2_2 (const Symbol &name):_GetLexFSBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && opt.isAtom(0,fsRegs,lexID)) {
      Lexicon &lexicon=g.lex.findLexicon(lexID);
      
      if (lexicon.readID()==symNull) setFailedResult(result,sem);
      else {
	const EFArg::Flags &flags=req.readFlags(1);
	
	assert(flags[EFArg::isSet] && (flags[EFArg::tFS] || flags[EFArg::tFSPath]));
	
	if (flags[EFArg::tFS]) {
	  // quoted FS
	  if ((req.size()==3?
	       lexicon.findLex(sem,*req.readValue(2).begin(),req.readFS(1),result,opt.readBool(1)):
	       lexicon.findLex(sem,req.readFS(1),result,opt.readBool(1)))==0)
	    setFailedResult(result,sem);
	}
	else {
	  // (FS Path)
	  const FSPath &fsPath=req.readFSPath(1);
	  FStruc fs;
	  
	  if (fsRegs[fsPath.fsIdx].getFS(fsPath.path,fs)) {
	    if ((fs.removeAtomValue(catPath,cat)?
		 lexicon.findLex(sem,cat,fs,result,opt.readBool(1)):
		 lexicon.findLex(sem,fs,result,opt.readBool(1)))==0)
	      setFailedResult(result,sem);
	  }
	  else return false;
	}
      }
      return true;
    }
    return false;
  }
};

// get-lex-fs:
// req: (sem, fs)
// opt: (check,ambiguity)
struct GetLexFS2_3: public _GetLexFSBase {
  GetLexFS2_3 (const Symbol &name):_GetLexFSBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem)) {
      Lexicons &l=g.lex;
      const EFArg::Flags &flags=req.readFlags(1);

      assert(flags[EFArg::isSet] && (flags[EFArg::tFS] || flags[EFArg::tFSPath]));

      if (flags[EFArg::tFS])
	// quoted FS
	if ((req.size()==3?
	     l.findLex(sem,*req.readValue(2).begin(),req.readFS(1),result,opt.readBool(1)):
	     l.findLex(sem,req.readFS(1),result,opt.readBool(1)))>0 &&
	    check(opt,0,fsRegs,result))
	  return true;
	else setFailedResult(result,sem);
      
      else {
	// (FS Path)
	const FSPath &fsPath=req.readFSPath(1);
	FStruc fs;
	
	if (fsRegs[fsPath.fsIdx].getFS(fsPath.path,fs))
	  if ((fs.removeAtomValue(catPath,cat)?
	       l.findLex(sem,cat,fs,result,opt.readBool(1)):
	       l.findLex(sem,fs,result,opt.readBool(1)))>0 &&
	      check(opt,0,fsRegs,result))
	    return true;
	  else setFailedResult(result,sem);
	else return false;
      }
      return true;
    }
    return false;
  }
};

// get-lex-fs:
// req: (sem, fs)
// opt: (lexID,check,ambiguity)
struct GetLexFS3: public _GetLexFSBase {
  GetLexFS3 (const Symbol &name):_GetLexFSBase(name) {}
  bool operator () (Grammar &g, const FSRegisters &fsRegs, 
		    const EFArgs &req, const EFArgs &opt, FStruc &result) {
    if (req.isAtom(0,fsRegs,sem) && opt.isAtom(0,fsRegs,lexID))
      {
	Lexicon &lexicon=g.lex.findLexicon(lexID);
	
	if (lexicon.readID()==symNull) setFailedResult(result,sem);
	else {
	  const EFArg::Flags &flags=req.readFlags(1);
	  
	  assert(flags[EFArg::isSet] && (flags[EFArg::tFS] || flags[EFArg::tFSPath]));
	  
	  if (flags[EFArg::tFS])
	    // quoted FS
	    if ((req.size()==3?
		 lexicon.findLex(sem,*req.readValue(2).begin(),req.readFS(1),result,opt.readBool(2)):
		 lexicon.findLex(sem,req.readFS(1),result,opt.readBool(2)))>0 &&
		check(opt,1,fsRegs,result))
	      return true;
	    else setFailedResult(result,sem);
	  
	  else {
	    // (FS Path)
	    const FSPath &fsPath=req.readFSPath(1);
	    FStruc fs;
		
	    if (fsRegs[fsPath.fsIdx].getFS(fsPath.path,fs))
	      if ((fs.removeAtomValue(catPath,cat)?
		   lexicon.findLex(sem,cat,fs,result,opt.readBool(2)):
		   lexicon.findLex(sem,fs,result,opt.readBool(2)))>0 &&
		  check(opt,1,fsRegs,result))
		return true;
	      else setFailedResult(result,sem);
	    else return false;
	  }
	}
	return true;
      }
    return false;
  }
};

void _EFOptEntry::insert (const EFArgConvTable &table, EFunc *efPtr) {
  _EFOptKey k;
  EFArgConvTable::const_iterator ti;

  for (ti=table.begin();ti!=table.end();++ti) k.insert(ti->first);
  _Parent::insert(make_pair(k,_EFuncEntry(table,efPtr)));
}

void _EFTable::clear () {
  iterator ti;
  _EFReqEntry::iterator ri;
  _EFOptEntry::iterator oi;

  for (ti=begin();ti!=end();++ti)
    for (ri=ti->second.begin();ri!=ti->second.end();++ri)
      for (oi=ri->second.begin();oi!=ri->second.end();++oi)
	delete oi->second.efPtr;

  _Parent::clear();
  initFlag=false;
}

void _EFTable::init () {
  EFArgConvTable table;

  // -------------------- get-lex --------------------
  
  // get-lex: 2 required args, no optional arg
  table.clear();
  operator[](symGetLex)[2].insert(table,new GetLex0(symGetLex));

  // get-lex: 2 required args, 1 optional arg
  table.clear();
  table.insert(symLexID,0);
  operator[](symGetLex)[2].insert(table,new GetLex1_1(symGetLex));
  table.clear();
  table.insert(symCheck,0);
  operator[](symGetLex)[2].insert(table,new GetLex1_2(symGetLex));
  table.clear();
  table.insert(symAmbiguity,0);
  operator[](symGetLex)[2].insert(table,new GetLex1_3(symGetLex));

  // get-lex: 2 required args, 2 optional args
  table.clear();
  table.insert(symLexID,0);
  table.insert(symCheck,1);
  operator[](symGetLex)[2].insert(table,new GetLex2_1(symGetLex));
  table.clear();
  table.insert(symLexID,0);
  table.insert(symAmbiguity,1);
  operator[](symGetLex)[2].insert(table,new GetLex2_2(symGetLex));
  table.clear();
  table.insert(symCheck,0);
  table.insert(symAmbiguity,1);
  operator[](symGetLex)[2].insert(table,new GetLex2_3(symGetLex));

  // get-lex: 2 required args, 3 optional args
  table.clear();
  table.insert(symLexID,0);
  table.insert(symCheck,1);
  table.insert(symAmbiguity,2);
  operator[](symGetLex)[2].insert(table,new GetLex3(symGetLex));

  // -------------------- get-lex-fs --------------------

  // get-lex-fs: 2 required args, no optional arg
  table.clear();
  operator[](symGetLexFS)[2].insert(table,new GetLexFS0(symGetLexFS));

  // get-lex-fs: 2 required args, 1 optional arg
  table.clear();
  table.insert(symLexID,0);
  operator[](symGetLexFS)[2].insert(table,new GetLexFS1_1(symGetLexFS));
  table.clear();
  table.insert(symCheck,0);
  operator[](symGetLexFS)[2].insert(table,new GetLexFS1_2(symGetLexFS));
  table.clear();
  table.insert(symAmbiguity,0);
  operator[](symGetLexFS)[2].insert(table,new GetLexFS1_3(symGetLexFS));

  // get-lex-fs: 2 required args, 2 optional args
  table.clear();
  table.insert(symLexID,0);
  table.insert(symCheck,1);
  operator[](symGetLexFS)[2].insert(table,new GetLexFS2_1(symGetLexFS));
  table.clear();
  table.insert(symLexID,0);
  table.insert(symAmbiguity,1);
  operator[](symGetLexFS)[2].insert(table,new GetLexFS2_2(symGetLexFS));
  table.clear();
  table.insert(symCheck,0);
  table.insert(symAmbiguity,1);
  operator[](symGetLexFS)[2].insert(table,new GetLexFS2_3(symGetLexFS));

  // get-lex-fs: 2 required args, 3 optional args
  table.clear();
  table.insert(symLexID,0);
  table.insert(symCheck,1);
  table.insert(symAmbiguity,2);
  operator[](symGetLexFS)[2].insert(table,new GetLexFS3(symGetLexFS));

  initFlag=true;
}

EFunc *_EFTable::find (const Symbol &funcName, EFArgs &req, EFArgs &opt) const {
  const_iterator ti;
  _EFReqEntry::const_iterator ri;

  if ((ti=_Parent::find(funcName))!=end() &&
      (ri=ti->second.find(req.size()))!=ti->second.end()) {
    _EFOptEntry::const_iterator oi;
    _EFOptKey k;
    EFArgs::size_type i,s=opt.size();
    
    for (i=0;i<s;++i) k.insert(opt.readName(i));
    
    if ((oi=ri->second.lower_bound(k))!=ri->second.end()) {
      opt.convert(oi->second.table);           // reordering opt args
      oi->second.efPtr->processArgs(req,opt);  // preprocess args
      return oi->second.efPtr;
    }
    else return NULL;
  }
  else return NULL;
}

};
